package com.jpattern.orm.query.clause;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.jpattern.orm.query.BaseFindQuery;
import com.jpattern.orm.query.clause.where.ExpressionElement;

/**
 * 
 * @author Francesco Cina
 * 
 *         18/giu/2011
 */
public interface Where<T extends Where<?>> extends SqlClause<T> {

	/**
	 * All Equal - Map containing property names and their values.
	 * @param propertyMap
	 * @return
	 */
	T allEq(Map<String,Object> propertyMap);

	/**
	 * Express the "Equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	T eq(String property, Object value);

	/**
	 * Express the "Equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	T eqProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Lesser or equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	T le(String property, Object value);

	/**
	 * Express the "Lesser or equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	T leProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Greater or equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	T ge(String property, Object value);

	/**
	 * Express the "Greater or equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	T geProperties(String firstProperty, String secondProperty);

	/**
	 * 
	 * Express the "Lesser than" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	T lt(String property, Object value);

	/**
	 * Express the "Lesser than" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	T ltProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Greater than" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	T gt(String property, Object value);

	/**
	 * Express the "Greater than" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	T gtProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Insensitive Equal To" between an object's property
	 * and a fixed value (it uses a lower() function to make both case insensitive).
	 * 
	 * @param propertyName
	 * @param value
	 * @return
	 */
	T ieq(String property, String value);

	/**
	 * Express the "Insensitive Equal To" bbetween objects properties
	 * (it uses a lower() function to make both case insensitive).
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	T ieqProperties(String firstProperty, String secondProperty);

	/**
	 * Express the "Not Equals to" relation between objects properties.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	T ne(String property, Object value);

	/**
	 * Express the "Not Equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	T neProperties(String firstProperty, String secondProperty);

	/**
	 * Case insensitive Like - property like value where the value contains the
	 * SQL wild card characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 * @return
	 */
	T ilike(String property, String value);

	/**
	 * Not In - property has a value in the collection of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	T nin(String property, Collection<?> values);

	/**
	 * Not In - property has a value in the array of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	T nin(String property, Object[] values);

	/**
	 * Not In - using a subQuery.
	 * 
	 * @param propertyName
	 * @param subQuery
	 * @return
	 */
	T nin(String property, BaseFindQuery subQuery);


	/**
	 * In - property has a value in the collection of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	T in(String property, Collection<?> values);

	/**
	 * In - property has a value in the array of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	T in(String property, Object[] values);

	/**
	 * In - using a subQuery.
	 * 
	 * @param propertyName
	 * @param subQuery
	 * @return
	 */
	T in(String property, BaseFindQuery subQuery);

	/**
	 * Is Not Null - property is not null.
	 * 
	 * @param propertyName
	 * @return
	 */
	T isNotNull(String property);

	/**
	 * Is Null - property is null.
	 * 
	 * @param propertyName
	 * @return
	 */
	T isNull(String property);

	/**
	 * Like - property like value where the value contains the SQL wild card
	 * characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 */
	T like(String property, String value);

	/**
	 * Not Like - property like value where the value contains the SQL wild card
	 * characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 */
	T nlike(String property, String value);

	/**
	 * Negate the expression (prefix it with NOT).
	 * 
	 * @param exp
	 * @return
	 */
	T not(ExpressionElement expression);

	/**
	 * Or - join more expressions with a logical or.
	 * 
	 * @param expressionElements
	 * @return
	 */
	T or(final ExpressionElement... expressionElements);

	/**
	 * Or - join more expressions with a logical or.
	 * 
	 * @param expressionElements
	 * @return
	 */
	T or(final List<ExpressionElement> expressionElements);

	/**
	 * And - join more expressions with a logical and.
	 * 
	 * @param expressionElements
	 * @return
	 */
	T and(final List<ExpressionElement> expressionElements);

	/**
	 * And - join more expressions with a logical and.
	 * 
	 * @param expressionElements
	 * @return
	 */
	T and(final ExpressionElement... expressionElements);

}
