package com.jpattern.orm.query;

import java.util.LinkedHashMap;
import java.util.Map;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmQueryFormatException;
import com.jpattern.orm.mapper.IOrmClassTool;
import com.jpattern.orm.mapper.IOrmClassToolMap;
import com.jpattern.orm.mapper.clazz.IClassMap;

/**
 * 
 * @author Francesco Cina
 *
 * 22/giu/2011
 */
public class OrmClassToolMapNameSolver implements NameSolver {

	private final Map<String, IClassMap<?>> registeredClass = new LinkedHashMap<String, IClassMap<?>>();
	private final Map<Integer, String> classAlias = new LinkedHashMap<Integer, String>();
	private final IOrmClassToolMap ormClassToolMap;
	private boolean resolveWithoutAlias = false;
	private String defaultAlias = null;
	private int registeredClassCount = 0;

	public OrmClassToolMapNameSolver(final IOrmClassToolMap ormClassToolMap) {
		this.ormClassToolMap = ormClassToolMap;
	}

	@Override
	public String solvePropertyName(final String property) throws OrmException {
		if (this.resolveWithoutAlias) {
			return solvePropertyNameWithoutAlias(property);
		}
		final String alias = this.alias(property);
		final String field = field(property);
		if (!this.registeredClass.containsKey(alias)) {
			throw new OrmException("Alias [" + alias + "] is not associated with an Orm Entity. Registered alias are: " + this.registeredClass.keySet());
		}
		final String dbColumn = getDbColumn(alias, field);
		return alias + "." + dbColumn;
	}

	@Override
	public String solvePropertyName(final String property, final String defaultValue) throws OrmException {
		final String alias = this.alias(property);
		final String field = field(property);
		if (!this.registeredClass.containsKey(alias)) {
			return defaultValue;
		}
		final String dbColumn = getDbColumn(alias, field);
		return alias + "." + dbColumn;
	}

	@Override
	public <P> Integer register(final Class<P> clazz) throws OrmException {
		return this.register(clazz, clazz.getSimpleName());
	}

	@Override
	public <P> Integer register(final Class<P> clazz, final String alias) throws OrmException {
		return register(clazz, alias, this.ormClassToolMap.getOrmClassTool(clazz));
	}

	private <P> Integer register(final Class<P> clazz, final String alias, final IOrmClassTool<P> ormClassTool) throws OrmException {
		Integer classId = this.registeredClassCount++;
		this.registeredClass.put(alias, ormClassTool.getClassMap());
		this.classAlias.put(classId, alias);
		if (this.defaultAlias==null) {
			this.defaultAlias = alias;
		}
		return classId;
	}


	@Override
	public String alias(final Integer classId) throws OrmException {
		if (!this.classAlias.containsKey(classId)) {
			throw new OrmException("No class are registered in this query with the id " + classId  );
		}
		return this.classAlias.get(classId);
	}

	private String alias(final String property) throws OrmException {
		try {
			return property.substring(0, property.lastIndexOf("."));
		} catch (final Exception e) {
			//			throw new OrmException("Error parsing property [" + property + "], the format must be CLASS_NAME.CLASS_FIELD or CLASS_ALIAS.CLASS_FIELD" );
			return this.defaultAlias;
		}
	}

	private String field(final String property) throws OrmException {
		try {
			return property.substring(property.lastIndexOf(".")+1);
		} catch (final Exception e) {
			throw new OrmException("Error parsing property [" + property + "], the format must be CLASS_NAME.CLASS_FIELD or CLASS_ALIAS.CLASS_FIELD" );
		}
	}

	@Override
	public String solvePropertyNameWithoutAlias( final String property) throws OrmException {
		final String alias = this.alias(property);
		final String field = field(property);
		if (!this.registeredClass.containsKey(alias)) {
			throw new OrmException("Alias [" + alias + "] is not associated with an Orm Entity. Registered alias are: " + this.registeredClass.keySet());
		}
		return getDbColumn(alias, field);
	}

	@Override
	public void alwaysResolveWithoutAlias(final boolean resolveWithoutAlias) {
		this.resolveWithoutAlias = resolveWithoutAlias;

	}

	@Override
	public boolean getAlwaysResolveWithoutAlias() {
		return this.resolveWithoutAlias;
	}

	private String getDbColumn(final String alias, final String field) {
		String dbColumn = this.registeredClass.get(alias).getClassFieldByJavaName(field).getColumnInfo().getDBColumnName();
		if (dbColumn.isEmpty()) {
			throw new OrmQueryFormatException("Field with name [" + field + "] is not present or ignored for alias [" + alias + "]");
		}
		return dbColumn;
	}

}
