package com.jpattern.orm.persistor;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jpattern.orm.crud.AColumnValueGenerator;
import com.jpattern.orm.crud.ColumnValueGeneratorFactory;
import com.jpattern.orm.exception.OrmConfigurationException;
import com.jpattern.orm.exception.OrmReflectionException;
import com.jpattern.orm.mapper.clazz.ClassField;
import com.jpattern.orm.mapper.clazz.IClassField;
import com.jpattern.orm.mapper.clazz.IClassMap;
import com.jpattern.orm.persistor.generator.GeneratorManipulator;
import com.jpattern.orm.persistor.generator.GeneratorManipulatorImpl;
import com.jpattern.orm.persistor.generator.NullGeneratorManipulator;
import com.jpattern.orm.persistor.generator.ValueChecker;
import com.jpattern.orm.persistor.generator.ValueCheckerFactory;
import com.jpattern.orm.persistor.version.VersionManipulator;

/**
 * A persistor implementation based on reflection
 * @author Francesco Cina'
 *
 * Mar 24, 2012
 */
public class ReflectionOrmPersistor<BEAN> implements IOrmPersistor<BEAN> {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final Map<String, PropertyPersistor<BEAN, ?, ?>> propertyPersistors;
	private GeneratorManipulator<BEAN> generatorManipulator = new NullGeneratorManipulator<BEAN>();
	private final IClassMap<BEAN> classMap;
	private final VersionManipulator<BEAN> versionManipulator;

	public ReflectionOrmPersistor(final IClassMap<BEAN> classMap, final Map<String, PropertyPersistor<BEAN, ?, ?>> propertyPersistors, final VersionManipulator<BEAN> versionManipulator) throws OrmConfigurationException, SecurityException, NoSuchMethodException, IllegalArgumentException, IllegalAccessException, NoSuchFieldException {
		this.classMap = classMap;
		this.propertyPersistors = propertyPersistors;
		this.versionManipulator = versionManipulator;
		this.initGeneratorManipulator();
	}

	private <P> void initGeneratorManipulator() throws OrmConfigurationException, SecurityException, NoSuchMethodException {
		if (this.classMap.getAllGeneratedColumnJavaNames().length>0) {
			final String columnJavaName = this.classMap.getAllGeneratedColumnJavaNames()[0];
			final IClassField<BEAN, P> column = this.classMap.getClassFieldByJavaName(columnJavaName);

			@SuppressWarnings("unchecked")
			final PropertyPersistor<BEAN, P, ?> fieldManipulator = (PropertyPersistor<BEAN, P, ?>) this.propertyPersistors.get(columnJavaName);

			final AColumnValueGenerator generator = ColumnValueGeneratorFactory.getColumnValueGenerator( (ClassField<BEAN, ? extends Object>) this.classMap.getClassFieldByJavaName(columnJavaName) , false);
			final long[] validValues = generator.getActivateValues();
			ValueChecker<P> valueChecker = ValueCheckerFactory.getValueChecker(column.getType());
			this.generatorManipulator = new GeneratorManipulatorImpl<BEAN, P>(fieldManipulator, valueChecker, validValues);
		}
	}

	@Override
	public BEAN mapRow(final String rowNamePrefix, final ResultSet rs, final int rowNum) {
		final String[] allColumnNames = this.classMap.getAllColumnJavaNames();
		try {
			final BEAN entity = this.classMap.getMappedClass().newInstance();
			for (final String columnJavaName : allColumnNames) {
				this.propertyPersistors.get(columnJavaName).getFromResultSet(entity, rs, rowNamePrefix);
			}
			return entity;
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

	@Override
	public BEAN clone(final BEAN entity) {
		try {
			final BEAN entityCopy = this.classMap.getMappedClass().newInstance();
			for ( final Entry<String, PropertyPersistor<BEAN, ?, ?>> persistorEntry : this.propertyPersistors.entrySet()) {
				persistorEntry.getValue().clonePropertyValue(entity, entityCopy);
			}
			return entityCopy;
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

	@Override
	public Object[] allValues(final BEAN entity) {
		return this.getValues(this.classMap.getAllColumnJavaNames(), entity);
	}

	@Override
	public void setAllValues(final BEAN entity, final PreparedStatement ps) {
		final String[] columnNames = this.classMap.getAllColumnJavaNames();
		this.setValues(columnNames, entity, ps);
	}

	@Override
	public Object[] allNotGeneratedValues(final BEAN entity) {
		return this.getValues(this.classMap.getAllNotGeneratedColumnJavaNames(), entity);
	}

	@Override
	public void setAllNotGeneratedValues(final BEAN entity, final PreparedStatement ps) {
		final String[] columnNames = this.classMap.getAllNotGeneratedColumnJavaNames();
		this.setValues(columnNames, entity, ps);
	}

	@Override
	public Object[] primaryKeyValues(final BEAN entity) {
		return this.getValues(this.classMap.getPrimaryKeyColumnJavaNames(), entity);
	}

	@Override
	public void setPrimaryKeyValues(final BEAN entity, final PreparedStatement ps) {
		final String[] columnNames = this.classMap.getPrimaryKeyColumnJavaNames();
		this.setValues(columnNames, entity, ps);
	}

	@Override
	public Object[] notPrimaryKeyValues(final BEAN entity) {
		return this.getValues(this.classMap.getNotPrimaryKeyColumnJavaNames(), entity);
	}

	@Override
	public void setNotPrimaryKeyValues(final BEAN entity, final PreparedStatement ps) {
		final String[] columnNames = this.classMap.getNotPrimaryKeyColumnJavaNames();
		this.setValues(columnNames, entity, ps);
	}

	@Override
	public void setNotPrimaryKeyAndThenPrimaryKeyValues(final BEAN entity,
			final PreparedStatement ps) {
		String[] npkArgs = this.classMap.getNotPrimaryKeyColumnJavaNames();;
		String[] pkArgs = this.classMap.getPrimaryKeyColumnJavaNames();;
		final String[] columns = new String[ npkArgs.length + pkArgs.length ];
		int i=0;
		for (final String value : npkArgs) {
			columns[i++] = value;
		}
		for (final String value : pkArgs) {
			columns[i++] = value;
		}
		this.setValues(columns, entity, ps);
	}

	@Override
	public Object[] primaryKeyAndVersionValues(final BEAN entity) {
		return this.getValues(this.classMap.getPrimaryKeyAndVersionColumnJavaNames(), entity);
	}

	@Override
	public void setPrimaryKeyAndVersionValues(final BEAN entity, final PreparedStatement ps) {
		final String[] columnNames = this.classMap.getPrimaryKeyAndVersionColumnJavaNames();
		this.setValues(columnNames, entity, ps);
	}

	@Override
	public void increaseVersion(final BEAN entity, final boolean firstVersionNumber) {
		try {
			this.versionManipulator.updateVersion(entity, firstVersionNumber);
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

	@Override
	public boolean isVersionable() {
		return this.versionManipulator.isVersionable();
	}

	@Override
	public void updateGeneratedValues(final ResultSet rs, final BEAN entity) {
		final String[] allColumnNames = this.classMap.getAllGeneratedColumnJavaNames();
		try {
			int i=1;
			for (final String columnJavaName : allColumnNames) {
				this.propertyPersistors.get(columnJavaName).getFromResultSet(entity, rs, i);
				i++;
			}
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

	private Object[] getValues(final String[] javaColumnNames, final BEAN entity) {
		final Object[] result = new Object[javaColumnNames.length];
		try {
			for (int i=0; i<javaColumnNames.length ; i++) {
				final String javaColumnName = javaColumnNames[i];
				result[i] = this.propertyPersistors.get(javaColumnName).getDBReadyValueFromBean(entity);
			}
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
		return result;
	}

	@Override
	public boolean useKeyGenerators(final BEAN entity) {
		try {
			return this.generatorManipulator.useGenerator(entity);
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

	@Override
	public boolean hasGenerators() {
		try {
			return this.generatorManipulator.hasGenerator();
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

	@Override
	public boolean hasConditionalGenerator() {
		try {
			return this.generatorManipulator.hasConditionalGenerator();
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

	private void setValues(final String[] allColumnNames, final BEAN entity, final PreparedStatement ps) {
		try {
			Object[] allValues = this.getValues(allColumnNames, entity);
			if (this.logger.isDebugEnabled()) {
				this.logger.debug("Query params: " + Arrays.toString(allValues));
			}
			for (int i=0; i<allColumnNames.length; i++) {
				@SuppressWarnings("unchecked")
				PropertyPersistor<BEAN, Object, Object> persistor = (PropertyPersistor<BEAN, Object, Object>) this.propertyPersistors.get(allColumnNames[i]);
				persistor.setToPreparedStatement(allValues[i], ps, i+1);

			}
		} catch (final Exception e) {
			throw new OrmReflectionException(e);
		}
	}

}
