package com.jpattern.orm.persistor;

import java.lang.reflect.InvocationTargetException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import com.jpattern.orm.mapper.clazz.IClassField;
import com.jpattern.orm.persistor.type.JdbcIO;
import com.jpattern.orm.persistor.type.TypeWrapper;
import com.jpattern.orm.persistor.version.VersionMath;

/**
 * 
 * @author ufo
 *
 * @param <P> the type of the bean to manipulate
 * @param <P> the type of the bean's property to manipulate
 * @param <DB> the type of the field in the {@link PreparedStatement} and {@link ResultSet}
 */
public class PropertyPersistor<BEAN, P, DB> {

	private final JdbcIO<DB> jdbcIO;
	private final TypeWrapper<P, DB> typeWrapper;
	private final VersionMath<P> math;
	private final IClassField<BEAN, P> classField;

	public PropertyPersistor (final TypeWrapper<P, DB> typeWrapper, final JdbcIO<DB> jdbcIO,
			final IClassField<BEAN, P> classField, final VersionMath<P> math) {
		this.jdbcIO = jdbcIO;
		this.typeWrapper = typeWrapper;
		this.classField = classField;
		this.math = math;

	}

	/**
	 * Extract the value of a bean's property and set it to the {@link PreparedStatement}
	 * @param bean
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws SQLException
	 * @throws IllegalArgumentException
	 */
	public void setToPreparedStatement(final DB value, final PreparedStatement ps, final int psIndex) throws IllegalArgumentException, SQLException, IllegalAccessException, InvocationTargetException {
		this.jdbcIO.setValueToPreparedStatement( value, ps, psIndex);
	}

	/**
	 * Extract the value from a {@link ResultSet} and set it to a bean's property
	 * @param bean
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 * @throws SQLException
	 */
	public void getFromResultSet(final BEAN bean, final ResultSet rs, final String rsColumnNamePrefix) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, SQLException {
		this.setDBValueToBean( bean, this.jdbcIO.getValueFromResultSet(rs, rsColumnNamePrefix + this.classField.getColumnInfo().getDBColumnName() ) );
	}

	/**
	 * Extract the value from a {@link ResultSet} and set it to a bean's property
	 * @param bean
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 * @throws SQLException
	 */
	public void getFromResultSet(final BEAN bean, final ResultSet rs, final int rsColumnIndex) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, SQLException {
		this.setDBValueToBean( bean, this.jdbcIO.getValueFromResultSet(rs, rsColumnIndex) );
	}

	/**
	 * Set copy the property value from source to destination
	 * @return
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	public void clonePropertyValue(final BEAN source, final BEAN destination) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		this.setPropertyValueToBean(destination, this.typeWrapper.clone(this.getPropertyValueFromBean(source)));
	}

	public void increaseVersion(final BEAN bean, final boolean firstVersionNumber) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		this.setPropertyValueToBean(bean, this.math.increase(firstVersionNumber, this.getPropertyValueFromBean(bean)));
	}

	public DB getDBReadyValueFromBean(final BEAN bean) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		return this.typeWrapper.unWrap(this.getPropertyValueFromBean(bean));
	}

	public void setDBValueToBean(final BEAN bean, final DB value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		this.setPropertyValueToBean(bean, this.typeWrapper.wrap(value));
	}

	public P getPropertyValueFromBean(final BEAN bean) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		return this.classField.getGetManipulator().getValue(bean);
	}

	public void setPropertyValueToBean(final BEAN bean, final P value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		this.classField.getSetManipulator().setValue(bean, value);
	}

	public Class<DB> jdbcType() {
		return this.typeWrapper.jdbcType();
	}

	public Class<P> propertyType(){
		return this.typeWrapper.propertyType();
	}

	public String getDbColumnName() {
		return this.classField.getColumnInfo().getDBColumnName();
	}
}
