package com.jpattern.orm.crud;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jpattern.orm.dialect.Dialect;
import com.jpattern.orm.mapper.clazz.ClassField;
import com.jpattern.orm.mapper.clazz.IClassField;
import com.jpattern.orm.mapper.clazz.IClassMap;

/**
 * 
 * @author Francesco Cina
 *
 * 22/mag/2011
 */
public class OrmCRUDQueryGenerator<BEAN> {

	private boolean generatedKey = false;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final Dialect dialect;
	private final IClassMap<BEAN> classMap;

	public OrmCRUDQueryGenerator(final Dialect dialect, final IClassMap<BEAN> classMap) {
		this.dialect = dialect;
		this.classMap = classMap;
	}

	/* (non-Javadoc)
	 * @see com.jpattern.orm.generator.IOrmCRUDQueryGenerator#generate()
	 */
	public CRUDQuery generate() {
		final OrmCRUDQuery crud = new OrmCRUDQuery();
		crud.setDeleteQuery( this.generateDeleteQuery() );
		crud.setExistQuery( this.generateExistQuery() );
		crud.setLoadQuery(  this.generateLoadQuery() );
		crud.setSaveQuery(  this.generateSaveQuery(false) );
		crud.setSaveQueryWithoutGenerators( this.generateSaveQuery(true) );
		crud.setUpdateQuery(  this.generateUpdateQuery() );
		crud.setBaseSelectClause(  this.generateBaseSelectClause() );
		crud.setBaseFromClause(  this.generateBaseFromClause() );
		crud.setBeanVersionQuery( this.generateBeanVersionQuery() );
		return crud;
	}

	private String generateBaseFromClause() {
		return this.classMap.getTableInfo().getTableNameWithSchema();
	}

	private String generateBaseSelectClause() {
		return this.columnToCommaSepareted( OrmCrudConstants.ROW_NAME_PREFIX_PLACEHOLDER, this.getColumnNames( this.classMap.getAllColumnJavaNames()), true );
	}

	private String generateUpdateQuery() {
		final StringBuilder builder = new StringBuilder("UPDATE ");
		builder.append(this.classMap.getTableInfo().getTableNameWithSchema());
		builder.append(" SET ");
		if ( this.classMap.getNotPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append( this.columnToSetClause( this.getColumnNames( this.classMap.getNotPrimaryKeyColumnJavaNames()) ));
		}
		if ( this.classMap.getPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append(" WHERE ");
			builder.append( this.columnToWhereClause( this.getColumnNames( this.classMap.getPrimaryKeyColumnJavaNames()) ));
		}
		final String query = builder.toString();
		this.logger.info("Generated UPDATE query for table [" + this.classMap.getTableInfo().getTableNameWithSchema() + "]:\n" + query );
		return query;
	}

	private String generateSaveQuery(final boolean ignoreGenerators) {
		final StringBuilder builder = new StringBuilder("INSERT INTO ");
		builder.append(this.classMap.getTableInfo().getTableNameWithSchema());
		builder.append(" (");
		builder.append( this.columnToCommaSepareted( "" , this.getColumnNames( this.classMap.getAllColumnJavaNames()), ignoreGenerators ));
		builder.append(") VALUES (");
		builder.append( this.questionCommaSepareted( this.getColumnNames( this.classMap.getAllColumnJavaNames()), ignoreGenerators ));
		builder.append(")");
		final String query = builder.toString();
		this.logger.info("Generated INSERT query (with generators? " + !ignoreGenerators + ") for table [" + this.classMap.getTableInfo().getTableNameWithSchema() + "]:\n" + query );
		return query;
	}

	private String generateExistQuery() {
		final String query = buildLoadQueryCommon("count(*)");
		this.logger.info("Generated EXIST query for table [" + this.classMap.getTableInfo().getTableNameWithSchema() + "]:\n" + query );
		return query;
	}

	private String generateLoadQuery() {
		//		final StringBuilder builder = new StringBuilder("SELECT * FROM ");
		//		builder.append(this.classMap.getTableInfo().getTableNameWithSchema());
		//		if ( this.classMap.getPrimaryKeyColumnJavaNames().length>0 ) {
		//			builder.append(" WHERE ");
		//			builder.append( this.columnToWhereClause( this.getColumnNames( this.classMap.getPrimaryKeyColumnJavaNames()) ));
		//		}
		//		final String query = builder.toString();
		final String query = buildLoadQueryCommon("*");
		this.logger.info("Generated SELECT query for table [" + this.classMap.getTableInfo().getTableNameWithSchema() + "]:\n" + query );
		return query;
	}

	private String buildLoadQueryCommon(final String selectClause) {
		final StringBuilder builder = new StringBuilder("SELECT ");
		builder.append(selectClause);
		builder.append(" FROM ");
		builder.append(this.classMap.getTableInfo().getTableNameWithSchema());
		if ( this.classMap.getPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append(" WHERE ");
			builder.append( this.columnToWhereClause( this.getColumnNames( this.classMap.getPrimaryKeyColumnJavaNames()) ));
		}
		return builder.toString();
	}

	private String generateBeanVersionQuery() {
		String query = "";
		for (final String columnJavaName : this.classMap.getAllColumnJavaNames()) {
			final IClassField<BEAN, ?> classField = this.classMap.getClassFieldByJavaName(columnJavaName);
			if (classField.getVersionInfo().isVersionable()) {
				final StringBuilder builder = new StringBuilder("SELECT count(*) FROM ");
				builder.append(this.classMap.getTableInfo().getTableNameWithSchema());
				builder.append(" WHERE ");
				builder.append( this.columnToWhereClause( this.getColumnNames( this.classMap.getPrimaryKeyAndVersionColumnJavaNames() ) ));
				builder.append( this.classMap.getClassFieldByJavaName( classField.getFieldName() ).getVersionInfo().getLockMode().getMode() );
				query = builder.toString();
			}
		}

		this.logger.info("Generated query to load the version of a record for table [" + this.classMap.getTableInfo().getTableNameWithSchema() + "]:\n" + query );
		return query;
	}

	private String generateDeleteQuery() {
		final StringBuilder builder = new StringBuilder("DELETE FROM ");
		builder.append(this.classMap.getTableInfo().getTableNameWithSchema());
		if ( this.classMap.getPrimaryKeyColumnJavaNames().length>0 ) {
			builder.append(" WHERE ");
			builder.append( this.columnToWhereClause( this.getColumnNames(this.classMap.getPrimaryKeyColumnJavaNames()) ));
		}
		final String query = builder.toString();
		this.logger.info("Generated DELETE query for table [" + this.classMap.getTableInfo().getTableNameWithSchema() + "]:\n" + query );
		return query;
	}


	private String questionCommaSepareted(final List<String> columnNames, final boolean ignoreGenerators) {
		final StringBuilder builder = new StringBuilder();
		final int length = columnNames.size();

		if (length > 0) {
			for (int i=0; i<(length-1) ; i++) {
				final AColumnValueGenerator columnValueGenerator = this.getColumnValueGenerator(columnNames.get(i), ignoreGenerators);
				this.generatedKey = this.generatedKey || columnValueGenerator.isAutoGenerated();
				final String queryParameter = columnValueGenerator.insertQueryParameter(this.dialect, "?");
				if (queryParameter.length()>0) {
					builder.append( queryParameter );
					builder.append(", ");
				}
			}
			final AColumnValueGenerator columnValueGenerator = this.getColumnValueGenerator(columnNames.get(length-1), ignoreGenerators);
			this.generatedKey = this.generatedKey || columnValueGenerator.isAutoGenerated();
			builder.append( columnValueGenerator.insertQueryParameter(this.dialect, "?") );
		}
		return builder.toString();
	}

	private String columnToCommaSepareted(final String prefix, final List<String> columnNames, final boolean ignoreGenerators) {
		final StringBuilder builder = new StringBuilder();
		final int length = columnNames.size();

		if (length > 0) {
			for (int i=0; i<(length-1) ; i++) {
				final AColumnValueGenerator columnValueGenerator = this.getColumnValueGenerator(columnNames.get(i), ignoreGenerators);
				final String queryParameter = columnValueGenerator.insertColumn(this.dialect, prefix + columnNames.get(i));
				if (queryParameter.length()>0) {
					builder.append( queryParameter );
					builder.append(", ");
				}
			}
			final AColumnValueGenerator columnValueGenerator = this.getColumnValueGenerator(columnNames.get(length-1), ignoreGenerators);
			builder.append( columnValueGenerator.insertColumn(this.dialect, prefix + columnNames.get(length-1)) );
		}

		return builder.toString();
	}

	private AColumnValueGenerator getColumnValueGenerator(final String columnName, final boolean ignoreGenerator) {
		return ColumnValueGeneratorFactory.getColumnValueGenerator( (ClassField<BEAN, ? extends Object>) this.classMap.getClassFieldByDBColumnName(columnName ), ignoreGenerator );
	}

	private String columnToWhereClause(final List<String> columnNames) {
		final StringBuilder builder = new StringBuilder();
		final int length = columnNames.size();

		if (length > 0) {
			for (int i=0; i<(length-1) ; i++) {
				builder.append(columnNames.get(i) + " = ? AND ");
			}
			builder.append(columnNames.get(length-1) + " = ? ");
		}
		return builder.toString();
	}

	private String columnToSetClause(final List<String> columnNames) {
		final StringBuilder builder = new StringBuilder();
		final int length = columnNames.size();

		if (length > 0) {
			for (int i=0; i<(length-1) ; i++) {
				builder.append(columnNames.get(i) + " = ? , ");
			}
			builder.append(columnNames.get(length-1) + " = ? ");
		}
		return builder.toString();
	}

	private List<String> getColumnNames(final String[] javaNames) {
		final List<String> result = new ArrayList<String>();
		for (final String javaName : javaNames) {
			result.add( this.classMap.getClassFieldByJavaName(javaName).getColumnInfo().getDBColumnName());
		}
		return result;
	}
}
