package com.jpattern.orm;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jpattern.orm.crud.CRUDQuery;
import com.jpattern.orm.crud.OrmCRUDQueryGenerator;
import com.jpattern.orm.dialect.Dialect;
import com.jpattern.orm.exception.OrmConfigurationException;
import com.jpattern.orm.mapper.IOrmClassTool;
import com.jpattern.orm.mapper.IOrmClassToolMap;
import com.jpattern.orm.mapper.NullOrmClassToolMap;
import com.jpattern.orm.mapper.OrmClassTool;
import com.jpattern.orm.mapper.OrmClassToolMap;
import com.jpattern.orm.mapper.clazz.ClassMapBuilder;
import com.jpattern.orm.mapper.clazz.IClassMap;
import com.jpattern.orm.persistor.IOrmPersistor;
import com.jpattern.orm.persistor.ReflectionPersistorGenerator;
import com.jpattern.orm.persistor.type.ExtendedTypeWrapper;
import com.jpattern.orm.persistor.type.TypeFactory;
import com.jpattern.orm.session.NullSessionProvider;
import com.jpattern.orm.session.OrmSession;
import com.jpattern.orm.session.Session;
import com.jpattern.orm.session.SessionProvider;
import com.jpattern.orm.validator.NullValidator;
import com.jpattern.orm.validator.Validator;

/**
 * 
 * @author Francesco Cina'
 *
 * 26/ago/2011
 */
public class JPOrm implements JPO {

	private static Integer JPORM_INSTANCES_COUNT = Integer.valueOf(0);
	private SessionProvider sessionProvider;
	private IOrmClassToolMap ormClassToolMap = new OrmClassToolMap(this);
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final Dialect dialect;
	private final Integer instanceCount;
	private final TypeFactory typeFactory = new TypeFactory();
	private Validator validator = new NullValidator();

	/**
	 * Create a new instance of JPOrm.
	 * 
	 * @param sessionProvider
	 */
	public JPOrm(final SessionProvider sessionProvider) {
		synchronized (JPORM_INSTANCES_COUNT) {
			this.instanceCount = JPORM_INSTANCES_COUNT++;
		}
		this.logger.info("Building new instance of JPO (instance [{}])", this.instanceCount);
		this.sessionProvider = sessionProvider;
		this.dialect = sessionProvider.getDBType().getDialect();
	}

	@Override
	public final Session session() {
		return new OrmSession(this.ormClassToolMap, this.sessionProvider.getSessionStrategy(), this.typeFactory, this.validator);
	}

	@Override
	public synchronized <BEAN> void register(final Class<BEAN> clazz) throws OrmConfigurationException {
		try {
			if (!this.ormClassToolMap.containsTool(clazz)) {
				this.logger.info("register new class: " + clazz.getName());
				final IClassMap<BEAN> classMap = new ClassMapBuilder<BEAN>(clazz).generate();
				final IOrmPersistor<BEAN> ormPersistor =  new ReflectionPersistorGenerator<BEAN>(classMap, this.typeFactory).generate();
				final CRUDQuery ormCRUDQuery = new OrmCRUDQueryGenerator<BEAN>(this.dialect, classMap).generate();
				final IOrmClassTool<BEAN> ormClassTool = new OrmClassTool<BEAN>(classMap, ormPersistor, ormCRUDQuery);
				this.ormClassToolMap.put(clazz, ormClassTool);
			}
		} catch (final Exception e) {
			throw new OrmConfigurationException(e);
		}
	}

	@Override
	public synchronized void destory() {
		this.sessionProvider = new NullSessionProvider();
		this.ormClassToolMap = new NullOrmClassToolMap();
	}

	@Override
	public synchronized void register(final List<Class<?>> classes) throws OrmConfigurationException {
		for (final Class<?> clazz : classes) {
			this.register(clazz);
		}
	}

	public SessionProvider getSessionProvider() {
		return this.sessionProvider;
	}

	@Override
	public synchronized void register(final ExtendedTypeWrapper<?, ?> typeWrapper) throws OrmConfigurationException {
		this.typeFactory.addTypeWrapper(typeWrapper);
	}

	@Override
	public synchronized void setValidator(final Validator validator) {
		if (validator!=null) {
			this.validator = validator;
		}
	}

	public TypeFactory getTypeFactory() {
		return this.typeFactory;
	}

}
