package cn.gtmap.landtax.util;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * 操作字符串的实用工具
 * @author	zenglihuan
 * @version	2.0
 */
public class GtmapStringUtils extends StringUtils {

    private static final Log logger = LogFactory.getLog(StringUtils.class);

    /**
     * SQL中in语句的参数过多时，拆分成 in (...) or in (...)时，每一个子in语句中包含的个数
     */
    private static int IN_MOD = 3;

    /**
     * job参数解析==>Properties
     * @param args
     * @return
     */
    public static Properties jobArgsParse(String args) {
        String[] arguFactors = args.split("&");
        Properties propFactor = new Properties();
        for(String factor:arguFactors){
            String[] factorMaps = factor.split("=");
            if(factorMaps != null && factorMaps.length == 2 ) {
                propFactor.setProperty(factorMaps[0],factorMaps[1]);
            }
        }
        return propFactor;
    }

    /**
     * 生成in语句
     * @param "1,2,3"
     * @return 形如('a','b')的语句
     */
    public static String getInString(String arg) {
        String[] args = arg.split(",");
        if(null == args || args.length==0){
            return "('')"; //避免语法错误
        }
        StringBuilder sb = new StringBuilder();
        sb.append("(");
        for(String str : args) {
            sb.append(',');
            if (!(str).startsWith("'")) {
                sb.append("'");
            }
            sb.append(str);
            if (!(str).endsWith("'")) {
                sb.append("'");
            }

        }
        sb.delete(1, 2);		//删除多加的","
        sb.append(")");
        String s = sb.toString();
        if ("()".equals(s)) {	// 避免语法错误
            s = "('')";
        }
        return s;
    }


    /**
     * 生成in语句
     * @param coll 值列表
     * @return 形如('a','b')或(1,2,3)的语句
     */
    public static String getInString(Collection<?> coll) {
        if(null == coll || coll.size()==0){
            return "('')"; //避免语法错误
        }
        StringBuilder sb = new StringBuilder();
        sb.append("(");
        for(Object obj : coll) {
            if(obj == null) continue;

            sb.append(',');
            if (obj instanceof Date) {
                DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                sb.append("'").append(df.format((Date)obj)).append("'");
            } else if (obj instanceof String) {
                if (!((String) obj).startsWith("'")) {
                    sb.append("'");
                }
                sb.append(obj);
                if (!((String) obj).endsWith("'")) {
                    sb.append("'");
                }
            } else {
                sb.append(obj);
            }
        }
        sb.delete(1, 2);		//删除多加的","
        sb.append(")");
        String s = sb.toString();
        if ("()".equals(s)) {	// 避免语法错误
            s = "('')";
        }
        return s;
    }


    /**
     * 将字符数组拼接成由 separator 分隔开的字符串
     * @author ChenWen 2010-12-09
     */
    public static String joinToString(final String[] src, String separator){
        if(src == null) return "";
        if(separator == null) separator = ",";
        StringBuffer dest = new StringBuffer();
        for(String s : src) {
            dest.append(s).append(separator).append(" ");
        }
        dest.delete(dest.lastIndexOf(separator), dest.length());
        return dest.toString();
    }
    /**
     * 将字符集拼接成由 separator 分隔开的字符串
     * @author ChenWen 2010-12-09
     */
    public static String joinToString(final Collection<String> src, String separator){
        if(src == null) return "";
        if(separator == null) separator = ",";
        StringBuffer dest = new StringBuffer();
        for(String s : src) {
            dest.append(s).append(separator).append(" ");
        }
        dest.delete(dest.lastIndexOf(separator), dest.length());
        return dest.toString();
    }



    /**
     * 将传入的Str转化成前补<len>个<replacement>的字符串
     * 根据regex做截断
     * @param str               源字符串
     * @param regex				分隔符
     * @param replacement       替换字符串
     * @param len               长度
     * @return
     * 如：传入(1.10, "\\.", "0", 5), 返回值：0000100010
     * 调用示例：replace(str, "\\.", "0", 5)
     */
    public static String replace(String str, String regex, String replacement, int len){
        if(null==str){
            return null;
        }
        String[] tmpArr = str.split(regex);
        String returnStr = "";
        for(String s : tmpArr){
            returnStr = returnStr + makeup(s, replacement, len-s.length());
        }
        return returnStr;
    }

    /**
     * 将len个的replacement补到Str的前面
     * @param str
     * @param replacement
     * @param len
     * @return
     */
    public static String makeup(String str, String replacement, int len){
        if(len<1){
            return str;
        }
        String upStr = "";
        for(int i=0; i<len; i++){
            upStr = upStr + replacement;
        }
        return upStr + str;
    }

    /**
     * 比较两个字符串的相等性。
     * @return true:相等 , false:不等 .
     * @author ChenWen 2010-12-21
     */
    public static boolean equals(String str1, String str2) {
        return equals(str1, str2, false, false);
    }
    /**
     * 比较两个字符串的相等性。
     * @param ignoreCase 是否忽略大小写
     * @param trim 是否截取首尾的空白符
     * @return true:相等 , false:不等 .
     * @author ChenWen 2010-12-21
     */
    public static boolean equals(String str1, String str2, boolean ignoreCase, boolean trim) {
        if(str1 == null && str2 == null) return true;
        if(str1 == null || str2 == null) return false;

        if(trim) {
            str1 = str1.trim();
            str2 = str2.trim();
        }

        if(ignoreCase) {
            return str1.equalsIgnoreCase(str2);
        }
        return str1.equals(str2);
    }

    /**
     * 获取行分隔符
     */
    public static String newLine() {
        return System.getProperty("line.separator", "\r\n");
    }

    /**
     * 根据给定的list转化为in表达式的sql语句超长,改为以下返回格式,默认属性名为id,默认实体别名为enttiy
     * @param list
     * @return list.size() >  mod  时返回      (entity.id in('a','b'...) or entity.id in ('c','d'...) or entity.id in ...)
     * 									或   (entity.id in(1,2...) or entity.id in (3,4...) or entity.id in ...)
     * 		   list.size() <= mod  时返回      (entity.id in ('a','b'...))
     * 								          或   (entity.id in(1,2...))
     */
    public static String getToOrInString(List<?> list){
        return getToOrInString(list, "id");
    }

    /**
     * 根据给定的list转化为in表达式的sql语句超长,改为以下返回格式,默认实体别名为enttiy
     * @param list
     * @param proName
     * @return list.size() >  mod  时返回       (entity.proName in('a','b'...) or entity.proName in ('c','d'...) or entity.proName in ...)
     * 								或              (entity.proName in(1,2...) or entity.proName in (3,4...) or entity.proName in ...)
     * 		   list.size() <= mod  时返回       (entity.proName in ('a','b'...))
     * 								 或             (entity.proName in(1,2...))
     */
    public static String getToOrInString(List<?> list, String proName){
        return getToOrInString(list, "entity", proName);
    }

    /**
     * 根据给定的list转化为in表达式的sql语句超长,改为以下返回格式,默认实体别名为enttiy
     * @param list
     * @param alias
     * @param proName
     * @return list.size() >  mod  时返回       (entity.proName in('a','b'...) or entity.proName in ('c','d'...) or entity.proName in ...)
     * 								或              (entity.proName in(1,2...) or entity.proName in (3,4...) or entity.proName in ...)
     * 		   list.size() <= mod  时返回       (entity.proName in ('a','b'...))
     * 								 或             (entity.proName in(1,2...))
     */
    public static String getToOrInString(List<?> list, String alias, String proName){
        if(null == list || list.size()==0){
            return "";
        }
        //把list包装成in
        return builedToInOrNotInString(list, alias, proName, 1);
    }

    /**
     * 根据给定的list转化为not in表达式的sql语句超长,改为以下返回格式,默认属性名为id,默认实体别名为enttiy
     * @param list
     * @return list.size() >  mod  时返回      (entity.id not in('a','b'...) and entity.id not in ('c','d'...) and entity.id not in ...)
     * 									或   (entity.id not in(1,2...) and entity.id not in (3,4...) and entity.id not in ...)
     * 		   list.size() <= mod  时返回      (entity.id not in ('a','b'...))
     * 								          或   (entity.id not in(1,2...))
     */
    public static String getToOrNotInString(List<?> list){
        return getToOrNotInString(list, "id");
    }

    /**
     * 根据给定的list转化为not in表达式的sql语句超长,改为以下返回格式,默认实体别名为enttiy
     * @param list
     * @param proName
     * @return list.size() >  mod  时返回       (entity.proName not in('a','b'...) and entity.proName not in ('c','d'...) and entity.proName not in ...)
     * 								或              (entity.proName not in(1,2...) and entity.proName not in (3,4...) and entity.proName not in ...)
     * 		   list.size() <= mod  时返回       (entity.proName not in ('a','b'...))
     * 								 或             (entity.proName not in(1,2...))
     */
    public static String getToOrNotInString(List<?> list, String proName){
        return getToOrNotInString(list, "entity", proName);
    }

    /**
     * 根据给定的list转化为not in表达式的sql语句超长,改为以下返回格式,默认实体别名为enttiy
     * @param list
     * @param alias
     * @param proName
     * @return list.size() >  mod  时返回       (entity.proName not in('a','b'...) and entity.proName not in ('c','d'...) and entity.proName not in ...)
     * 								或              (entity.proName not in(1,2...) and entity.proName not in (3,4...) and entity.proName not in ...)
     * 		   list.size() <= mod  时返回       (entity.proName not in ('a','b'...))
     * 								 或             (entity.proName not in(1,2...))
     */
    public static String getToOrNotInString(List<?> list, String alias, String proName){
        if(null == list || list.size()==0){
            return "";
        }
        //把list包装成 not in
        return builedToInOrNotInString(list, alias, proName, 2);
    }

    /**
     *  拼装in 或者 not in 连接hql语句
     * @param list
     * @param alias
     * @param proName
     * @param type = 1 : 拼装成in语句, type = 2 : 拼装成 not in 语句
     * @return
     */
    @SuppressWarnings("unchecked")
    private static String builedToInOrNotInString(List<?> list, String alias,
                                                  String proName, int type) {
        List tempList = new ArrayList();
        for(Object obj : list) {
            if(obj == null) continue;
            tempList.add(obj);
        }

        StringBuffer sb = new StringBuffer("");
        if (type == 1)
            sb.append("(" + alias + "." + proName + " in ");
        else
            sb.append("(" + alias + "." + proName + " not in ");
        if (tempList.size() > IN_MOD) {
            int times = tempList.size() / IN_MOD;
            for (int i = 1; i <= times; i++) {
                List subList = tempList.subList((i-1)*IN_MOD, i*IN_MOD);
                if (i != 1) {
                    if (type == 1)
                        sb.append(" or " + alias + "." + proName + " in ");
                    else
                        sb.append(" and " + alias + "." + proName + " not in ");
                }
                sb.append(getInString(subList));
            }
            if (tempList.size() % IN_MOD != 0) {
                if (type == 1)
                    sb.append(" or " + alias + "." + proName + " in " + getInString(tempList.subList(times*IN_MOD, tempList.size())));
                else
                    sb.append(" and " + alias + "." + proName + " not in " + getInString(tempList.subList(times*IN_MOD, tempList.size())));
            }
        } else {
            sb.append(getInString(tempList));
        }
        sb.append(")");
        return sb.toString();
    }
    /**
     * 根据给定的list转化为or表达式的sql语句,默认属性名为id,默认实体别名为enttiy
     * @param list
     * @return
     */
    public static String convertListToOrExpression(List<String> list){
        return convertListToOrExpression(list,"entity","id");
    }

    /**
     * 根据给定的list转化为or表达式的sql语句,默认属性名为id
     * @param list
     * @param alias
     * @return
     */
    public static String convertListToOrExpression(List<String> list,String alias){
        return convertListToOrExpression(list,alias,"id");
    }

    /**
     * 根据给定的array转化为or表达式的sql语句
     * @param valueArr
     * @param alias
     * @param field
     * @return
     */
    public static String convertListToOrExpression(String[] valueArr,String alias,String field){
        List<String> list = Arrays.asList(valueArr);
        return convertListToOrExpression(list,alias,field);
    }

    /**
     * 根据给定的list转化为or表达式的sql语句
     * @param list
     * @param alias
     * @param field
     * @return
     */
    public static String convertListToOrExpression(List<String> list,String alias,String field){
        StringBuffer hql = new StringBuffer("");
        if(list != null && list.size() > 0) {
            hql.append(" and ( ");
            boolean needOr = false;
            for(String value : list ){
                if(needOr) {
                    hql.append(" or " + alias + "." + field + " = '" + value + "' ");
                } else {
                    hql.append(alias + "." + field + " = '" + value + "' ");
                    needOr = true;
                }
            }
            hql.append(" ) ");
        }
        return hql.toString();
    }

    /**
     * 生成Like从句
     *
     * @param 	str
     * 			用于产生组合的字符串数组
     *
     * @param 	propertyName
     * 			字段名
     *
     * @return	Like从句
     */
    public static String genLikeClause(String[] str, String propertyName) {
        if (str != null && str.length > 0) {
            String likeClause = Arrays.toString(str).replace(", ",
                    "%' or " + propertyName + " like '%");
            likeClause = likeClause.substring(1, likeClause.length() - 1);
            return "(" + propertyName + " like '%" + likeClause + "%')";
        }
        else {
            return "(" + propertyName + " is null)";
        }
    }

    /**
     * 判断ID是否存在于列表中
     * @param id 标识
     * @param objList 对象列表,可以是String,WsdGenericDto,WsdEntity
     * @return true 在列表中存在包含该标识的对象,否则返回false
     */
    @SuppressWarnings("unchecked")
    public static boolean idInList(String id, List objList) {
        if (objList == null) {
            return false;
        }
        for (Object obj: objList) {
            if (obj instanceof String) {
                return ((String)obj).equals(id);
            }
        }
        return false;
    }

    /**
     * 全字匹配替换字符串（单词中的出现的字符不替换）
     * @param str 原字符串
     * @param oldStr 待替换字符
     * @param newString 替换后的字符
     * @return 替换后的字符串
     */
    public static String replaceExactly(String str, String oldStr, String newString) {
        String regex = "\\W" +oldStr+ "(\\W|$)";
        Pattern ptn = Pattern.compile(regex);
        Matcher m = ptn.matcher(str);
        boolean matched = false;
        List<int[]> intList = new ArrayList<int[]>();
        List<String> strList = new ArrayList<String>();
        while (m.find()) {
            matched = true;
            String s = m.group();
            strList.add(s);
            int[] ia = new int[2];
            ia[0] = m.start();
            ia[1] = m.end();
            intList.add(ia);
        }
        StringBuilder sb = new StringBuilder();
        for (int i=0; i<intList.size(); i++) {
            int[] ia = intList.get(i);
            if (i == 0) {
                sb.append(str.substring(0, ia[0]));
            }
            if (i > 0) {
                int[] iaBefore = intList.get(i-1);
                String s = strList.get(i-1);
                s = s.replaceAll(oldStr, newString);
                sb.append(s).append(str.substring(iaBefore[1], ia[0]));
            }
            if (i == intList.size()-1) {
                String s = strList.get(i);
                s = s.replaceAll(oldStr, newString);
                sb.append(s).append(str.substring(ia[1]));
            }
        }
        str = matched ? sb.toString() : str;
        return str;
    }

    /**
     * 返回去掉单引号后的字符串
     * @param quotedString 带单引号的字符串
     */
    public static String getUnquotedString(String quotedString) {
        if (quotedString == null) {
            return quotedString;
        }
        if (quotedString.startsWith("'") && quotedString.endsWith("'")) {
            return quotedString.substring(1, quotedString.length()-1);
        } else {
            return quotedString;
        }
    }


    /**
     * 返回点后面的字符串（如类名的简称）
     * @param str 等处理字符串
     * @return 点后面的字符串
     */
    public static String getStringAfterPoint(String str) {
        if (str == null) {
            return null;
        }
        int iPoint = str.lastIndexOf('.');
        if (iPoint > 0) {
            str = str.substring(iPoint+1);
        }
        return str;
    }

    /**
     * 将组合id用下划线隔开，组成一个id
     * @param ids ID列表
     * @return 组合id用下划线隔开，组成一个id
     */
    public static String getCompositeId(String ... ids) {
        return joinToString(ids, "_");
    }

    /**
     * 获取首字母小写的字符串
     * @param str 原字符串
     * @return 首字母小写的字符串
     */
    public static String getLowerCaseInitialString(String str) {
        if (str == null || str.length()<1) {
            return null;
        }
        return str.substring(0, 1).toLowerCase() + str.substring(1, str.length());
    }

    /**
     * 返回子字符串
     * @param str 原字符串 ****(???)****(?)**
     * @param beginSeparate 前标识符
     * @param endSeparate 后标识符
     * @return 子字符串数组 (???),(?)
     */
    public static List<String> getSubStr(String str,String beginSeparate,String endSeparate){
        List<String> array = new ArrayList<String>();
        if(StringUtils.isNotBlank(str)){
            int i = 0;
            int j = 0;
            while(i < str.length()){
                i = str.indexOf(beginSeparate,i);
                if(i < 0){
                    return array;
                }else{
                    j = str.indexOf(endSeparate,i);
                    if(j < 0){
                        return array;
                    }else{
                        array.add(str.substring(i, j+1));
                        i = ++j;
                    }
                }
            }
            return array;
        }
        return array;
    }
}
