package cn.gtmap.landtax.support.jpa;

import cn.gtmap.landtax.util.QueryCondition;
import com.google.common.collect.Lists;
import com.mysema.query.jpa.JPQLQuery;
import com.mysema.query.jpa.impl.JPAQuery;
import com.mysema.query.types.EntityPath;
import com.mysema.query.types.Expression;
import com.mysema.query.types.OrderSpecifier;
import com.mysema.query.types.Predicate;
import com.mysema.query.types.path.PathBuilder;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Repository;

import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.persistence.criteria.CriteriaQuery;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.Iterator;
import java.util.List;


/**
 * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
 * @version 1.0, 2014/6/25
 */

@Repository
public class BaseRepositoryImpl_hl implements BaseRepository_hl {

    @PersistenceContext(unitName="oracle_hl")
    private EntityManager entityManager;

    private List<EntityPath> entityPathList = Lists.newLinkedList();


    @Override
    public <T> T get(Class<T> clazz, Object id) {
        return entityManager.find(clazz, id);
    }

    @Override
    public <T> void delete(Class<T> clazz, Object id) {
        if (id != null) {
            T entity = get(clazz, id);
            if (entity != null) {
                delete(entity);
            }
        }
    }

    @Override
    public <T> void delete(Class<T> clazz, Object[] ids) {
        if (ids != null) {
            for (Object id : ids) {
                delete(clazz, id);
            }
        }
    }

    @Override
    public <T> void delete(Object entity) {
        entityManager.remove(entity);
    }

    @Override
    public <T> List<T> dslList(EntityPath<T> entityPath) {
        return new JPAQuery(entityManager).from(entityPath).list(entityPath);
    }

    @Override
    public List<Object> dslList(JPQLQuery query, Expression expression) {
        return query.list(expression);
    }

    @Override
    public <T> List<T> dslList(EntityPath<T> entityPath, Expression<T> expression) {
        return expression != null ? new JPAQuery(entityManager).from(entityPath).list(expression) : dslList(entityPath);
    }

    @Override
    public <T> List<T> dslList(EntityPath<T> entityPath, Expression<T> expression, Predicate... predicates) {
        JPAQuery jpaQuery = new JPAQuery(entityManager).from(entityPath);
        if (predicates != null)
            jpaQuery = jpaQuery.where(predicates);
        return expression != null ? jpaQuery.list(expression) : jpaQuery.list(entityPath);
    }

    @Override
    public <T> List<T> sql(String sql) {
        return entityManager.createNativeQuery(sql).getResultList();
    }

    @Override
    public <T> List<T> sql(String sql, Class<T> clazz) {
        return entityManager.createNativeQuery(sql, clazz).getResultList();
    }

    @Override
    public void save(Object entity) {
        entityManager.persist(entity);
    }

    @Override
    public void update(Object entity) {
        entityManager.merge(entity);
    }

    @Override
    public Object get(JPQLQuery query) {
        return get(query,null);
    }

    @Override
    public Object get(JPQLQuery query, Expression expression) {
        return query.uniqueResult(expression);
    }


    @SuppressWarnings("unchecked")
    public <T> List<T> find(Class<T> clazz, List<QueryCondition> queryConditions, String orderBy, int currentPage, int pageSize) {
        Query query = getQuery(clazz, queryConditions, orderBy, false);
        if (currentPage == 0 && pageSize == 0) {
            return query.getResultList();
        } else {
            return query.setFirstResult((currentPage - 1) * pageSize).setMaxResults(pageSize).getResultList();
        }

    }

    @Override
    public <T> PageImpl<T> find(Class<T> clazz, List<QueryCondition> queryConditions, String orderBy, Pageable pageable) {
        Query query = getQuery(clazz, queryConditions, orderBy, false);
        if (pageable.getPageNumber() == 0 && pageable.getPageSize() == 0) {
            List<T> resultList = query.getResultList();
            return new PageImpl<T>(resultList, pageable, resultList.size());
        } else {
            return new PageImpl<T>(query.setFirstResult((pageable.getPageNumber() - 1) * pageable.getPageSize()).setMaxResults(pageable.getPageSize()).getResultList(), pageable, getRecordCount(clazz, queryConditions));
        }
    }

    @Override
    public <T> PageImpl<T> find(EntityPath<T> entityPath, Pageable pageable, Predicate... predicates) {
        return find(entityPath, pageable, null, predicates);
    }

    @Override
    public <T> PageImpl<T> find(EntityPath entityPath, Pageable pageable, Expression expression, Predicate... predicates) {
        JPQLQuery jpaQuery = predicates != null ? new JPAQuery(entityManager).from(entityPath).where(predicates) : new JPAQuery(entityManager).from(entityPath);
        if (pageable.getSort() != null){
            addEntityPath(entityPath);
            jpaQuery = applySorting(pageable.getSort(), jpaQuery);
        }
        List<T> resultList = null;
        if (pageable.getPageNumber() == 0 && pageable.getPageSize() == 0) {
            resultList = expression != null ? jpaQuery.list(expression) : jpaQuery.list(entityPath);
            return new PageImpl<T>(resultList, pageable, resultList.size());
        } else {
            resultList = expression != null ? jpaQuery.offset(pageable.getOffset() - pageable.getPageSize()).limit(pageable.getPageSize()).list(expression) : jpaQuery.offset((pageable.getPageNumber() - 1) * pageable.getPageSize()).limit(pageable.getPageSize()).list(entityPath);
            return new PageImpl<T>(resultList, pageable, jpaQuery.count());
        }
    }

    @Override
    public <T> Page<T> find(JPQLQuery jpaQuery, Expression expression, Pageable pageable) {
        return find(jpaQuery, expression, null, pageable);
    }

    @Override
    public <T> Page<T> find(JPQLQuery jpaQuery, Expression expression, Query countQuery, Pageable pageable) {
        List<T> resultList = null;
        if (pageable.getSort() != null)
            jpaQuery = applySorting(pageable.getSort(), jpaQuery);
        if (pageable.getPageNumber() == 0 && pageable.getPageSize() == 0) {
            resultList = expression != null ? jpaQuery.list(expression) : jpaQuery.list();
            return new PageImpl<T>(resultList, pageable, resultList.size());
        } else {
            resultList = expression != null ? jpaQuery.offset((pageable.getPageNumber() - 1) * pageable.getPageSize()).limit(pageable.getPageSize()).list(expression) : jpaQuery.offset((pageable.getPageNumber() - 1) * pageable.getPageSize()).limit(pageable.getPageSize()).list();
            long totalCount = countQuery != null ? getRecordCount(countQuery) : jpaQuery.distinct().count();
            return new PageImpl<T>(resultList, pageable, totalCount);
        }
    }

    @Override
    public <T> Page<T> find(Query query, Query countQuery, Pageable pageable) {
        List<T> resultList = null;
        if (pageable.getPageNumber() == 0 && pageable.getPageSize() == 0) {
            resultList = query.getResultList();
            return new PageImpl<T>(resultList, pageable, resultList.size());
        } else {
            return new PageImpl<T>(query.setFirstResult((pageable.getPageNumber() - 1) * pageable.getPageSize()).setMaxResults(pageable.getPageSize()).getResultList(), pageable, getRecordCount(countQuery));
        }
    }


    /**
     * 根据查询条件获取Query
     *
     * @param clazz
     * @param queryConditions
     * @param orderBy
     * @param isQueryTotal    是否查询记录总数, true 则查询记录总数
     * @return
     */
    @SuppressWarnings("rawtypes")
    private Query getQuery(Class clazz, List<QueryCondition> queryConditions, String orderBy, boolean isQueryTotal) {
        String className = clazz.getSimpleName();
        String preJPQL = isQueryTotal ? "select count(*) from " : "select o from ";
        StringBuffer jpql = new StringBuffer(preJPQL);
        jpql.append(className).append(" o where 1=1 ");
        Query query = null;
        if (queryConditions != null && queryConditions.size() > 0) {
            //构造jpql语句
            Iterator<QueryCondition> iterator = queryConditions.iterator();
            while (iterator.hasNext()) {
                QueryCondition queryCondition = iterator.next();
                if (queryCondition != null) {
                    if (queryCondition.getOperator().equals(QueryCondition.CUSTOM)) {
                        jpql.append(" and (").append(queryCondition.getCustomJPQL()).append(")");
                    }
                    if (queryCondition.getValue() != null && !"".equals(queryCondition.getValue())) {
                        //如果占位符名称是*.*格式，则换成*_*格式。且：和名称之间不能有空格
                        String placeholder = queryCondition.getField().indexOf(".") != -1 ? queryCondition.getField().replace(".", "_") : queryCondition.getField();
                        String operator = queryCondition.getOperator();
                        jpql.append(" and o.").append(queryCondition.getField().trim())
                                .append(" ").append(operator).append(":").append(placeholder.trim());
                    }
                }

            }
        }
        if (orderBy != null && !"".equals(orderBy)) {
            jpql.append(" ").append(orderBy);
        }

        query = entityManager.createQuery(jpql.toString());

        if (queryConditions != null && queryConditions.size() > 0) {
            //为参数赋值
            Iterator<QueryCondition> iterator2 = queryConditions.iterator();
            while (iterator2.hasNext()) {
                QueryCondition queryCondition = iterator2.next();
                if (queryCondition != null) {
                    if (queryCondition.getValue() != null && !"".equals(queryCondition.getValue())) {
                        //将占位符中的.替换成_
                        String queryConditionField = queryCondition.getField().indexOf(".") != -1 ? queryCondition.getField().replace(".", "_") : queryCondition.getField();
                        if (queryCondition.getOperator().equals(QueryCondition.LK)) {
                            query.setParameter(queryConditionField, "%" + queryCondition.getValue() + "%");
                        } else {
                            query.setParameter(queryConditionField, queryCondition.getValue());
                        }
                    }
                }

            }
        }
        return query;
    }


    public <T> List<T> list(Class<T> clazz, List<QueryCondition> queryConditions) {
        return find(clazz, queryConditions, null, 0, 0);
    }


    public <T> List<T> list(Class<T> clazz, List<QueryCondition> queryConditions, String orderBy) {
        return find(clazz, queryConditions, orderBy, 0, 0);
    }

    @Override
    public <T> List<T> list(CriteriaQuery<T> criteriaQuery) {
        return entityManager.createQuery(criteriaQuery).getResultList();
    }

    @SuppressWarnings("rawtypes")
    public Object getSingleResult(Class clazz, List<QueryCondition> queryConditions) {
        Query query = getQuery(clazz, queryConditions, null, false);
        return query.getSingleResult();
    }

    @SuppressWarnings("rawtypes")
    public long getRecordCount(Class clazz, List<QueryCondition> queryConditions) {
        Query query = getQuery(clazz, queryConditions, null, true);
        return getRecordCount(query);
    }

    public long getRecordCount(Query query) {
        long recordCount = 0L;
        Object result = query.getSingleResult();

        if (result != null) {
            if (result instanceof Long)
                recordCount = ((Long) result).longValue();
            else if (result instanceof BigDecimal)
                recordCount = ((BigDecimal) result).longValue();
        }
        return recordCount;
    }


    @SuppressWarnings("unchecked")
    public <T> List<T> getByJpql(String jpql, Object... objects) {
        Query query = entityManager.createQuery(jpql);
        if (objects != null) {
            if (objects != null) {
                for (int i = 0; i < objects.length; i++) {
                    query.setParameter(i, objects[i]);
                }
            }
        }
        return query.getResultList();
    }

    public int executeJpql(String jpql, Object... objects) {
        Query query = entityManager.createQuery(jpql);
        if (objects != null) {
            for (int i = 0; i < objects.length; i++) {
                query.setParameter(i, objects[i]);
            }
        }
        return query.executeUpdate();
    }

    public Object getUniqueResultByJpql(String jpql, Object... objects) {
        Query query = entityManager.createQuery(jpql);
        if (objects != null) {
            for (int i = 0; i < objects.length; i++) {
                query.setParameter(i, objects[i]);
            }
        }
        try {
            return query.getSingleResult();
        } catch (NoResultException e) {
            return null;
        }

    }

    private JPQLQuery applySorting(Sort sort, JPQLQuery query) {
        if (sort == null) {
            return query;
        }

        for (Sort.Order order : sort) {
            query.orderBy(toOrder(order));
        }

        return query;
    }

    private OrderSpecifier<?> toOrder(Sort.Order order) {
        Expression<Object> property = getOrderExpression(order.getProperty());
        return new OrderSpecifier(order.isAscending() ? com.mysema.query.types.Order.ASC
                : com.mysema.query.types.Order.DESC, property);
    }

    @Override
    public EntityManager getEntityManager() {
        return entityManager;
    }

    private PathBuilder getBuilder(String property) {
        for (EntityPath entityPath : entityPathList) {
            PathBuilder tmpBuilder = new PathBuilder(entityPath.getType(), entityPath.getMetadata());
            if (tmpBuilder.get(property) != null)
                return tmpBuilder;
        }
        return null;
    }

    private Expression<Object> getOrderExpression(String property) {
        for (EntityPath entityPath : entityPathList) {
            PathBuilder tmpBuilder = new PathBuilder(entityPath.getType(), entityPath.getMetadata());
            if (getClassField(entityPath.getType(), property) != null) {
                return tmpBuilder.get(property);
            }
        }
        return null;
    }

    public void addEntityPath(EntityPath... entityPaths) {
        entityPathList.clear();
        for (EntityPath entityPath : entityPaths) {
            if (!entityPathList.contains(entityPath))
                entityPathList.add(entityPath);
        }
    }

    public Field getClassField(Class clazz, String name) {
        Class superClazz = clazz.getSuperclass();
        try {
            return superClazz.getDeclaredField(name);
        } catch (NoSuchFieldException e) {
            try {
                return clazz.getDeclaredField(name);
            } catch (NoSuchFieldException e1) {
                return null;
            }
        }
    }
}
