package com.gtis.gserver.web;

import org.geowebcache.GeoWebCacheDispatcher;
import org.geowebcache.GeoWebCacheException;
import org.geowebcache.GeoWebCacheExtensions;
import org.geowebcache.conveyor.Conveyor;
import org.geowebcache.conveyor.ConveyorTile;
import org.geowebcache.grid.OutsideCoverageException;
import org.geowebcache.io.ByteArrayResource;
import org.geowebcache.io.FileResource;
import org.geowebcache.io.Resource;
import org.geowebcache.layer.TileLayer;
import org.geowebcache.layer.TileLayerDispatcher;
import org.geowebcache.mime.ImageMime;
import org.geowebcache.service.OWSException;
import org.geowebcache.service.Service;
import org.geowebcache.stats.RuntimeStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 12-6-19
 */
@Controller
public class ServiceController {
    public static final Logger LOG = LoggerFactory.getLogger(ServiceController.class);

    @Autowired
    private TileLayerDispatcher tileLayerDispatcher;
    @Autowired
    private RuntimeStats runtimeStats;

    private volatile Map<String, Service> services;
    private Resource blankTile;

    @ResponseBody
    @RequestMapping(value = "service/{serviceType}/**")
    public void service(
            @PathVariable("serviceType") String serviceType,
            HttpServletRequest request, HttpServletResponse response) throws Exception {
        Service service = getService(serviceType);
        ConveyorTile conv = (ConveyorTile) service.getConveyor(request, response);
        String layerName = conv.getLayerId();
        TileLayer layer = conv.getTileLayer();
        if (layer == null) {
            layer = tileLayerDispatcher.getTileLayer(layerName);
            conv.setTileLayer(layer);
        }
        if (!layer.isEnabled()) {
            throw new OWSException(400, "InvalidParameterValue", "LAYERS", "Layer '" + layerName + "' is disabled");
        }
        if (conv.reqHandler == Conveyor.RequestHandler.SERVICE) {
            service.handleRequest(conv);
        } else {
            layer.applyRequestFilters(conv);
            try {
                writeData(layer.getTile(conv));
            } catch (OutsideCoverageException e) {
                writeEmpty(conv, e.getMessage());
            }
        }
    }

    private void writeData(ConveyorTile tile) throws IOException {
        String ifNoneMatch = tile.servletReq.getHeader("If-None-Match");
        long ts = tile.getTSCreated();
        if (ts == 0) {
            Resource res = tile.getBlob();
            if (res instanceof FileResource) {
                ts = ((FileResource) res).lastModified();
            }
        }
        String hexTag = Long.toHexString(ts);
        if (ifNoneMatch != null && ifNoneMatch.equals(hexTag)) {
            tile.servletResp.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
            return;
        }
        tile.servletResp.setHeader("ETag", hexTag);
        tile.servletResp.setHeader("Cache-Control", "max-age=7200");
        tile.servletResp.addDateHeader("Expires", System.currentTimeMillis() + 7200 * 1000);
        writeFixedResponse(tile.servletResp, 200, tile.getMimeType().getMimeType(), tile.getBlob(), tile.getCacheResult());
    }

    private void writeEmpty(ConveyorTile tile, String message) {
        tile.servletResp.setHeader("geowebcache-message", message);
        TileLayer layer = tile.getLayer();
        if (layer != null) {
            layer.setExpirationHeader(tile.servletResp, (int) tile.getTileIndex()[2]);
            if (layer.useETags()) {
                String ifNoneMatch = tile.servletReq.getHeader("If-None-Match");
                if (ifNoneMatch != null && ifNoneMatch.equals("gwc-blank-tile")) {
                    tile.servletResp.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
                    return;
                } else {
                    tile.servletResp.setHeader("ETag", "gwc-blank-tile");
                }
            }
        }
        writeFixedResponse(tile.servletResp, 200, ImageMime.png.getMimeType(), this.blankTile, Conveyor.CacheResult.OTHER);
    }

    private void writeFixedResponse(HttpServletResponse response, int httpCode, String contentType, Resource resource, Conveyor.CacheResult cacheRes) {
        response.setStatus(httpCode);
        response.setContentType(contentType);
        if (resource != null) {
            int size = (int) resource.getSize();
            if (size > -1) {
                response.setContentLength(size);
            }
            try {
                OutputStream os = response.getOutputStream();
                resource.transferTo(Channels.newChannel(os));
                runtimeStats.log(size, cacheRes);
            } catch (IOException ioe) {
                LOG.debug("Caught IOException: " + ioe.getMessage() + "\n\n" + ioe.toString());
            }
        }
    }

    private Service getService(String serviceType) throws GeoWebCacheException {
        if (services == null) {
            synchronized (this) {
                if (services == null) {
                    this.services = new HashMap<String, Service>();
                    List<Service> services = GeoWebCacheExtensions.extensions(Service.class);
                    for (Service s : services) {
                        this.services.put(s.getPathName(), s);
                        LOG.info("Found service: [{}]", s.getPathName());
                    }
                }
            }
            try {
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                FileCopyUtils.copy(GeoWebCacheDispatcher.class.getResource("blank.png").openStream(), out);
                blankTile = new ByteArrayResource(out.toByteArray());
            } catch (IOException e) {
                LOG.error("Load blank tile error", e);
            }
        }
        Service service = services.get(serviceType);
        if (service == null) {
            throw new GeoWebCacheException("Unable to find handler for service:[" + serviceType + "]");
        }
        return service;
    }
}
