package com.gtis.gserver.service.impl;

import com.gtis.gserver.model.ArcgisLayer;
import com.gtis.gserver.service.LayerManager;
import org.apache.commons.lang.StringUtils;
import org.geowebcache.GeoWebCacheException;
import org.geowebcache.arcgis.layer.ArcGISCacheLayer;
import org.geowebcache.config.XMLConfiguration;
import org.geowebcache.grid.GridSetBroker;
import org.geowebcache.layer.TileLayer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 12-6-20
 */
public class LayerManagerImpl implements LayerManager {
    public static final Logger LOG = LoggerFactory.getLogger(LayerManagerImpl.class);

    @Autowired
    private XMLConfiguration configuration;
    @Autowired
    private GridSetBroker gridSetBroker;

    public List<TileLayer> getLayers() {
        List<TileLayer> tls = new ArrayList<TileLayer>(configuration.getTileLayers());
        Collections.sort(tls);
        return tls;
    }

    public TileLayer getLayer(String layerId) {
        return configuration.getTileLayer(layerId);
    }

    public void removeLayer(String layerId) {
        configuration.removeLayer(layerId);
        doSave();
    }

    public TileLayer saveLayer(ArcgisLayer layer) throws IOException {
        if (StringUtils.isBlank(layer.getId())) {
            throw new IOException("名称不能为空");
        }
        if (!layer.getPath().endsWith("/conf.xml")) {
            layer.setPath(layer.getPath() + "/conf.xml");
        }
        TileLayer tl = configuration.getTileLayer(layer.getId());
        if (!layer.getId().equals(layer.getSrcId()) && tl != null) {
            throw new IOException("此名称已被使用");
        }
        File file = new File(layer.getPath());
        if (!file.exists()) {
            throw new IOException("不存在的路径");
        }
        ArcGISCacheLayer al = StringUtils.isEmpty(layer.getSrcId()) ? new ArcGISCacheLayer() : (ArcGISCacheLayer) configuration.getTileLayer(layer.getSrcId());
        al.setName(layer.getId());
        al.setTilingScheme(file);
        al.setLayerBounds(layer.toBoundingBox());
        al.setOrder(layer.getOrder());
        if (StringUtils.isEmpty(layer.getSrcId())) {
            configuration.addLayer(al);
        } else {
            configuration.modifyLayer(tl);
        }
        doSave();
        return al;
    }

    public void stopLayer(String layerId) {
        getLayer(layerId).setEnabled(false);
        doSave();
    }

    public void startLayer(String layerId) {
        TileLayer tl = configuration.getTileLayer(layerId);
        if (tl instanceof ArcGISCacheLayer) {
            File f = ((ArcGISCacheLayer) tl).getTilingScheme();
            if (!f.exists()) {
                throw new RuntimeException("Arcgis缓存配置文件 [" + f.getAbsolutePath() + "]不存在");
            }
        }
        tl.setEnabled(true);
        configuration.modifyLayer(tl);
        doSave();
    }

    public void reload() {
        try {
            configuration.initialize(gridSetBroker);
        } catch (GeoWebCacheException e) {
            LOG.error("Error to reload configuration", e);
            throw new RuntimeException(e);
        }
    }

    private void doSave() {
        try {
            configuration.save();
        } catch (Exception e) {
            LOG.error("Error to save configuration", e);
            throw new RuntimeException(e);
        }
    }
}
