/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: AwtImageTransform.java
 *
 * LastModified: 2009-09-24 10:41:27
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.generic.image.impl;

import com.gtis.generic.image.ImageTransform;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.ImagingOpException;
import java.io.File;
import java.io.IOException;

/**
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 */
public class AwtImageTransform implements ImageTransform {

    private static final Logger logger = LoggerFactory.getLogger(AwtImageTransform.class);
    private BufferedImage srcBImage;
    private BufferedImage destBImage;
    private int imgWidth;
    private int imgHeight;

    public boolean load(String srcFilePath) {
        try {
            srcBImage = ImageIO.read(new File(srcFilePath));
            imgWidth = srcBImage.getWidth();
            imgHeight = srcBImage.getHeight();
        } catch (IOException e) {
            logger.info("Cannot open image [{}]", srcFilePath);
            return false;
        }
        return true;
    }

    public boolean save(String destFilePath) {
        if (destBImage == null)
            return false;
        try {
            String postfix = destFilePath.substring(destFilePath.lastIndexOf(".") + 1);
            ImageIO.write(destBImage, postfix, new File(destFilePath));
        } catch (IOException e) {
            logger.info("Cannot open image [{}]", destFilePath);
            return false;
        }
        return true;
    }

    public void resize(int width, int height) {
        double xRatio = (double) height / imgHeight;
        double yRatio = (double) width / imgWidth;
        double ratio = (xRatio < yRatio) ? xRatio : yRatio;
        zoomImage(imgWidth, imgHeight, ratio);
    }

    public void rotate(double rotate) {
        rotateImage(imgWidth, imgHeight, rotate, 0);
    }

    public void resizeWithMaxWidth(int maxWidth) {
        if (imgWidth > maxWidth) {
            double ratio = (double) maxWidth / imgWidth;
            zoomImage(imgWidth, imgHeight, ratio);
        } else
            destBImage = srcBImage;
    }

    public void rotateWithMaxWidth(double rotate, int maxWidth) {
        rotateImage(imgWidth, imgHeight, rotate, maxWidth);
    }

    private void zoomImage(int width, int height, double ratio) {
        try {
            int w = (int) (width * ratio);
            int h = (int) (height * ratio);
            destBImage = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = destBImage.createGraphics();
            g.setColor(Color.white);
            g.fillRect(0, 0, w, h);
            destBImage.getGraphics().drawImage(srcBImage.getScaledInstance(w, h, Image.SCALE_SMOOTH), 0, 0, null);

        } catch (Exception e) {
            throw new ImagingOpException("Unable to transform src image");
        }
    }

    private void rotateImage(int imgWidth, int imgHeight, double rotate, int maxWidth) {
        if (rotate < 1e-3) {
            return;
        }
        try {
            //zdd 原来的图形转换方法
//            double radian = rotate * Math.PI / 180;  //get radian
//            double w = Math.abs(imgHeight * Math.sin(radian)) + Math.abs(imgWidth * Math.cos(radian));  //get rotated height
//            double h = Math.abs(imgHeight * Math.cos(radian)) + Math.abs(imgWidth * Math.sin(radian));  //get rotated width
//            double ratio = 1;
//            if (maxWidth > 0) {
//                ratio = (double) maxWidth / w;
//            }
//            AffineTransform transform = new AffineTransform();
//            transform.setToScale(ratio, ratio);
//            transform.rotate(radian, w / 2, h / 2);
//            transform.translate(w / 2 - imgWidth / 2, h / 2 - imgHeight / 2);
//            AffineTransformOp ato = new AffineTransformOp(transform, null);
//            w *= ratio;
//            h *= ratio;
//            destBImage = new BufferedImage((int) w, (int) h, srcBImage.getType());
//            Graphics gs = destBImage.getGraphics();
//            gs.setColor(Color.white); //set canvas background to white
//            gs.fillRect(0, 0, (int) w, (int) h);
//            ato.filter(srcBImage, destBImage);

            /**
             * zdd 2016-05-13 新的图片旋转的方法  可以解决 有些图片在旋转后  颜色变化（模糊）问题。
             */
            //zdd 原图形的模板
            Rectangle srcRect = new Rectangle(new Dimension(imgWidth, imgHeight));
            double rotateTemp = rotate;
            if (rotate >= 90) {
                if (rotate / 90 % 2 == 1) {
                    int temp = srcRect.height;
                    srcRect.height = srcRect.width;
                    srcRect.width = temp;
                }
                rotateTemp = rotate % 90;
            }
            double radius = Math.sqrt(srcRect.height * srcRect.height + srcRect.width * srcRect.width) / 2;
            double len = 2 * Math.sin(Math.toRadians(rotateTemp) / 2) * radius;
            double angelAlpha = (Math.PI - Math.toRadians(rotateTemp)) / 2;
            double angelDaltaWidth = Math.atan((double) srcRect.height / srcRect.width);
            double angelDaltaHeight = Math.atan((double) srcRect.width / srcRect.height);
            int lenDaltaWidth = (int) (len * Math.cos(Math.PI - angelAlpha - angelDaltaWidth));
            int lenDaltaHeight = (int) (len * Math.cos(Math.PI - angelAlpha - angelDaltaHeight));
            int desWidth = srcRect.width + lenDaltaWidth * 2;
            int desHeight = srcRect.height + lenDaltaHeight * 2;

            //zdd 经过转换后生成目标图片的模板
            Rectangle desRect = new Rectangle(new Dimension(desWidth, desHeight));


            //zdd 设置目标图形的BufferedImage
            destBImage = new BufferedImage(desRect.width, desRect.height,
                    BufferedImage.TYPE_INT_RGB);
            //zdd 设置 图形转换的相关参数
            Graphics2D graphics = destBImage.createGraphics();

            double ratio = 1;
            if (maxWidth > 0) {
                ratio = (double) maxWidth / desRect.width;
            }
            AffineTransform transform = new AffineTransform();
            transform.setToScale(ratio, ratio);
            graphics.translate((desRect.width - imgWidth) / 2, (desRect.height - imgHeight) / 2);
            graphics.rotate(Math.toRadians(rotate), imgWidth / 2, imgHeight / 2);
            //zdd 将原图形按照一定规则输入到目标图形上
            graphics.drawImage(srcBImage, transform, null);
        } catch (Exception e) {
            throw new ImagingOpException("Unable to transform src image");
        }
    }
}
