/*
 * Project: tdsc
 *
 * File: Struts2Utils.java
 *
 * LastModified: 2010-04-28 03:48:32
 *
 * Copyright (c) 2010 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.generic.util;

import org.apache.commons.lang.StringUtils;
import org.apache.struts2.ServletActionContext;
import org.apache.struts2.json.JSONUtil;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.io.PrintWriter;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-4-28
 */
public class Struts2Utils {

    //-- header 常量定义 --//
    private static final String HEADER_ENCODING = "encoding";
    private static final String HEADER_NOCACHE = "no-cache";
    private static final String DEFAULT_ENCODING = "UTF-8";
    private static final boolean DEFAULT_NOCACHE = true;

    /**
     * 取得HttpSession中Attribute的简化函数.
     */
    public static Object getSessionAttribute(String name) {
        HttpSession session = getSession(false);
        return (session != null ? session.getAttribute(name) : null);
    }

    //-- 取得Request/Response/Session的简化函数 --//

    /**
     * 取得HttpSession的简化函数.
     */
    public static HttpSession getSession() {
        return ServletActionContext.getRequest().getSession();
    }

    /**
     * 取得HttpSession的简化函数.
     */
    public static HttpSession getSession(boolean isNew) {
        return ServletActionContext.getRequest().getSession(isNew);
    }

    /**
     * 取得HttpRequest的简化函数.
     */
    public static HttpServletRequest getRequest() {
        return ServletActionContext.getRequest();
    }

    /**
     * 取得HttpRequest中Parameter的简化方法.
     */
    public static String getParameter(String name) {
        return getRequest().getParameter(name);
    }

    /**
     * 取得HttpResponse的简化函数.
     */
    public static HttpServletResponse getResponse() {
        return ServletActionContext.getResponse();
    }

    /**
     * 直接输出文本.
     *
     * @see #render(String, String, String...)
     */
    public static void renderText(final String text, final String... headers) {
        render(ServletUtils.TEXT_TYPE, text, headers);
    }

    //-- 绕过jsp/freemaker直接输出文本的函数 --//

    /**
     * 直接输出内容的简便函数.
     * <p/>
     * eg.
     * render("text/plain", "hello", "encoding:GBK");
     * render("text/plain", "hello", "no-cache:false");
     * render("text/plain", "hello", "encoding:GBK", "no-cache:false");
     *
     * @param headers 可变的header数组，目前接受的值为"encoding:"或"no-cache:",默认值分别为UTF-8和true.
     */
    public static void render(final String contentType, final String content, final String... headers) {
        HttpServletResponse response = initResponseHeader(contentType, headers);
        try {
            response.getWriter().write(content);
            response.getWriter().flush();
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 分析并设置contentType与headers.
     */
    private static HttpServletResponse initResponseHeader(final String contentType, final String... headers) {
        //分析headers参数
        String encoding = DEFAULT_ENCODING;
        boolean noCache = DEFAULT_NOCACHE;
        for (String header : headers) {
            String headerName = StringUtils.substringBefore(header, ":");
            String headerValue = StringUtils.substringAfter(header, ":");

            if (StringUtils.equalsIgnoreCase(headerName, HEADER_ENCODING)) {
                encoding = headerValue;
            } else if (StringUtils.equalsIgnoreCase(headerName, HEADER_NOCACHE)) {
                noCache = Boolean.parseBoolean(headerValue);
            } else {
                throw new IllegalArgumentException(headerName + "不是一个合法的header类型");
            }
        }

        HttpServletResponse response = ServletActionContext.getResponse();

        //设置headers参数
        String fullContentType = contentType + ";charset=" + encoding;
        response.setContentType(fullContentType);
        if (noCache) {
            ServletUtils.setNoCacheHeader(response);
        }

        return response;
    }

    /**
     * 直接输出HTML.
     *
     * @see #render(String, String, String...)
     */
    public static void renderHtml(final String html, final String... headers) {
        render(ServletUtils.HTML_TYPE, html, headers);
    }

    /**
     * 直接输出js.
     *
     * @see #render(String, String, String...)
     */
    public static void renderJs(final String js, final String... headers) {
        render(ServletUtils.HTML_TYPE, "<script type=\"text/javascript\">" + js + "</script>", headers);
    }

    /**
     * 直接输出XML.
     *
     * @see #render(String, String, String...)
     */
    public static void renderXml(final String xml, final String... headers) {
        render(ServletUtils.XML_TYPE, xml, headers);
    }

    /**
     * 直接输出JSON.
     *
     * @param jsonString json字符串.
     * @see #render(String, String, String...)
     */
    public static void renderJson(final String jsonString, final String... headers) {
        render(ServletUtils.JSON_TYPE, jsonString, headers);
    }

    /**
     * 直接输出JSON,使用Jackson转换Java对象.
     *
     * @param data 可以是List<POJO>, POJO[], POJO, 也可以Map名值对.
     * @see #render(String, String, String...)
     */
    public static void renderJson(final Object data, final String... headers) {
        HttpServletResponse response = initResponseHeader(ServletUtils.JSON_TYPE, headers);
        try {
            PrintWriter out = response.getWriter();
            JSONUtil.serialize(response.getWriter(), data);
        } catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * 直接输出支持跨域Mashup的JSONP.
     *
     * @param callbackName callback函数名.
     * @param object       Java对象,可以是List<POJO>, POJO[], POJO ,也可以Map名值对, 将被转化为json字符串.
     */
    public static void renderJsonp(final String callbackName, final Object object, final String... headers) {
        String jsonString = null;
        try {
            jsonString = JSONUtil.serialize(object);
        } catch (Exception e) {
            throw new IllegalArgumentException(e);
        }

        String result = new StringBuilder().append(callbackName).append("(").append(jsonString).append(");").toString();

        //渲染Content-Type为javascript的返回内容,输出结果为javascript语句, 如callback197("{html:'Hello World!!!'}");
        render(ServletUtils.JS_TYPE, result, headers);
    }

    public static String getRemoteAddr() {
        System.out.println("-------------ip获取-------------------");
        HttpServletRequest request = ServletActionContext.getRequest();
        String remoteAddr = "";
        if (request!=null){
            remoteAddr = request.getHeader("x-forwarded-for");
            if (remoteAddr == null || remoteAddr.length() == 0 || "unknown".equalsIgnoreCase(remoteAddr)) {
                if (remoteAddr == null || remoteAddr.length() == 0 || "unknown".equalsIgnoreCase(remoteAddr)) {
                    remoteAddr = request.getHeader("Proxy-Client-IP");
                }
                if (remoteAddr == null || remoteAddr.length() == 0 || "unknown".equalsIgnoreCase(remoteAddr)) {
                    remoteAddr = request.getHeader("WL-Proxy-Client-IP");
                }
                if (remoteAddr == null || remoteAddr.length() == 0 || "unknown".equalsIgnoreCase(remoteAddr)) {
                    remoteAddr = request.getHeader("HTTP_CLIENT_IP");
                }
                if (remoteAddr == null || remoteAddr.length() == 0 || "unknown".equalsIgnoreCase(remoteAddr)) {
                    remoteAddr = request.getHeader("HTTP_X_FORWARDED_FOR");
                }
                if (remoteAddr == null || remoteAddr.length() == 0 || "unknown".equalsIgnoreCase(remoteAddr)) {
                    remoteAddr = request.getHeader("X-Real-IP");
                }

            } else if (remoteAddr.length() > 15) {
                String[] ips = remoteAddr.split(",");
                for (int index = 0; index < ips.length; index++) {
                    String strIp = (String) ips[index];
                    if (!("unknown".equalsIgnoreCase(strIp))) {
                        remoteAddr = strIp;
                        break;
                    }
                }
            }
            if (remoteAddr == null || remoteAddr.length() == 0 || "unknown".equalsIgnoreCase(remoteAddr)) {
                remoteAddr = request.getRemoteAddr();
            }

        }
        System.out.println("-------------ip获取："+remoteAddr);
        return remoteAddr;
    }
}