package com.gtis.support.hibernate;

import com.gtis.common.Page;
import org.hibernate.*;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.orm.hibernate3.SessionFactoryUtils;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-8-13
 */
@SuppressWarnings("unchecked")
public class HibernateDao {
// ------------------------------ FIELDS ------------------------------

    protected Logger logger = LoggerFactory.getLogger(getClass());

    protected SessionFactory sessionFactory;

// --------------------- GETTER / SETTER METHODS ---------------------

    /**
     * 取得sessionFactory.
     */
    public SessionFactory getSessionFactory() {
        return sessionFactory;
    }

// -------------------------- OTHER METHODS --------------------------

    /**
     * 执行HQL进行批量修改/删除操作.
     */
    public int batchExecute(final String hql, final Object... values) {
        return createQuery(hql, values).executeUpdate();
    }

    /**
     * 执行HQL进行批量修改/删除操作.
     *
     * @return 更新记录数.
     */
    public int batchExecute(final String hql, final Map<String, ?> values) {
        return createQuery(hql, values).executeUpdate();
    }

    /**
     * 根据查询HQL与参数列表创建Query对象.
     * <p/>
     * 本类封装的find()函数全部默认返回对象类型为T,当不为T时使用本函数.
     *
     * @param values 数量可变的参数,按顺序绑定.
     */
    public Query createQuery(final String queryString, final Object... values) {
        Assert.hasText(queryString, "queryString不能为空");
        Query query = getSession().createQuery(queryString);
        if (values != null) {
            for (int i = 0; i < values.length; i++) {
                query.setParameter(i, values[i]);
            }
        }
        return query;
    }

    /**
     * 根据查询HQL与参数列表创建Query对象.
     *
     * @param values 命名参数,按名称绑定.
     */
    public Query createQuery(final String queryString, final Map<String, ?> values) {
        Assert.hasText(queryString, "queryString不能为空");
        Query query = getSession().createQuery(queryString);
        if (values != null) {
            query.setProperties(values);
        }
        return query;
    }

    /**
     * 为Query添加distinct transformer.
     */
    public Query distinct(Query query) {
        query.setResultTransformer(CriteriaSpecification.DISTINCT_ROOT_ENTITY);
        return query;
    }

    /**
     * 为Criteria添加distinct transformer.
     */
    public Criteria distinct(Criteria criteria) {
        criteria.setResultTransformer(CriteriaSpecification.DISTINCT_ROOT_ENTITY);
        return criteria;
    }

    /**
     * 按HQL查询对象列表.
     *
     * @param values 数量可变的参数,按顺序绑定.
     */
    public <X> List<X> find(final String hql, final Object... values) {
        return createQuery(hql, values).list();
    }

    /**
     * 按HQL查询对象列表.
     *
     * @param values 命名参数,按名称绑定.
     */
    public <X> List<X> find(final String hql, final Map<String, ?> values) {
        return createQuery(hql, values).list();
    }

    /**
     * 按HQL查询唯一对象.
     *
     * @param values 数量可变的参数,按顺序绑定.
     */
    public <X> X findUnique(final String hql, final Object... values) {
        return (X) createQuery(hql, values).uniqueResult();
    }

    /**
     * 按HQL查询唯一对象.
     *
     * @param values 命名参数,按名称绑定.
     */
    public <X> X findUnique(final String hql, final Map<String, ?> values) {
        return (X) createQuery(hql, values).uniqueResult();
    }

    /**
     * Flush当前Session.
     */
    public void flush() {
        getSession().flush();
    }

    /**
     * 取得当前Session.
     */
    protected final Session getSession() throws DataAccessResourceFailureException, IllegalStateException {
        return SessionFactoryUtils.getSession(sessionFactory, true);
    }

    /**
     * 初始化对象.
     * 使用load()方法得到的仅是对象Proxy, 在传到View层前需要进行初始化.
     * 只初始化entity的直接属性,但不会初始化延迟加载的关联集合和属性.
     * 如需初始化关联属性,可实现新的函数,执行:
     * Hibernate.initialize(user.getRoles())，初始化User的直接属性和关联集合.
     * Hibernate.initialize(user.getDescription())，初始化User的直接属性和延迟加载的Description属性.
     */
    public void initProxyProperty(Object proxyProperty) {
        Hibernate.initialize(proxyProperty);
    }

    /**
     * 采用@Autowired按类型注入SessionFactory, 当有多个SesionFactory的时候Override本函数.
     */
    @Autowired
    public void setSessionFactory(final SessionFactory sessionFactory) {
        this.sessionFactory = sessionFactory;
        afterSessionFactorySet();
    }

    public <T> Page<T> search(Class clazz, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        Criteria criteria = getSession().createCriteria(clazz);
        if (criterions != null)
            for (Criterion c : criterions) {
                if (c != null)
                    criteria.add(c);
            }
        if (orders != null)
            for (Order order : orders) {
                if (order != null)
                    criteria.addOrder(order);
            }
        return search(criteria, start, limit);
    }

    public <T> Page<T> search(Criteria criteria, int start, int limit) {
        Page page = new Page(Page.toIndex(start, limit), limit);
        if (limit == Page.ALL) {
            List items = criteria.list();
            page.setItems(items);
            page.setTotalCount(items.size());
        } else {
            int totalCount = toInt(((Long) criteria.setProjection(Projections.rowCount()).uniqueResult()), 0);
            page.setTotalCount(totalCount);
            if (totalCount > 0) {
                criteria.setFirstResult(start).setMaxResults(limit);
                page.setItems(criteria.setProjection(null).setResultTransformer(CriteriaSpecification.ROOT_ENTITY).list());
            }
        }
        return page;
    }

    protected static int toInt(Long v, int def) {
        if (v == null)
            return def;
        return v.intValue();
    }

    protected void afterSessionFactorySet() {
    }
}
