package com.gtis.archive.service.impl;

import com.gtis.archive.core.ModelChangeEvent;
import com.gtis.archive.core.cache.Cache;
import com.gtis.archive.entity.Permission;
import com.gtis.archive.service.PermissionService;
import com.gtis.common.Page;
import com.gtis.plat.vo.UserInfo;
import com.gtis.support.hibernate.HibernateTemplate;
import com.gtis.web.SessionUtil;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-12-8
 */
public class PermissionServiceImpl extends HibernateTemplate<Permission, String> implements PermissionService, ApplicationEventPublisherAware {
    private Cache cache;
    private ApplicationEventPublisher publisher;

    public void setCache(Cache cache) {
        this.cache = cache;
    }

    public void setApplicationEventPublisher(ApplicationEventPublisher publisher) {
        this.publisher = publisher;
    }

    @Transactional
    public void grant(Permission[] permissions) {
        if (permissions == null) {
            return;
        }
        for (Permission perm : permissions) {
            if (hasPermission(perm))
                continue;
            save(perm);
            cache.put(perm, true);
        }
        publisher.publishEvent(new ModelChangeEvent(permissions));
    }

    @Transactional
    public void revoke(Permission[] permissions) {
        if (permissions == null) {
            return;
        }
        for (Permission perm : permissions) {
            if (!hasPermission(perm))
                continue;
            delete(cache.<Permission>getKey(perm));
            cache.put(perm, false);
        }
        publisher.publishEvent(new ModelChangeEvent(permissions));
    }

    public boolean hasPermission(String resourceId, String operation) {
        UserInfo user = SessionUtil.getCurrentUser();
        if (user.isAdmin())
            return true;
        String roleIds = user.getRoleIds();
        if (roleIds == null)
            return false;
        for (String roleId : roleIds.split(",")) {
            roleId = roleId.substring(1, roleId.length() - 1);
            if (hasPermission(new Permission(resourceId, roleId, operation)))
                return true;
        }
        return false;
    }

    @SuppressWarnings("unchecked")
    public List<Permission> getPermissions(String resourceId, String roleId) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId)) {
            criteria.add(Restrictions.eq("resourceId", resourceId));
        }
        if (StringUtils.isNotBlank(roleId)) {
            criteria.add(Restrictions.eq("roleId", roleId));
        }
        return criteria.list();
    }

    public Set<String> getReadableRoleIds(String resourceId) {
        Set<String> set = new HashSet<String>();
        for (Permission p : search(0, Page.ALL, new Order[]{}, Restrictions.eq("resourceId", resourceId), Restrictions.eq("operation", Permission.VIEW_PERMISSION)).getItems()) {
            set.add(p.getRoleId());
        }
        return set;
    }

    private Boolean hasPermission(Permission perm) {
        Boolean exist = cache.get(perm);
        if (exist == null) {
            for (Permission item : getPermissions(null, perm.getRoleId())) {//批量把某个role的perm提取出来缓存，提高db查询性能
                cache.put(item, true);
            }
            exist = cache.get(perm);
            if (exist == null)
                cache.put(perm, exist = false);
        }
        return exist;
    }
}
