package com.gtis.archive.service.impl;


import com.gtis.archive.Switch;
import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.core.ex.EntityNotFoundException;
import com.gtis.archive.entity.*;
import com.gtis.archive.service.*;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.lang.StringUtils;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.FileCopyUtils;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.StringReader;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-9-30
 */
@Service
public class GatewayServiceImpl implements GatewayService {
    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private AjhRangeService ajhRangeService;
    @Autowired
    private ArchiveService archiveService;
    @Autowired
    private OriginalService originalService;
    @Autowired
    private EntityService entityService;

    @Autowired
    private RecordAjhRangeService recordAjhRangeService;

    private HttpClient httpClient;

    public GatewayServiceImpl() {
        HttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
        connectionManager.getParams().setDefaultMaxConnectionsPerHost(20);
        connectionManager.getParams().setConnectionTimeout(30000);
        connectionManager.getParams().setSoTimeout(30000);
        httpClient = new HttpClient(connectionManager);
    }

    public String in(String content) throws Exception {
        logger.debug("gateway in:[\n{}\n]", content);
        org.dom4j.Document xmlDoc;
        try {
            SAXReader reader = new SAXReader();
            xmlDoc = reader.read(new StringReader(content)).getDocument();
            for (Object node : xmlDoc.selectNodes("/list/archive")) {
                Element element = (Element) node;
                String modelName = element.attributeValue("type");
                Archive archive = saveArchive(element, modelName);
                if (archive != null) {
                    for (Object node1 : element.selectNodes("doc")) {
                        Element el = (Element) node1;
                        saveDoc(el, archive.getId(), modelName);
                    }
                    for (Object node1 : element.selectNodes("file")) {
                        Element el = (Element) node1;
                        saveOriginal(el, archive.getId(), modelName);
                    }
                    //档案卷内附属信息
                    for (Object node1 : element.selectNodes("archiveinfo")) {
                        Element el = (Element) node1;
                        saveArchiveInfo(el, archive.getId(), modelName);
                    }
                }
            }
            for (Object node : xmlDoc.selectNodes("/list/doc")) {
                Element element = (Element) node;
                String modelName = element.attributeValue("type");
                Document doc = saveDoc(element, null, modelName);
                for (Object node1 : element.selectNodes("file")) {
                    Element el = (Element) node1;
                    saveOriginal(el, doc.getId(), modelName);
                }
            }
            xmlDoc.getRootElement().addAttribute("result", "succeed");
        } catch (Exception e) {
            xmlDoc = DocumentHelper.createDocument();
            Element result = xmlDoc.addElement("list");
            result.addAttribute("result", "fail");
            result.addAttribute("msg", e.getMessage());
        }
        String ret = xmlDoc.asXML();
        logger.debug("gateway ret:[\n{}\n]", ret);
        return ret;
    }

    public String out(String key) {

        Archive archive = archiveService.getSimpleArchive(key);
        StringBuilder sb = new StringBuilder();
        if (archive != null) {
            sb.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
            sb.append("<archive>\n");
            List<String> archiveFields = getFieldNames("Archive");
            for (String archiveField : archiveFields) {
                try {
                    sb.append("<field name=\"" + archiveField + "\">" + PropertyUtils.getProperty(archive, archiveField) + "</field>\n");
                } catch (Exception e) {
                    logger.error(e.toString());
                }
            }
            List<Original> originalList = originalService.getOriginals(archive.getId());
            for (Original original : originalList) {
                sb.append("<file id=\"" + original.getId() + "\"/>\n");
            }
            List<Document> documentList = archiveService.getArchiveDocuments(archive);
            List<String> documentFields = new ArrayList<String>();
            if (documentList.size() != 0)
                documentFields = getFieldNames("Document");
            for (Document document : documentList) {
                sb.append("<document>\n");
                for (String documentField : documentFields) {
                    try {
                        sb.append("<field name=\"" + documentField + "\">" + PropertyUtils.getProperty(document, documentField) + "</field>\n");
                    } catch (Exception e) {
                        logger.error(e.toString());
                    }
                }
                List<Original> originals = originalService.getOriginals(document.getId());
                for (Original original : originals) {
                    sb.append("<file id=\"" + original.getId() + "\"/>\n");
                }
                sb.append("</document>\n");
            }
            sb.append("</archive>");
            return sb.toString();
        }
        return null;
    }

    /**
     * 更新档案
     *
     * @param content the content of archive(before update,format the content)
     * @return
     */
    public String update(String content) {
        logger.debug("gateway before update: [\n{}\n]", content);
        org.dom4j.Document xmlDoc;
        try {
            SAXReader reader = new SAXReader();
            xmlDoc = reader.read(new StringReader(content)).getDocument();
            for (Object node : xmlDoc.selectNodes("/list/archive")) {
                Element element = (Element) node;
                String modelName = element.attributeValue("type");
                String id = getIdKey(element);
                Archive archive = archiveService.getArchive(modelName, id);
                if (archive == null) continue;
                archiveService.saveArchive((Archive) parseXmlField2Entity(archive, element));
                //卷内
                for (Object docNode : element.selectNodes("doc")) {
                    Element doc = (Element) docNode;
                    if (StringUtils.isNotBlank(getIdKey(doc))) {
                        //更新卷内
                        String docId = getIdKey(doc);
                        Document document = archiveService.getDocument(modelName, docId);
                        parseXmlField2Entity(document, doc);
                        document = archiveService.saveDocument(document);
                        //卷内原文
                        for (Object docOg : doc.selectNodes("file")) {
                            saveOriginal((Element) docOg, document.getId(), modelName);
                        }
                    } else {
                        //新增卷内
                        saveDoc(doc, archive.getId(), modelName);
                    }
                }
                //原文
                for (Object ogNode : element.selectNodes("file")) {
                    Element og = (Element) ogNode;
                    saveOriginal(og, archive.getId(), modelName);
                }
                element.addAttribute("result", "succeed");
            }
        } catch (Exception ex) {
            logger.error("更新档案异常【{}】", ex.getLocalizedMessage());
            xmlDoc = DocumentHelper.createDocument();
            Element result = xmlDoc.addElement("list");
            result.addAttribute("result", "fail");
            result.addAttribute("msg", ex.getMessage());
        }
        String res = xmlDoc.asXML();
        logger.debug("gateway after update: [\n{}\n]", res);
        return res;
    }

    public String preIn(String content) throws Exception {
        logger.debug("gateway in:[\n{}\n]", content);
        org.dom4j.Document xmlDoc;
        try {
            SAXReader reader = new SAXReader();
            xmlDoc = reader.read(new StringReader(content)).getDocument();
            for (Object node : xmlDoc.selectNodes("/list/archive")) {
                Element element = (Element) node;
                String modelName = element.attributeValue("type");
                Archive archive = saveDGDArchive(element, modelName);
                if (archive != null) {
                    for (Object node1 : element.selectNodes("doc")) {
                        Element el = (Element) node1;
                        saveDoc(el, archive.getId(), modelName);
                    }
                    for (Object node1 : element.selectNodes("file")) {
                        Element el = (Element) node1;
                        saveOriginal(el, archive.getId(), modelName);
                    }
                    //档案卷内附属信息
                    for (Object node1 : element.selectNodes("archiveinfo")) {
                        Element el = (Element) node1;
                        saveArchiveInfo(el, archive.getId(), modelName);
                    }
                }
            }
            for (Object node : xmlDoc.selectNodes("/list/doc")) {
                Element element = (Element) node;
                String modelName = element.attributeValue("type");
                Document doc = saveDoc(element, null, modelName);
                for (Object node1 : element.selectNodes("file")) {
                    Element el = (Element) node1;
                    saveOriginal(el, doc.getId(), modelName);
                }
            }
            xmlDoc.getRootElement().addAttribute("result", "succeed");
        } catch (Exception e) {
            xmlDoc = DocumentHelper.createDocument();
            Element result = xmlDoc.addElement("list");
            result.addAttribute("result", "fail");
            result.addAttribute("msg", e.getMessage());
        }
        String ret = xmlDoc.asXML();
        logger.debug("gateway ret:[\n{}\n]", ret);
        return ret;
    }

    public String preDispose(String modelName, String id, String mlh) throws Exception{
        if(StringUtils.isBlank(modelName)||StringUtils.isBlank(id)) return null;
        Archive archive=archiveService.getArchive(modelName,id);
        if(archive==null) return null;
        if(archive.getState()==3&&(archive.getMlh()!=null&&archive.getAjh()!=null)){  //撤销后重新进行处理，其案卷号和目录号都是保留的
            archive.setState(2);
            archiveService.saveArchive(archive);
        }else {
            archive.setState(2);
            RecordAjhRange recordAjhRange=recordAjhRangeService.getAvailable(mlh,modelName,archive.getDwdm());
            if(recordAjhRange!=null){
                   archive.setAjh(recordAjhRange.getAjh());
                   archive.setMlh(recordAjhRange.getMlh());
                   archiveService.saveArchive(archive);
                recordAjhRangeService.remove(recordAjhRange.getId());
            }else {
                AjhRange ajhRange;
                if(StringUtils.isBlank(mlh)){
                    ajhRange=getAvailAjhRange(archive,modelName);
                    if (ajhRange != null) {
                        int ajh = ajhRange.getNextValue();
                        archive.setMlh(ajhRange.getMlh());
                        archive.setAjh(ajh);
                        ajhRange.setCurrentValue(ajh);
                        archiveService.saveArchive(archive);
                        ajhRangeService.saveAjhRange(ajhRange);
                    } else {
                        throw new RuntimeException("ajh not available");
                    }
                }else {
                    ajhRange=ajhRangeService.getAvailableAjhRange(modelName,mlh);
                    if (ajhRange != null) {
                        int ajh = ajhRange.getNextValue();
                        archive.setMlh(ajhRange.getMlh());
                        archive.setAjh(ajh);
                        ajhRange.setCurrentValue(ajh);
                        archiveService.saveArchive(archive);
                        ajhRangeService.saveAjhRange(ajhRange);
                    } else {
                        throw new RuntimeException("ajh not available");
                    }
                }
            }

        }
        return null;
    }

    public String cancelDispose(String modelName, String id) throws Exception {
        if(StringUtils.isBlank(modelName)||StringUtils.isBlank(id)) return null;
        Archive archive=archiveService.getArchive(modelName,id);
        if(archive==null) return null;
        RecordAjhRange recordAjhRange=new RecordAjhRange();
        recordAjhRange.setAjh(archive.getAjh());
        recordAjhRange.setMlh(archive.getMlh());
        recordAjhRange.setDwdm(archive.getDwdm());
        recordAjhRange.setModelName(archive.getModelName());
        archive.setState(3);
        archive.setAjh(null);
        archive.setMlh(null);
        archiveService.saveArchive(archive);
        recordAjhRangeService.saveRecordAjh(recordAjhRange);
        return null;
    }

    public String dispose(String modelName, String id) throws Exception {
        if(StringUtils.isBlank(modelName)||StringUtils.isBlank(id)) return null;
        Archive archive=archiveService.getArchive(modelName,id);
        if(archive==null) return null;
        if(archive.getState()==2){
            archive.setState(0);
            archiveService.saveArchive(archive);
        }else {
           return "改条档案记录不处于预处理状态";
        }
        return "success";
    }

    private Archive saveDGDArchive(Element el, String modelName) {
        try {
            Archive archive = readArchive(el, modelName);
            archive.setDh(null);
            archive.setState(3);//****
            boolean returnId = StringUtils.isBlank(archive.getId());
            archiveService.saveArchive(archive);
            el.addAttribute("result", "succeed");
            if (returnId) {
                Element idField = el.addElement("field");
                idField.addAttribute("name", "id");
                idField.setText(archive.getId());
            }
            return archive;
        } catch (Exception e) {
            el.addAttribute("result", "fail");
            el.addAttribute("msg", e.getMessage());
        }

         return null;
    }


    /**
     * 获取包含id的field值
     *
     * @param element
     * @return
     */
    private String getIdKey(Element element) {
        for (Object node : element.selectNodes("field")) {
            Element el = (Element) node;
            if (el.attributeValue("name").equals("id")) {
                return el.getTextTrim();
            }
        }
        return null;
    }

    private Archive saveArchive(Element el, String modelName) {
        try {
            Archive archive = readArchive(el, modelName);
            archive.setDh(null);
            boolean returnId = StringUtils.isBlank(archive.getId());
            if ((archive.getAjh() == null && !EnvHolder.isEnable(Switch.ALLOW_NULL_DH))||archive.getMlh()!=null) {
                AjhRange ajhRange = getAvailAjhRange(archive, modelName);
                if (ajhRange != null) {
                    int ajh = ajhRange.getNextValue();
                    archive.setMlh(ajhRange.getMlh());
                    archive.setAjh(ajh);
                    ajhRange.setCurrentValue(ajh);
                    archiveService.saveArchive(archive);
                    ajhRangeService.saveAjhRange(ajhRange);
                } else {
                    throw new RuntimeException("ajh not available");
                }
            } else {
                archiveService.saveArchive(archive);
            }
            el.addAttribute("result", "succeed");
            if (returnId) {
                Element idField = el.addElement("field");
                idField.addAttribute("name", "id");
                idField.setText(archive.getId());
            }
            if (archive.getMlh() != null) {
                Element mlhField = el.addElement("field");
                mlhField.addAttribute("name", "mlh");
                mlhField.setText(archive.getMlh());
            }
            Element ajhField = el.addElement("field");
            ajhField.addAttribute("name", "ajh");
            ajhField.setText(String.valueOf(archive.getAjh()));
            return archive;
        } catch (Exception e) {
            el.addAttribute("result", "fail");
            el.addAttribute("msg", e.getMessage());
        }
        return null;
    }

    private AjhRange getAvailAjhRange(Archive archive, String modelName) {
        String defaultDwdm=archiveService.newArchive(modelName).getDwdm();
        AjhRange ajhRange = null;
        if (StringUtils.isNotBlank(archive.getDwdm()) && StringUtils.isNotBlank(archive.getMlh())) {
            if(archive.getDwdm()==defaultDwdm){     //如果单位代码使用的是默认值，则根据目录号获取案卷号
                ajhRange = ajhRangeService.getAvailableAjhRange(modelName, archive.getMlh());
            }else {
                ajhRange = ajhRangeService.getAvailableAjhRange(modelName, archive.getDwdm(), archive.getMlh());
            }
        }else if(StringUtils.isNotBlank(archive.getMlh())){
            ajhRange = ajhRangeService.getAvailableAjhRange(modelName, archive.getMlh());
        }else if (StringUtils.isNotBlank(archive.getDwdm())) {
            if(archive.getDwdm()==defaultDwdm){ //如果单位代码使用的是默认值，则根据modelName获取案卷号
                ajhRange = ajhRangeService.getAvailableAjhRange(modelName);
            }else{
                ajhRange = ajhRangeService.getAvailableAjhRangeByDwdm(modelName, archive.getDwdm());
            }
        }  else
            ajhRange = ajhRangeService.getAvailableAjhRange(modelName);
        return ajhRange;
    }

    /**
     * 解析xml中的field至对应实体类对象中
     *
     * @param entity
     * @param el
     * @return
     */
    private Object parseXmlField2Entity(Object entity, Element el) {
        if (entity == null) return null;
        for (Object node : el.selectNodes("field")) {
            Element field = (Element) node;
            String name = field.attributeValue("name");
            if (StringUtils.isBlank(name)) continue;
            try {
                BeanUtils.setProperty(entity, name, field.getTextTrim());
            } catch (Exception e) {
                logger.debug(e.getLocalizedMessage() + "字段值[{}]有误", name);
            }
        }
        return entity;
    }

    private Archive readArchive(Element el, String modelName) throws Exception {
        Archive archive = archiveService.newArchive(modelName);
        if (archive != null) {
            for (Object node : el.selectNodes("field")) {
                Element field = (Element) node;
                String name = field.attributeValue("name");
                if (StringUtils.isBlank(name))
                    continue;
                try {
                    BeanUtils.setProperty(archive, name, field.getTextTrim());
                } catch (Exception e) {
                    logger.debug(e.getLocalizedMessage() + "字段值[{}]有误", name);
                }
            }
        }

        return archive;
    }

    private Document saveDoc(Element el, String archiveId, String modelName) {
        try {
            Document doc = readDoc(el, modelName);
            boolean returnId = StringUtils.isBlank(doc.getId());
            if (archiveId != null)
                doc.setArchiveId(archiveId);
            archiveService.saveDocument(doc);
            el.addAttribute("result", "succeed");
            if (returnId) {
                Element idField = el.addElement("field");
                idField.addAttribute("name", "id");
                idField.setText(doc.getId());
            }
            return doc;
        } catch (Exception e) {
            el.addAttribute("result", "fail");
            el.addAttribute("msg", e.getMessage());
        }
        return null;
    }

    private Document readDoc(Element el, String modelName) throws Exception {
        Document entity = archiveService.newDocument(modelName);
        if (entity != null) {
            for (Object node : el.selectNodes("field")) {
                Element field = (Element) node;
                String name = field.attributeValue("name");
                if (StringUtils.isBlank(name))
                    continue;
                try {
                    BeanUtils.setProperty(entity, name, field.getTextTrim());
                } catch (Exception e) {
                    logger.debug(e.getLocalizedMessage() + "字段值[{}]有误", name);
                }

            }
        }
        return entity;
    }

    private Original saveOriginal(Element el, String ownerId, String modelName) {
        String url = el.attributeValue("url");
        String id = el.attributeValue("id");
        String status=el.attributeValue("status");
        try {
            GetMethod get = new GetMethod(url);
            get.getParams().setHttpElementCharset("iso-8859-1");
            httpClient.executeMethod(get);
            String fileName = new String(get.getResponseHeader("Content-Disposition").getValue().getBytes("iso-8859-1"));
            fileName = fileName.substring(fileName.indexOf("\"") + 1, fileName.lastIndexOf("\""));
            Original original = new Original();
            original.setName(fileName);
            original.setOwnerId(ownerId);
            if (StringUtils.isNotBlank(id)) {
                original.setId(id);
            }
            if(StringUtils.isNotBlank(status)){
                original.setStatus(status);
            }
            File tmpFile = File.createTempFile("archive_gw", null);
            FileCopyUtils.copy(get.getResponseBodyAsStream(), new BufferedOutputStream(new FileOutputStream(tmpFile)));
            originalService.saveOriginalFile(original, tmpFile, StringUtils.isNotBlank(id));
            tmpFile.delete();
            originalService.saveOriginal(original);
            Element idField = el.addElement("field");
            idField.addAttribute("name", "id");
            idField.setText(original.getId());
            get.releaseConnection();
        } catch (Exception e) {
            el.addAttribute("result", "fail");
            el.addAttribute("msg", e.getMessage());
        }
        return null;
    }

    private ArchiveInfo saveArchiveInfo(Element e, String ownerId, String modelName) {
        ArchiveInfo archiveInfo = null;
        String aiModelName = modelName.concat("_ai");
        try {
            archiveInfo = entityService.newInstance(aiModelName);
        } catch (Exception e1) {
            logger.error("实例化[{}]模型出错,检查该卷内模型是否配置！".concat(e1.getLocalizedMessage()), aiModelName);
        }
        if (archiveInfo == null) return null;
        archiveInfo.setArchiveId(ownerId);
        archiveInfo.setArchiveModel(modelName);
        for (Object node : e.selectNodes("field")) {
            Element field = (Element) node;
            String name = field.attributeValue("name");
            if (StringUtils.isBlank(name)) continue;
            try {
                BeanUtils.setProperty(archiveInfo, name, field.getTextTrim());
            } catch (Exception e1) {
                logger.debug(e1.getLocalizedMessage() + "字段值[{}]有误", name);
            }
        }
        entityService.save(archiveInfo);
        return archiveInfo;
    }

    private Object doUpdate(Object ob, String content, String modelName) {
        org.dom4j.Document xmlDoc;
        try {
            SAXReader reader = new SAXReader();
            xmlDoc = reader.read(new StringReader(content)).getDocument();
            for (Object node : xmlDoc.selectNodes("/" + modelName)) {
                Element element = (Element) node;
                for (Object node1 : element.selectNodes("field")) {
                    Element field = (Element) node1;
                    String name = field.attributeValue("name");
                    if (StringUtils.isBlank(name))
                        continue;
                    PropertyUtils.setProperty(ob, name, field.getTextTrim());
                }
            }
            return ob;
        } catch (Exception ex) {
            logger.debug(ex.toString());
        }
        return null;
    }

    private List<String> getFieldsName(Object o) {
        Field[] fields = o.getClass().getDeclaredFields();
        ArrayList fieldsName = new ArrayList();
        for (Field field : fields) {
            if (field.getModifiers() == 2) {
                fieldsName.add(field.getName());
            }
        }
        return fieldsName;
    }

    /**
     * 根据modelName获取对象属性列表
     *
     * @param modelName
     * @return
     */
    private List<String> getFieldNames(String modelName) {
        Field[] fields;
        ArrayList fieldsName = new ArrayList();
        if (modelName == "Archive") {
            fields = Archive.class.getDeclaredFields();

        } else if (modelName == "Document") {
            fields = Document.class.getDeclaredFields();
        } else {
            return null;
        }
        for (Field field : fields) {
            if (field.getModifiers() == 2) {
                fieldsName.add(field.getName());
            }
        }
        return fieldsName;
    }
}
