package com.gtis.archive.service.impl;

import com.gtis.archive.Constants;
import com.gtis.archive.Switch;
import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.core.environment.Environment;
import com.gtis.archive.entity.*;
import com.gtis.archive.service.ArchiveService;
import com.gtis.archive.service.OriginalPermissionService;
import com.gtis.archive.service.OriginalService;
import com.gtis.common.Page;
import com.gtis.generic.util.ImageUtils;
import com.gtis.plat.vo.UserInfo;
import com.gtis.support.hibernate.HibernateTemplate;
import com.gtis.web.SessionUtil;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.FileCopyUtils;
import org.springframework.util.ResourceUtils;

import java.net.URLEncoder;
import java.util.*;
import java.io.*;
import com.gtis.archive.core.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 11-11-30
 */
public class OriginalServiceImpl extends HibernateTemplate<Original, String> implements OriginalService {
    public static final String THUMB_PREFIX = "__thumb.jpg";
    public static final long MAX_SIZE = 1024 * 1024 * 5;
    public static final long TASK_SIZE = 1024 * 1024;
    public static final int MICRO_IMAGE_WIDTH = 96;
    public static final int MICRO_IMAGE_HEIGHT = 96;

    private final static Logger logger = LoggerFactory.getLogger(OriginalServiceImpl.class);
    private String previewServerUrl;

    private float percent = 0.f;

    @Autowired
    private ModelService modelService;
    @Autowired
    private ArchiveService archiveService;
    @Autowired
    private OriginalPermissionService originalPermissionService;
    @Autowired
    private TaskExecutor taskExecutor;

    @Autowired
    protected EntityService entityService;

    public void setPreviewServerUrl(String previewServerUrl) {
        this.previewServerUrl = previewServerUrl;
    }

    public float getPercent() {
        return percent;
    }

    public void setPercent(float percent) {
        this.percent = percent;
    }

    public Original getOriginal(String id) {
        return get(id);
    }

    @Transactional
    public void removeOriginal(String[] ids) {
        for (String id : ids) {
            Original original = getOriginal(id);
            if (Constants.ORIGINAL_POWER.equals(original.getStatus())) {
                logger.info("该原文处于只读状态，无法编辑,[original_id=" + original.getId() + ";original_name="
                        + original.getName() + "]");
            } else {
                if (original != null) {
                    if (!original.getPath().startsWith("${myOriginalPath}")) { //关联的文件不删除
                        String path = EnvHolder.getAppEnv().getExpr(original.getPath());
                        File file = getFile(path);
                        if (file != null && file.exists()&& !file.delete()) {
                            logger.info("delete file [" + file.getAbsolutePath() + "] error");
                        }
                        file = getFile(path + THUMB_PREFIX);
                        if (file != null && file.exists() && !file.delete()) {
                            logger.info("delete file [" + file.getAbsolutePath() + "] error");
                        }
                    }
                }
                delete(id);
                if (originalPermissionService.isOriginalPermissionEnable()) {
                    batchExecute("delete from DownloadLog d where d.originalId=?", id);
                    batchExecute("delete from OriginalPermission d where d.originalId=?", id);
                }
            }
        }
    }

    @Transactional
    public void saveOriginal(Original original) {
        original.setUpdateTime(new Date());
        save(original);
        if (originalPermissionService.isOriginalPermissionEnable()) {
            UserInfo user = SessionUtil.getCurrentUser();
            if (user != null && !user.isAdmin()) {
                OriginalPermission op = new OriginalPermission();
                op.setUserId(user.getId());
                op.setUserName(user.getUsername());
                op.setOperation("download");
                op.setOriginalId(original.getId());
                originalPermissionService.saveOriginalPermission(op);
            }
        }
        if (!original.canMakeThumb())
            return;
        String path = EnvHolder.getAppEnv().getExpr(original.getPath()) + THUMB_PREFIX;
        File file = getFile(path);
        if ((file == null || !file.exists()) && original.getFileSize() > TASK_SIZE) {
            createThumbFile(original);
        }
    }

    @Transactional
    public List<Original> getOriginals(String ownerId) {
        List<Original> list = syncOriginals(ownerId);
        if (list == null) {
            list = getDbOriginals(ownerId);
        }
        return list;
    }

    private List<Original> syncOriginals(String ownerId) {
        if (EnvHolder.isEnable(Switch.AUTO_IDENTIFY_ORIGINAL)) {
            List<Original> list = getDbOriginals(ownerId);
            Map<String, Original> fsOgs = getFsOriginals(ownerId);
            if (fsOgs.size() != list.size()) {
                for (Original og : list) {
                    if (fsOgs.containsKey(og.getName())) {
                        fsOgs.remove(og.getName());
                    } else {
                        //delete(og.getId()); //删除不存在的文件,暂时不处理
                    }
                }
                if (!fsOgs.isEmpty()) {
                    for (Original og : fsOgs.values()) {
                        og.setUpdateTime(new Date());
                        save(og);
                    }
                    return null;
                }
            }
            return list;
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    private List<Original> getDbOriginals(String ownerId) {
        return createCriteria(Restrictions.eq("ownerId", ownerId)).addOrder(Order.asc("name")).list();
    }

    private Map<String, Original> getFsOriginals(String ownerId) {
        Object owner = getOriginalOwner(ownerId);
        if (owner == null)
            owner = getOwner(ownerId);
        Model model = modelService.getModel(owner);
        String relativePath = getRelativePath(owner, model);
        //File storePath = getFile(model.getEnv().get("originalRoot_2") + "/" + relativePath);
        File storePath = null ;
        Map<String, Original> map = new HashMap<String, Original>();

        for (String pathName : getOriginalPath(model)) {
            storePath = getFile(model.getEnv().get(pathName) + "/" + relativePath);
            if (storePath.exists()) {
                File[] fs = storePath.listFiles();
                if (fs != null) {
                    for (File file : fs) {
                        if (file.isDirectory())
                            continue;
                        String name = file.getName();
                        if (name.endsWith(THUMB_PREFIX) || "thumb.db".equals(name))
                            continue;
                        Original og = new Original();
                        og.setOwnerId(ownerId);
                        og.setName(file.getName());
                        og.setFileSize(file.length());
                        og.setPath("${originalRoot}/" + relativePath + "/" + og.getName());
                        og.setOwnerModelName(model.getName());
                        map.put(file.getName(), og);
                    }
                }
            }
        }
        return map;
    }

    public Object getOriginalOwner(String id) {
        Archive archive = archiveService.getSimpleArchive(id);
        return archive != null ? archive : archiveService.getSimpleDocument(id);
    }

    public  Object getOriginalOwner(String id, String modelName){
        if(StringUtils.isBlank(modelName)||"Original".equals(modelName)){
            Archive archive = archiveService.getSimpleArchive(id);
            return archive != null ? archive : archiveService.getSimpleDocument(id);
        }else{
            Archive archive = archiveService.getArchive(modelName,id);
            return archive != null ? archive : archiveService.getDocument(Archive.toDocumentModelName(modelName),id);
        }
    }

    public Object getOwner(String ownerId) {
        Object entity = entityService.load("ythsw", ownerId);
        if (entity == null)
            entity = entityService.load("ythfw", ownerId);
        if (entity == null)
            entity = entityService.load("ythnw", ownerId);
        return entity;
    }

    @Transactional
    public File getOriginalFile(String id, String action) {
        Original original = getOriginal(id);
        String path = EnvHolder.getAppEnv().getExpr(original.getPath());
        if (originalPermissionService.isOriginalPermissionEnable()) {
            saveDownloadLog(original, action);
        }
        return getFile(path);
    }

    public File getOriginalThumbFile(String id) {
        Original original = getOriginal(id);
        String path = EnvHolder.getAppEnv().getExpr(original.getPath()) + THUMB_PREFIX;
        File file = getFile(path);
        if ((file == null || !file.exists()) && original.isImage() && original.getFileSize() < TASK_SIZE) {
            createThumbFile(original);
        }
        return getFile(path);
    }


    /**
     * 获取变量
     * @param evn
     * @return
     */
    private Collection getSet(Environment evn){
        Collection set=null;
         if(evn.getProps().size()==0){
            set=getSet(evn.getParent());
         } else{
             set= evn.getProps().keySet();
         }
        return set;
    }

    public void saveOriginalFile(Original original, File file, boolean overWrite) {
        Object owner = getOriginalOwner(original.getOwnerId());
        if (owner == null)
            owner = getEntity(original);
        if (owner == null) {
            throw new IllegalStateException("Owner for original [" + original + "] is null");
        }
        Model model = modelService.getModel(owner);
        try{
            original.setOwnerModelName(PropertyUtils.getProperty(owner,"modelName").toString());
        }catch (Exception ex){
            original.setOwnerModelName(model.getName());
        }
        String relativePath = getRelativePath(owner, model);
        Collection set =getSet(model.getEnv().getParent());
        List<String> al = new ArrayList<String>();
        al.addAll(set);
        for(Iterator i=al.iterator();i.hasNext();){
            String s = (String) i.next();
            if(!s.startsWith("originalRoot_")){
                i.remove();
            }
        }
        File path = null;
        for(int i=0;i<al.size();i++){
            path = getFile(model.getEnv().get(al.get(i)) + "/" + relativePath);
            if((path.getUsableSpace()) >= (original.getFileSize())){

                if (file != null && !path.exists())
                    if (!path.mkdirs())
                        throw new RuntimeException("mkdir [" + relativePath + "] error");
                String fileName = original.getName();
                File f = getFile(path + "/" + fileName);
                try {
                    if (f != null && f.exists() && !overWrite) {
                        String rename = rename(f, original.getName());
                        f = getFile(path + "/" + rename);
                        original.setName(rename);
                    }
                    FileCopyUtils.copy(file, f);
                    original.setPath("${"+al.get(i)+"}/" + relativePath + "/" + original.getName());
//                    original.setOwnerModelName(model.getName());
                    if (f != null)
                        original.setFileSize(f.length());
                    break;
                } catch (IOException ignored) {
                }
            } else{
                continue;
            }

        }
    }

    @Override
    public void saveOriginalFile(Original original, File file, boolean overWrite, String modelName) {
//        Object owner = getOriginalOwner(original.getOwnerId(),modelName);
        Object owner = getOriginalOwner(original.getOwnerId());
        if (owner == null)
            owner = getEntity(original);
        if (owner == null) {
            throw new IllegalStateException("Owner for original [" + original + "] is null");
        }
        Model model = modelService.getModel(owner);
//        try{
//            original.setOwnerModelName(PropertyUtils.getProperty(owner,"modelName").toString());
//        }catch (Exception ex){
//            original.setOwnerModelName(model.getName());
//        }
        original.setOwnerModelName(modelName);
        String relativePath = getRelativePath(owner, model);
        Collection set =getSet(model.getEnv().getParent());
        List<String> al = new ArrayList<String>();
        al.addAll(set);
        for(Iterator i=al.iterator();i.hasNext();){
            String s = (String) i.next();
            if(!s.contains("originalRoot")){
//            if(!s.startsWith("originalRoot_")){
                i.remove();
            }
        }
        File path = null;
        for(int i=0;i<al.size();i++){
            path = getFile(model.getEnv().get(al.get(i)) + "/" + relativePath);
            if((path.getUsableSpace()) >= (original.getFileSize())){

                if (file != null && !path.exists())
                    if (!path.mkdirs())
                        throw new RuntimeException("mkdir [" + relativePath + "] error");
                String fileName = original.getName();
                File f = getFile(path + "/" + fileName);
                try {
                    if (f != null && f.exists() && !overWrite) {
                        String rename = rename(f, original.getName());
                        f = getFile(path + "/" + rename);
                        original.setName(rename);
                    }
                    FileCopyUtils.copy(file, f);
                    original.setPath("${"+al.get(i)+"}/" + relativePath + "/" + original.getName());
//                    original.setOwnerModelName(model.getName());
                    if (f != null)
                        original.setFileSize(f.length());
                    break;
                } catch (IOException ignored) {
                }
            } else{
                continue;
            }

        }
    }

    /**
     * 获取文档一体化实体对象
     *
     * @param original
     * @return
     */
    public Object getEntity(Original original) {
        Object entity = entityService.load("ythsw", original.getOwnerId());
        if (entity == null)
            entity = entityService.load("ythfw", original.getOwnerId());
        if (entity == null)
            entity = entityService.load("ythnw", original.getOwnerId());
        return entity;
    }

    private String getRelativePath(Object owner, Model model) {
        String relativePath;
        if (owner instanceof Archive) {
            Archive archive = (Archive) owner;
            relativePath = new Environment(archive, model.getEnv()).get("archivePath");
            if (relativePath == null) {
                relativePath = model.getEnv().getExpr("archive/${.now?string('yyyy/MM/dd')}/" + archive.getId());
            }
        } else if (owner instanceof Document) {
            Document doc = (Document) owner;
            if (doc.getArchiveId() != null) {
                Archive archive = archiveService.getSimpleArchive(doc.getArchiveId());
                relativePath = new Environment(archive, model.getEnv()).get("archivePath");
                if (relativePath == null) {
                    relativePath = model.getEnv().getExpr("archive/${.now?string('yyyy/MM/dd')}/" + doc.getArchiveId() + "/" + doc.getId());
                } else {
                    relativePath += "/" + (doc.getYh() == null ? doc.getId() : doc.getYh());
                }
            } else {
                relativePath = model.getEnv().getExpr("document/${.now?string('yyyy/MM/dd')}/" + doc.getId());
            }
        } else {
            Object entity = owner;
            relativePath = new Environment(entity, model.getEnv()).get("genericPath");
            if (relativePath == null) {
                try {
                    relativePath = model.getEnv().getExpr("generic/${.now?string('yyyy/MM/dd')}/" + PropertyUtils.getProperty(entity, "id"));
                } catch (Exception e) {
                    relativePath = null;
                }
            }
        }
        if (relativePath == null)
            throw new RuntimeException("Original save path unset,please check");
        return relativePath;
    }

    @Transactional
    public Page<Original> searchOriginal(String ownerId, String name, int start, int limit) {
        if (originalPermissionService.isOriginalPermissionEnable()) {
            UserInfo user = SessionUtil.getCurrentUser();
            if (!user.isAdmin()) {
                return originalPermissionService.searchOriginal(ownerId, name, start, limit);
            }
        }
        List<Criterion> list = new ArrayList<Criterion>();
        list.add(Restrictions.eq("ownerId", ownerId));
        if (StringUtils.isBlank(name)) {
            syncOriginals(ownerId);
        } else {
            list.add(Restrictions.like("name", name, MatchMode.ANYWHERE));
        }
        return search(list, Collections.singletonList(Order.asc("name")), start, limit);
    }


    public boolean hasOriginal(String ownerId) {
        List<Original> list = find(Restrictions.eq("ownerId", ownerId));
        if (list.size() > 0)
            return true;
        return false;
    }

    public Object[] getIds(String ownerId) {
        List<Original> originals=createCriteria(Restrictions.eq("ownerId", ownerId)).addOrder(Order.asc("name")).list();
        List<String> ids=new ArrayList<String>();
        for(Original original :originals){
            ids.add(original.getId());
        }
        return  ids.toArray();
    }

    private void createThumbFile(Original original) {
        final String path = EnvHolder.getAppEnv().getExpr(original.getPath());
        if (!new File(path).exists())
            return;
        if (original.isImage() && original.getFileSize() < MAX_SIZE) {
            final String destPath = path + THUMB_PREFIX;
            try {
                if (original.getFileSize() > TASK_SIZE) {
                    taskExecutor.execute(new Runnable() {
                        public void run() {
                            try {
                                ImageUtils.resizeImage(path, destPath, MICRO_IMAGE_WIDTH, MICRO_IMAGE_HEIGHT);
                            } catch (Exception e) {
                                logger.error("Unable to transform image:[" + path + "] to [" + destPath + "]", e);
                            }
                        }
                    });

                } else {
                    ImageUtils.resizeImage(path, destPath, MICRO_IMAGE_WIDTH, MICRO_IMAGE_HEIGHT);
                }
            } catch (Exception ex) {
                logger.error("Unable to transform image:[" + path + "] to [" + destPath + "]", ex);
            }
        } else {
            if (previewServerUrl == null) {
                logger.warn("previewServerUrl not config");
                return;
            }
            taskExecutor.execute(new Runnable() {
                public void run() {
                    String url = null;
                    try {
                        url = previewServerUrl + "?path=" + URLEncoder.encode(path, "gbk");
                        HttpClient httpClient = new HttpClient();
                        HttpMethod method = new GetMethod(url);
                        if (httpClient.executeMethod(method) != HttpStatus.SC_OK)
                            logger.debug("Error to connect {}", url);
                        else
                            logger.debug("Get {} success", url);
                    } catch (Exception e) {
                        logger.error("Error to connect {" + url + "} " + e.getMessage());
                    }
                }
            });
        }
    }

    private void saveDownloadLog(Original original, String action) {
        UserInfo user = SessionUtil.getCurrentUser();
        if (user != null) {
            List<Criterion> criterions = new ArrayList<Criterion>();
            criterions.add(Restrictions.eq("originalId", original.getId()));
            criterions.add(Restrictions.eq("userId", user.getId()));
            criterions.add(Restrictions.eq("action", action));
            Page<DownloadLog> page = search(DownloadLog.class, criterions, null, 0, 1);
            DownloadLog dl = null;
            if (!page.isEmpty()) {
                dl = page.getItem(0);
                if (DateUtils.addMinutes(dl.getDownloadTime(), 2).after(new Date())) {
                    dl.setCount(dl.getCount() + 1);
                } else {
                    dl = null;
                }
            }
            if (dl == null) {
                dl = new DownloadLog();
                dl.setOriginalId(original.getId());
                dl.setUserId(user.getId());
                dl.setUserName(user.getUsername());
                dl.setAction(action);
                dl.setCount(1);
            }
            dl.setDownloadTime(new Date());
            getSession().save(dl);
        }
    }

    private static File getFile(String path) {
        try {
            return ResourceUtils.getFile(path);
        } catch (Exception e) {
            logger.error("Get File error,path [" + path + "]", e);
        }
        return null;
    }

    private static String rename(File f, String fileName) {
        int index = fileName.lastIndexOf('.');
        String name = index > -1 ? fileName.substring(0, index) : fileName;
        String ext = index > -1 ? fileName.substring(index) : "";
        String name1 = fileName;
        for (int i = 1; new File(f.getParentFile(), name1).exists(); i++) {
            name1 = name + "_" + i + ext;
        }
        return name1;
    }

    List<String> getOriginalPath(Model model){
        Collection set =getSet(model.getEnv().getParent());
        List<String> path = new ArrayList<String>();
        List<String> al = new ArrayList<String>();
        al.addAll(set);
        for(Iterator iter =al.iterator();iter.hasNext();){
            String s = (String) iter.next();
            if(s.contains("originalRoot_")){
                path.add(s) ;
            }
        }
        return path ;
    }

    /**
     * 批量同步原文的信息
     * 1.获取所有的档案
     * 2.便利档案的文件存放路径，是否包含文件
     * 3.包含文件的加入原文表信息
     * @param size
     * @throws Exception
     */
    @Transactional
    public void batchSyncOriginal(int size) throws Exception{
        int count = 0 ;
        int start = 0 ;
        int end = 100 ;
        for(int i=0; i<end; i++){
            List<Archive> archives = null ;

            if(end==size){
                archives = archiveService.findAllArchive(start,size%100) ;
            }else {
                archives = archiveService.findAllArchive(start, 100);
            }
            if(archives!=null){
                for(Archive archive : archives){
                    Object owner = getOriginalOwner(archive.getId());
                    if (owner == null)
                        owner = getOwner(archive.getId());
                    Model model = modelService.getModel(owner);
                    String relativePath = null ;
                    File storePath = null ;

                    if(archive.getQzh()!=null && archive.getMlh()!=null && archive.getAjh()!=null){
                        relativePath = getRelativePath(owner, model);
                        //storePath = getFile(model.getEnv().get("originalRoot_2") + "/" + relativePath);
                        Map<String, Original> map = new HashMap<String, Original>();
                        for(String pathName : getOriginalPath(model)){
                            storePath = getFile(model.getEnv().get(pathName) + "/" + relativePath);
                            if (storePath.exists()) {
                                File[] fs = storePath.listFiles();
                                if (fs != null) {
                                    for (File file : fs) {
                                        if (file.isDirectory())
                                            continue;
                                        String name = file.getName();
                                        if (name.endsWith(THUMB_PREFIX) || "thumb.db".equals(name))
                                            continue;
                                        Original og = new Original();
                                        og.setOwnerId(archive.getId());
                                        og.setName(file.getName());
                                        og.setFileSize(file.length());
                                        og.setPath("${"+ pathName+"}/" + relativePath + "/" + og.getName());
                                        og.setOwnerModelName(model.getName());
                                        map.put(file.getName(), og);
                                    }
                                }
                            }
                        }


                        List<Original> list = getDbOriginals(archive.getId());

                        if (map.size() != list.size()) {
                            for (Original og : list) {
                                if (map.containsKey(og.getName())) {
                                    map.remove(og.getName());
                                } else {
                                    //delete(og.getId()); //删除不存在的文件,暂时不处理
                                }
                            }
                            if (!map.isEmpty()) {
                                for (Original og : map.values()) {
                                    og.setUpdateTime(new Date());
                                    save(og);
                                }
                            }
                        }

                        count++ ;
                        //System.out.println(size+"-->"+count+"==="+archive.getId());

                        setPercent(count/(float)size);
                        list.clear();


                    }else{
                        //System.out.println(size+"-->kkkfkfk"+count+"==="+archive.getId());
                        count++ ;
                        setPercent(count/(float)size);
                        continue;
                    }


                }

                archives.clear();
            }

            if(end<size&&(end+100)<size){
                start = end+1;
                end = end + 100 ;
            }else if(end<size&&(end+100)>=size){
                start = end+1;
                end = size ;
            }else{
                break;
            }
        }

    }

    @Override
    public float batchSyncOriginalPercent() {
        return getPercent() ;
    }
}
