package cn.gtmap.hlw.core.util.pdf;

import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.font.PDType0Font;


import java.io.*;
import java.util.Base64;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2025/11/26
 * @description
 */
public class PdfSignatureUtils {

    /**
     * 在PDF右上角添加权利人签字（Base64输入输出）
     * @param pdfBase64 输入PDF的Base64字符串
     * @param signerName 权利人姓名
     * @return 添加签字后的PDF Base64字符串
     */
    public static String addSignatureToPdfBase64(String pdfBase64, String signerName) {
        try {
            // Base64解码
            byte[] pdfBytes = Base64.getDecoder().decode(pdfBase64);

            // 处理PDF
            byte[] signedPdfBytes = addSignatureToPdf(pdfBytes, signerName);

            // Base64编码返回
            return Base64.getEncoder().encodeToString(signedPdfBytes);

        } catch (Exception e) {
            throw new RuntimeException("处理PDF签字失败", e);
        }
    }



    /**
     * 在PDF右上角添加权利人签字
     * @param pdfBytes 输入PDF字节数组
     * @param signerName 权利人姓名
     * @return 添加签字后的PDF字节数组
     */
    private static byte[] addSignatureToPdf(byte[] pdfBytes, String signerName) {
        try (PDDocument document = PDDocument.load(new ByteArrayInputStream(pdfBytes));
             ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {

            // 获取第一页
            PDPage page = document.getPage(0);
            PDRectangle pageSize = page.getMediaBox();

            // 在PDF上添加签字内容
            try (PDPageContentStream contentStream = new PDPageContentStream(
                    document, page, PDPageContentStream.AppendMode.APPEND, true, true)) {
                InputStream inputStream = PdfSignatureUtils.class.getResourceAsStream("/fonts/simsun.ttf");
                // 设置字体 - PDFBox 3.x 方式
                PDType0Font chineseFont = PDType0Font.load(document,inputStream,true); // true=嵌入字体


                // 计算右上角位置
                float x = pageSize.getWidth() - 200; // 距离右边150点
                float y = pageSize.getHeight() - 50; // 距离顶部50点

                // 添加姓名（稍大字号）
                contentStream.beginText();
                contentStream.setFont(chineseFont, 16);
                contentStream.newLineAtOffset(x, y);
                contentStream.showText(signerName);
                contentStream.endText();
            }

            // 保存到字节数组
            document.save(outputStream);
            return outputStream.toByteArray();

        } catch (IOException e) {
            throw new RuntimeException("处理PDF文件失败", e);
        }
    }
}
