package cn.gtmap.hlw.core.util.es;

import com.alibaba.fastjson.JSON;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.indices.CreateIndexRequest;
import org.elasticsearch.client.indices.CreateIndexResponse;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2025/3/7
 * @description
 */
@Component
public class ESClient {

    @Autowired
    private RestHighLevelClient client;

    /**
     * 创建索引
     *
     * @param index   索引名
     * @param settings 分片/副本等设置
     * @param mappingJson 索引 mapping（JSON 字符串）
     * @return true-创建成功；false-已存在或其它失败
     */
    public boolean createIndex(String index, Settings.Builder settings, String mappingJson) throws IOException {
        CreateIndexRequest request = new CreateIndexRequest(index)
                .settings(settings)
                .mapping(mappingJson, XContentType.JSON);
        try {
            CreateIndexResponse resp = client.indices().create(request, RequestOptions.DEFAULT);
            return resp.isAcknowledged();
        } catch (ElasticsearchStatusException e) {
            if (e.status() == RestStatus.BAD_REQUEST &&
                    e.getMessage().contains("resource_already_exists_exception")) {
                System.out.println("索引已存在，跳过创建。");
                return true;
            }
            throw e;
        }
    }

    /**
     * 删除索引
     *
     * @param index 索引名
     * @return true-删除成功
     */
    public boolean deleteIndex(String index) throws IOException {
        DeleteIndexRequest request = new DeleteIndexRequest(index);
        AcknowledgedResponse resp = client.indices().delete(request, RequestOptions.DEFAULT);
        return resp.isAcknowledged();
    }

    /**
     * 写入/更新文档（指定 ID）
     *
     * @param index 索引
     * @param id    文档 ID
     * @param doc   文档对象（会 JSON 序列化）
     * @return 写入响应
     */
    public IndexResponse createDocument(String index, String id, Object doc) throws IOException {
        IndexRequest request = new IndexRequest(index)
                .id(id)
                .source(JSON.toJSONString(doc), XContentType.JSON);
        return client.index(request, RequestOptions.DEFAULT);
    }

    /**
     * 写入文档（自生成 ID）
     *
     * @param index 索引
     * @param doc   文档对象
     * @return 写入响应
     */
    public IndexResponse createDocument(String index, Object doc) throws IOException {
        IndexRequest request = new IndexRequest(index)
                .source(JSON.toJSONString(doc), XContentType.JSON);
        return client.index(request, RequestOptions.DEFAULT);
    }

    /**
     * 根据 ID 查询文档
     *
     * @param index 索引
     * @param id    文档 ID
     * @return GetResponse（含源文档）
     */
    public GetResponse getDocument(String index, String id) throws IOException {
        GetRequest request = new GetRequest(index, id);
        return client.get(request, RequestOptions.DEFAULT);
    }

    /**
     * 删除文档
     *
     * @param index 索引
     * @param id    文档 ID
     * @return true-删除成功
     */
    public boolean deleteDocument(String index, String id) throws IOException {
        DeleteRequest request = new DeleteRequest(index, id);
        return client.delete(request, RequestOptions.DEFAULT)
                .getResult() == DocWriteResponse.Result.DELETED;
    }

    /**
     * 更新文档（局部字段）
     *
     * @param index 索引
     * @param id    文档 ID
     * @param doc   待更新内容（会合并）
     * @return 更新响应
     */
    public UpdateResponse updateDocument(String index, String id, Object doc) throws IOException {
        UpdateRequest request = new UpdateRequest(index, id)
                .doc(JSON.toJSONString(doc), XContentType.JSON);
        return client.update(request, RequestOptions.DEFAULT);
    }

    /**
     * 搜索文档
     *
     * @param index  索引
     * @param source 查询构造器（DSL）
     * @return 搜索响应
     */
    public SearchResponse searchDocuments(String index, SearchSourceBuilder source) throws IOException {
        SearchRequest request = new SearchRequest(index).source(source);
        return client.search(request, RequestOptions.DEFAULT);
    }

    /**
     * 批量插入/更新文档
     *
     * @param index    索引
     * @param dataList Map&lt;id, doc&gt; 列表
     * @return 批量响应（含错误详情）
     */
    public <T> BulkResponse bulkCreateDocument(String index, List<Map<String, T>> dataList) throws IOException {
        BulkRequest request = new BulkRequest();
        for (Map<String, T> item : dataList) {
            String id = item.keySet().iterator().next();
            T doc = item.get(id);
            request.add(new IndexRequest(index)
                    .id(id)
                    .source(JSON.toJSONString(doc), XContentType.JSON));
        }
        return client.bulk(request, RequestOptions.DEFAULT);
    }

    /**
     * 批量删除文档
     *
     * @param index  索引
     * @param idList 待删除 ID 列表
     * @return 批量响应
     */
    public BulkResponse bulkDeleteDocument(String index, List<String> idList) throws IOException {
        BulkRequest request = new BulkRequest();
        for (String id : idList) {
            request.add(new DeleteRequest(index, id));
        }
        return client.bulk(request, RequestOptions.DEFAULT);
    }
}