package cn.gtmap.hlw.core.util.encryption.ght;

import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.asn1.pkcs.RSAPrivateKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * RSA加解密工具类
 *
 * @author <a href="mailto:sunchao@gtmap.cn">sunchao</a
 * @version 1.0
 * @description RSA加密解密工具，用于解密AES密钥
 */
public class RsaUtil {

    private static final Logger logger = LoggerFactory.getLogger(RsaUtil.class);

    private static final String ALGORITHM = "RSA";
    private static final String CIPHER_ALGORITHM = "RSA/ECB/PKCS1Padding";

    // 静态初始化，注册 Bouncy Castle Provider
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
            logger.info("Bouncy Castle Provider 已注册");
        }
    }

    /**
     * RSA公钥加密
     *
     * @param content 待加密内容
     * @param publicKeyStr 公钥字符串（Base64）
     * @return 加密后的Base64字符串
     */
    public static String encryptByPublicKey(String content, String publicKeyStr) {
        try {
            // 解码公钥
            byte[] keyBytes = Base64.decodeBase64(publicKeyStr);
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
            PublicKey publicKey = keyFactory.generatePublic(keySpec);

            // 加密
            Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
            byte[] encrypted = cipher.doFinal(content.getBytes(StandardCharsets.UTF_8));

            return Base64.encodeBase64String(encrypted);
        } catch (Exception e) {
            logger.error("RSA公钥加密失败", e);
            throw new RuntimeException("RSA公钥加密失败: " + e.getMessage());
        }
    }

    /**
     * RSA私钥解密（支持PKCS#8和PKCS#1格式）
     *
     * @param encryptedContent 加密的Base64字符串
     * @param privateKeyStr 私钥字符串（Base64）
     * @return 解密后的原文
     */
    public static String decryptByPrivateKey(String encryptedContent, String privateKeyStr) {
        PrivateKey privateKey = null;
        
        try {
            // 先尝试PKCS#8格式
            byte[] keyBytes = Base64.decodeBase64(privateKeyStr);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
            privateKey = keyFactory.generatePrivate(keySpec);
            logger.debug("使用PKCS#8格式解析私钥成功");
        } catch (Exception e1) {
            logger.warn("PKCS#8格式解析私钥失败，尝试PKCS#1格式: {}", e1.getMessage());
            try {
                // 尝试PKCS#1格式
                privateKey = loadPrivateKeyFromPKCS1(privateKeyStr);
                logger.info("使用PKCS#1格式解析私钥成功");
            } catch (Exception e2) {
                logger.error("PKCS#1格式解析私钥也失败", e2);
                throw new RuntimeException("RSA私钥解析失败（已尝试PKCS#8和PKCS#1格式）: " + e2.getMessage());
            }
        }

        try {
            // 解密
            Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, privateKey);
            byte[] decryptedBytes = cipher.doFinal(Base64.decodeBase64(encryptedContent));

            return new String(decryptedBytes, StandardCharsets.UTF_8);
        } catch (Exception e) {
            logger.error("RSA私钥解密失败", e);
            throw new RuntimeException("RSA私钥解密失败: " + e.getMessage());
        }
    }

    /**
     * 从PKCS#1格式加载私钥（使用 Bouncy Castle）
     * 
     * @param privateKeyStr Base64编码的PKCS#1格式私钥
     * @return PrivateKey对象
     * @throws Exception 解析失败时抛出异常
     */
    private static PrivateKey loadPrivateKeyFromPKCS1(String privateKeyStr) throws Exception {
        try {
            byte[] keyBytes = Base64.decodeBase64(privateKeyStr);
            
            // 使用 Bouncy Castle 解析 PKCS#1 格式的 DER 编码
            RSAPrivateKey rsaPrivateKey = RSAPrivateKey.getInstance(keyBytes);
            
            // 转换为 RSAPrivateCrtKeySpec
            RSAPrivateCrtKeySpec keySpec = new RSAPrivateCrtKeySpec(
                rsaPrivateKey.getModulus(),
                rsaPrivateKey.getPublicExponent(),
                rsaPrivateKey.getPrivateExponent(),
                rsaPrivateKey.getPrime1(),
                rsaPrivateKey.getPrime2(),
                rsaPrivateKey.getExponent1(),
                rsaPrivateKey.getExponent2(),
                rsaPrivateKey.getCoefficient()
            );
            
            KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
            return keyFactory.generatePrivate(keySpec);
        } catch (Exception e) {
            logger.error("PKCS#1格式私钥解析失败", e);
            throw new Exception("无法解析PKCS#1格式私钥: " + e.getMessage(), e);
        }
    }

    /**
     * 获取AES密钥（通过RSA解密加密的密钥）
     * 对应前端的 getRsa(rsaKey, rsaPrivateKey) 方法
     *
     * @param encryptedAesKey RSA加密后的AES密钥（Base64）
     * @param rsaPrivateKey RSA私钥（Base64）
     * @return 解密后的AES密钥
     */
    public static String getAesKey(String encryptedAesKey, String rsaPrivateKey) {
        return decryptByPrivateKey(encryptedAesKey, rsaPrivateKey);
    }
}

