package cn.gtmap.hlw.core.util.string;


import cn.gtmap.hlw.core.annotation.Desensitized;
import cn.gtmap.hlw.core.constant.SensitiveConstant;
import cn.gtmap.hlw.core.enums.SensitiveTypeEnum;
import cn.gtmap.hlw.core.util.encryption.aes.AesUtil;
import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.Environment;

import javax.annotation.Resource;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

/**
 * @version 1.0, 2019/4/29.
 * @auto <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
 * @description 数据脱敏
 */
public class DesensitizedUtils {
    private static org.slf4j.Logger logger = LoggerFactory.getLogger(DesensitizedUtils.class);

    private static final String JAVAX = "javax.";
    private static final String JAVA = "java.";
    //处理脱敏后字段的属性+Tm:dataTm，比如原先字段为user，那么处理的属性为userTm
    public static final String DATATM = "dataTm";
    //处理自身的属性:itself
    public static final String ITSELF = "itself";
    @Resource
    private static Environment environment;

    public static void main(String[] args) {

        logger.info("黄先生:" + chineseName("黄先生"));
        logger.info("410923199207092434:" + idCardNum("410923199207092434 "));
        logger.info("01086484484:" + fixedPhone("01086484484"));
        logger.info("010-86484484:" + fixedPhone("010-86484484"));
        logger.info("86484484:" + fixedPhone("86484484"));
        logger.info("18613881884:" + mobilePhone("18613881884"));
        logger.info("市区燕达花园G2幢402室" + address("市区燕达花园G2幢402室"));
        logger.info("1756401745@qq.com:" + email("1756401745@qq.com"));
        logger.info("6253624001899911:" + bankCard("6253624001899911"));
        logger.info("625362:" + companyBankCard("625362"));
        logger.info("陆春花 杭益松" + ListString("陆春花 杭益松", " ", SensitiveTypeEnum.CHINESE_NAME));
    }

    /**
     * @param
     * @param attributes 处理自身的属性:itself,dataTm
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 使用方法:需要在实体类上面extends dataTm.需要脱敏的实体类上新增注解，示例：@Desensitized(type = SensitiveTypeEnum.ID_CARD)
     * 调用该脱敏方法 GxyyYyxx gxyyYyxx = DesensitizedUtils.getBeanByJsonObj(requestData.getData(),GxyyYyxx.class);
     * 返回值会把脱敏后的数据放置到对应dataTm方法中的对应字段
     */
    public static <T> T getBeanByJsonObj(Object json, Class<T> tClass, String attributes) {
        T t = null;
        try {
            // 脱敏数据
            Object getJson = getJson(JSON.parseObject(JSON.toJSONString(json), tClass), attributes);
            t = JSON.parseObject(JSON.toJSONString(getJson), tClass);
        } catch (Exception e) {
            logger.error("getBeanByJsonObj ERROE", e);
        }
        return t;
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description
     */
    public static <T> List<T> getBeanListByJsonArray(Object obj, Class<T> tClass, String attributes) {
        List<T> t = null;
        try {
            t = JSON.parseArray(JSON.toJSONString(obj), tClass);
            if (t == null || t.size() <= 0) {
                return t;
            }
            //遍历List脱敏数据
            List list = new ArrayList<>();
            for (T json : t) {
                list.add(getBeanByJsonObj(json, tClass, attributes));
            }
            return list;
        } catch (Exception e) {
            logger.error(" ERROE", e);
        }
        return t;
    }

    public static String ListString(String strings, String Separator, SensitiveTypeEnum annotationType) {
        String[] strList = strings.split(Separator);
        List<String> list1 = new ArrayList<>();
        switch (annotationType) {
            case CHINESE_NAME:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.chineseName(str));
                }
                break;
            case ID_CARD:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.idCardNum(str));
                }
                break;
            case FIXED_PHONE:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.fixedPhone(str));
                }
                break;
            case MOBILE_PHONE:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.mobilePhone(str));
                }
                break;
            case ADDRESS:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.address(str));
                }
                break;
            case EMAIL:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.email(str));
                }
                break;
            case BANK_CARD:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.bankCard(str));
                }
                break;
            case COMPANY_BANK_CARD:
                for (String str : strList) {
                    list1.add(DesensitizedUtils.companyBankCard(str));
                }
                break;
            default:
                break;
        }
        return Joiner.on(Separator).join(list1);
    }

    /**
     * 获取脱敏json串(递归引用会导致java.lang.StackOverflowError)
     *
     * @param javaBean
     * @param attributes 处理自身的属性:itself,处理dataTm属性
     * @return
     */
    public static Object getJson(Object javaBean, String attributes) {
        if (null != javaBean) {
            try {
                if (javaBean.getClass().isInterface()) return null;
                /* 定义一个计数器，用于避免重复循环自定义对象类型的字段 */
                Set<Integer> referenceCounter = new HashSet<>();

                /* 对实体进行脱敏操作 */
                DesensitizedUtils.replace(DesensitizedUtils.getAllFields(javaBean), javaBean, referenceCounter, attributes);

                /* 清空计数器 */
                referenceCounter.clear();
            } catch (IllegalAccessException e) {
                logger.error("ERROE：getJson", e);
            }
        }
        return javaBean;
    }

    /**
     * 对需要脱敏的字段进行转化
     *
     * @param fields
     * @param javaBean
     * @param referenceCounter
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     */
    private static void replace(Field[] fields, Object javaBean, Set<Integer> referenceCounter, String attributes) throws IllegalAccessException {
        if (null != fields && fields.length > 0) {
            for (Field field : fields) {
                setAccessible(field);
                if (null != field && null != javaBean) {
                    Object value = field.get(javaBean);
                    if (null != value) {
                        Class<?> type = value.getClass();
                        //处理子属性，包括集合中的
                        if (type.isArray()) {//对数组类型的字段进行递归过滤
                            int len = Array.getLength(value);
                            for (int i = 0; i < len; i++) {
                                Object arrayObject = Array.get(value, i);
                                if (isNotGeneralType(arrayObject.getClass(), arrayObject, referenceCounter)) {
                                    replace(DesensitizedUtils.getAllFields(arrayObject), arrayObject, referenceCounter, attributes);
                                }
                            }
                        } else if (value instanceof Collection<?>) {//对集合类型的字段进行递归过滤
                            Collection<?> c = (Collection<?>) value;
                            Iterator<?> it = c.iterator();
                            while (it.hasNext()) {
                                //  待优化TODO:
                                Object collectionObj = it.next();
                                if (isNotGeneralType(collectionObj.getClass(), collectionObj, referenceCounter)) {
                                    replace(DesensitizedUtils.getAllFields(collectionObj), collectionObj, referenceCounter, attributes);
                                }
                            }
                        } else if (value instanceof Map<?, ?>) {//对Map类型的字段进行递归过滤
                            Map<?, ?> m = (Map<?, ?>) value;
                            Set<?> set = m.entrySet();
                            for (Object o : set) {
                                Map.Entry<?, ?> entry = (Map.Entry<?, ?>) o;
                                Object mapVal = entry.getValue();
                                if (isNotGeneralType(mapVal.getClass(), mapVal, referenceCounter)) {
                                    replace(DesensitizedUtils.getAllFields(mapVal), mapVal, referenceCounter, attributes);
                                }
                            }
                        } else if (value instanceof Enum<?>) {
                            continue;
                        }

                        /*除基础类型、jdk类型的字段之外，对其他类型的字段进行递归过滤*/
                        else {
                            if (!type.isPrimitive()
                                    && type.getPackage() != null
                                    && !StringUtils.startsWith(type.getPackage().getName(), JAVAX)
                                    && !StringUtils.startsWith(type.getPackage().getName(), JAVA)
                                    && !StringUtils.startsWith(field.getType().getName(), JAVAX)
                                    && !StringUtils.startsWith(field.getName(), JAVA)
                                    && referenceCounter.add(value.hashCode())) {
                                replace(DesensitizedUtils.getAllFields(value), value, referenceCounter, attributes);
                            }
                        }
                    }
                    //脱敏操作
                    setNewValueForField(javaBean, field, value, attributes);

                }
            }
        }
    }

    /**
     * 获取包括父类所有的属性
     *
     * @param objSource
     * @return
     */
    public static Field[] getAllFields(Object objSource) {
        /*获得当前类的所有属性(private、protected、public)*/
        List<Field> fieldList = new ArrayList<>();
        Class tempClass = objSource.getClass();
        while (tempClass != null && !tempClass.getName().equalsIgnoreCase("java.lang.object")) {//当父类为null的时候说明到达了最上层的父类(Object类).
            fieldList.addAll(Arrays.asList(tempClass.getDeclaredFields()));
            //这里得到父类的原因是objSource可能存在继承，获取objSource所有的属性
            tempClass = tempClass.getSuperclass(); //得到父类,然后赋给自己
        }
        Field[] fields = new Field[fieldList.size()];
        fieldList.toArray(fields);
        return fields;
    }

    /**
     * 排除基础类型、jdk类型、枚举类型的字段
     *
     * @param clazz
     * @param value
     * @param referenceCounter
     * @return
     */
    private static boolean isNotGeneralType(Class<?> clazz, Object value, Set<Integer> referenceCounter) {
        return !clazz.isPrimitive()
                && clazz.getPackage() != null
                && !clazz.isEnum()
                && !StringUtils.startsWith(clazz.getPackage().getName(), JAVAX)
                && !StringUtils.startsWith(clazz.getPackage().getName(), JAVA)
                && !StringUtils.startsWith(clazz.getName(), JAVAX)
                && !StringUtils.startsWith(clazz.getName(), JAVA)
                && referenceCounter.add(value.hashCode());
    }

    /**
     * 脱敏操作（按照规则转化需要脱敏的字段并设置新值）
     * 目前只支持String类型的字段，如需要其他类型如BigDecimal、Date等类型，可以添加
     *
     * @param javaBean
     * @param field
     * @param value
     * @param attributes 处理自身的属性:itself,dataTm
     * @throws IllegalAccessException
     */
    public static void setNewValueForField(Object javaBean, Field field, Object value, String attributes) throws IllegalAccessException {        //处理自身的属性
        Desensitized annotation = field.getAnnotation(Desensitized.class);
        if (field.getType().equals(String.class) && null != annotation && executeIsEffictiveMethod(javaBean, annotation)) {
            String valueStr = (String) value;
            if (StringUtils.isNotBlank(valueStr)) {
                Class clazz = javaBean.getClass();
                try {
                    if (DATATM.equals(attributes)) {
                        if (annotation.isEncrypter() && StringUtils.isNotBlank(valueStr)) {
                            //原来字段加密
                            field.set(javaBean, AesUtil.encrypt(valueStr, environment.getProperty(SensitiveConstant.AES_KEY)));
                        }
                        field = clazz.getDeclaredField(field.getName() + "Tm");
                        setAccessible(field);
                    }
                    switch (annotation.type()) {
                        case CHINESE_NAME:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.chineseName(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case ID_CARD:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.idCardNum(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case FIXED_PHONE:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.fixedPhone(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case MOBILE_PHONE:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.mobilePhone(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case ADDRESS:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.address(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case EMAIL:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.email(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case BANK_CARD:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.bankCard(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case COMPANY_BANK_CARD:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.companyBankCard(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        case PASSWORD:
                            if ("".equals(annotation.separator())) {
                                field.set(javaBean, DesensitizedUtils.password(valueStr));
                            } else {
                                field.set(javaBean, ListString(valueStr, annotation.separator(), annotation.type()));
                            }
                            break;
                        default:
                            break;
                    }
                } catch (NoSuchFieldException e) {
                    logger.error("ERROE：setNewValueForField", e);
                }
            }
        }
    }

    // 私有的变量，需要设置为可访问
    private static void setAccessible(Field field) {
        if (!field.isAccessible()) {
            field.setAccessible(true);
        }
    }

    /**
     * 执行某个对象中指定的方法
     *
     * @param javaBean     对象
     * @param desensitized
     * @return
     */
    private static boolean executeIsEffictiveMethod(Object javaBean, Desensitized desensitized) {
        boolean isAnnotationEffictive = true;//注解默认生效
        if (desensitized != null) {
            String isEffictiveMethod = desensitized.isEffictiveMethod();
            if (isNotEmpty(isEffictiveMethod)) {
                try {
                    Method method = javaBean.getClass().getMethod(isEffictiveMethod);
                    method.setAccessible(true);
                    isAnnotationEffictive = (Boolean) method.invoke(javaBean);
                } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
                    logger.error("ERROE：executeIsEffictiveMethod", e);
                }
            }
        }
        return isAnnotationEffictive;
    }

    private static boolean isNotEmpty(String str) {
        return str != null && !"".equals(str);
    }

    private static boolean isEmpty(String str) {
        return !isNotEmpty(str);
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 中文姓名脱敏：保留最后一位，前面以*替换，比如：**凯
     */
    public static String chineseName(String fullName) {
        if (StringUtils.isBlank(fullName)) {
            return "";
        }
        //截取第一位，后面以*代替
        String name = StringUtils.right(fullName, 1);
        return StringUtils.leftPad(name, StringUtils.length(fullName), "*");
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 公民身份证号码脱敏：比如：***********4432
     */
    public static String idCardNum(String id) {
        id = StringUtils.trim(id);
        int length = StringUtils.length(id);
        if (StringUtils.isBlank(id)) {
            return "";
        }
        return StringUtils.leftPad(StringUtils.right(id, 4), StringUtils.length(id), "*");
    }

    /**
     * @param id String 证件号
     * @return 公民身份证号码脱敏（隐藏出生年月）
     */
    public static String idCardNumByBirthday(String id){
        id = StringUtils.trim(id);
        int length = StringUtils.length(id);
        if (StringUtils.isBlank(id)) {
            return "";
        }
        return StringUtils.left(id, 6).concat(StringUtils.removeStart(StringUtils.leftPad(StringUtils.right(id, 4), StringUtils.length(id) - 9, "*"), "*"));
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 固定电话脱敏：保留后四位，比如：*******4484
     */
    public static String fixedPhone(String num) {
        if (StringUtils.isBlank(num)) {
            return "";
        }
        return StringUtils.leftPad(StringUtils.right(num, 4), StringUtils.length(num), "*");
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 手机号码脱敏：保留前3位，后面保留4位，比如：183****1084
     */
    public static String mobilePhone(String num) {
        num = StringUtils.trim(num);
        if (StringUtils.isBlank(num)) {
            return "";
        } else if (11 != num.trim().length()) {
            return num;
        }
        return StringUtils.left(num, 3).concat(StringUtils.removeStart(StringUtils.leftPad(StringUtils.right(num, 4), StringUtils.length(num) - 2, "*"), "*"));
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 地址脱敏：脱敏后四位，比如：江苏省南京市*****
     */
    public static String address(String address) {
        if (StringUtils.isBlank(address)) {
            return "";
        }
        int length = StringUtils.length(address);
        //保留前6位 StringUtils.rightPad(StringUtils.left(address, 6), length, "*")
        return StringUtils.leftPad(StringUtils.right(address, 6), length, "*");
        //脱敏后四位
        //return StringUtils.rightPad(StringUtils.left(address, length - 4), length, "*")
    }

    /**
     * 【电子邮箱 前缀隐藏，用星号代替，@及后面的地址显示，比如：***@126.com>
     *
     * @param email
     * @return
     */
    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 电子邮箱脱敏：保留@后面的部分，比如：**********@qq.com
     */
    public static String email(String email) {
        if (StringUtils.isBlank(email)) {
            return "";
        }
        int index = StringUtils.indexOf(email, "@");
        if (index <= 1) {
            return email;
        } else {
            return StringUtils.rightPad(StringUtils.left(email, 0), index, "*").concat(StringUtils.mid(email, index, StringUtils.length(email)));
        }
    }

    /**
     * 【银行卡号】前六位，后四位，其他用星号隐藏每位1个星号，比如：6222600**********1234>
     *
     * @param cardNum
     * @return
     */
    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 银行卡号脱敏：保留钱6位后4位，比如：625362******9911
     */
    public static String bankCard(String cardNum) {
        if (StringUtils.isBlank(cardNum)) {
            return "";
        }
        return StringUtils.left(cardNum, 6).concat(StringUtils.removeStart(StringUtils.leftPad(StringUtils.right(cardNum, 4), StringUtils.length(cardNum), "*"), "******"));
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 公司开户银行联号脱敏：保留前2位，比如：62******
     */
    public static String companyBankCard(String companyBardNum) {
        if (StringUtils.isBlank(companyBardNum)) {
            return "";
        }
        int length = StringUtils.length(companyBardNum);
        return StringUtils.rightPad(StringUtils.left(companyBardNum, 2), length, "*");
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @description 密码脱敏：全部*
     */
    public static String password(String password) {
        if (StringUtils.isBlank(password)) {
            return "";
        }
        String pwd = StringUtils.left(password, 0);
        return StringUtils.rightPad(pwd, StringUtils.length(password), "*");
    }
}
