package cn.gtmap.hlw.core.util.encryption;

import cn.gtmap.hlw.core.dto.util.EncryptDTO;
import cn.gtmap.hlw.core.enums.error.ErrorEnum;
import cn.gtmap.hlw.core.exception.BizException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import lombok.extern.slf4j.Slf4j;
import org.bouncycastle.jce.interfaces.ECPublicKey;
import org.bouncycastle.math.ec.ECPoint;

import org.bouncycastle.crypto.engines.SM4Engine;
import org.bouncycastle.crypto.modes.CBCBlockCipher;
import org.bouncycastle.crypto.paddings.PaddedBufferedBlockCipher;
import org.bouncycastle.crypto.params.*;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPrivateKeySpec;
import org.bouncycastle.jce.spec.ECPublicKeySpec;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.*;
import java.math.BigInteger;
import java.security.*;
import java.security.interfaces.ECPrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import java.util.Base64;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2025/9/15
 * @description
 */
@Slf4j
public class DigitalEnvelopeUtil {
    private static final Logger logger = LoggerFactory.getLogger(DigitalEnvelopeUtil.class);
    public static final BouncyCastleProvider BOUNCY_CASTLE_PROVIDER = new BouncyCastleProvider();

    //curves
    public static final String SM2_CURVES_NAME = "sm2p256v1";

    //algorithm 算法
    public static final String RSA_ALGORITHM = "RSA";

    public static final String SM2_ALGORITHM = "SM2";

    public static final String EC_ALGORITHM = "EC";

    public static final String SM4_ALGORITHM = "SM4";

    public static final String BC_ALGORITHM = "BC";

    public static final String SIGN_ALGORITHM = "SM3withSM2";

    public static final String UTF8_CHARSET = "UTF-8";

    private static String publicKey = "BGrWk/fz/kYymZo7ib24jTkQqf9Xn27TL6krnJTEVTBztikVBsPiMNQnwii4oQUUy81lHw8UAJLjctqPOlcgWKg=";
    private static String privateKey = "AN3OfV2Ujba3+EnhRCf2+Tr9b/3KgbpB+GUVLtTJz2Hg";

    private static String myPublicKey = "BOWt12afqs8adQxRoacrJfHt/my4HCxj5b3nTGS6t1sKwULaXpNM5oXDBe3CIImVJ1YqabRaO8NkYlICnONLmFo=";
    private static String myPrivateKey = "AKWFcc+nljrpGTcNnzJEQ5AvvOvT7u+wHf2aNbV+8+8g";

    static {
        Security.addProvider(new BouncyCastleProvider());
    }


    /**
     * 生成SM2密钥对
     *
     * @return SM2密钥对
     */
    private static KeyPair generateSm2KeyPair() throws Exception {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(EC_ALGORITHM, BC_ALGORITHM);
            ECNamedCurveParameterSpec ecSpec = ECNamedCurveTable.getParameterSpec(SM2_CURVES_NAME);
            keyPairGenerator.initialize(ecSpec);
            return keyPairGenerator.generateKeyPair();
        } catch (Exception e) {
            logger.error("生成SM2密钥对失败", e);
            throw new Exception("生成SM2密钥对失败", e);
        }
    }

    /**
     * 使用SM3签名
     *
     * @param data
     * @param privateKey
     * @return 签名
     */
    private static byte[] signWithSm3(String data, PrivateKey privateKey) throws Exception {
        try {
            Signature signature = Signature.getInstance(SIGN_ALGORITHM, BC_ALGORITHM);
            signature.initSign(privateKey);
            signature.update(data.getBytes());
            return signature.sign();
        } catch (Exception e) {
            logger.error("使用SM3签名失败", e);
            throw new Exception("使用SM3签名失败", e);
        }
    }

    /**
     * 使用SM3验证签名
     *
     * @param input     原始数据
     * @param signature 待验证的数字签名
     * @param publicKey 用于验证签名的公钥
     * @return 验证成功返回true
     */
    private static boolean verifyWithSm3(String input, byte[] signature, PublicKey publicKey) throws Exception {
        try {
            Signature signature2 = Signature.getInstance(SIGN_ALGORITHM, BC_ALGORITHM);
            signature2.initVerify(publicKey);
            signature2.update(input.getBytes());
            return signature2.verify(signature);
        } catch (Exception e) {
            logger.error("使用SM3验证签名失败", e);
            throw new Exception("使用SM3验证签名失败", e);
        }
    }

    /**
     * SM4加密
     *
     * @param plainText 加密数据
     * @param key       128位SM4密钥
     * @param iv        128位初始化向量
     * @return 密文
     */
    private static String encryptWithSm4(String plainText, byte[] key, byte[] iv) throws Exception {
        try {
            SM4Engine engine = new SM4Engine();
            PaddedBufferedBlockCipher cipher = new PaddedBufferedBlockCipher(new CBCBlockCipher(engine));
            cipher.init(true, new ParametersWithIV(new KeyParameter(key), iv));

            byte[] input = plainText.getBytes(UTF8_CHARSET);
            byte[] output = new byte[cipher.getOutputSize(input.length)];
            int len = cipher.processBytes(input, 0, input.length, output, 0);
            cipher.doFinal(output, len);

            return Hex.toHexString(output);
        } catch (Exception e) {
            logger.error("SM4加密失败", e);
            throw new Exception("SM4加密失败", e);
        }

    }

    /**
     * SM4解密
     *
     * @param cipherText 十六进制编码的密文
     * @param key        128位SM4密钥
     * @param iv         128位初始化向量
     * @return 明文
     */
    private static String decryptWithSm4(String cipherText, byte[] key, byte[] iv) throws Exception {
        try {
            SM4Engine engine = new SM4Engine();
            PaddedBufferedBlockCipher cipher = new PaddedBufferedBlockCipher(new CBCBlockCipher(engine));
            cipher.init(false, new ParametersWithIV(new KeyParameter(key), iv));

            byte[] input = Hex.decode(cipherText);
            byte[] output = new byte[cipher.getOutputSize(input.length)];
            int len = cipher.processBytes(input, 0, input.length, output, 0);
            cipher.doFinal(output, len);

            // 解密时如果长度不是16得倍数会补0导致验签失败，此处去掉多余0
            if (output != null && output.length != 0) {
                for (int i = output.length - 1; i >= 0; i--) {
                    if (output[i] != 0) {
                        output = Arrays.copyOfRange(output, 0, i + 1);
                        break;
                    }
                }
            }

            return new String(output, UTF8_CHARSET).trim();
        } catch (Exception e) {
            logger.error("SM4解密失败", e);
            throw new Exception("SM4解密失败", e);
        }
    }

    /**
     * 生成SM4密钥
     *
     * @return SM4密钥
     * @throws Exception
     */
    private static byte[] generateKey() throws Exception {
        try {
            KeyGenerator keyGenerator = KeyGenerator.getInstance(SM4_ALGORITHM, BC_ALGORITHM);
            // SM4密钥长度固定为128位
            keyGenerator.init(128, new SecureRandom());
            SecretKey secretKey = keyGenerator.generateKey();
            return secretKey.getEncoded();
        } catch (Exception e) {
            logger.error("生成SM4密钥失败", e);
            throw new Exception("生成SM4密钥失败", e);
        }

    }

    /**
     * 生成IV
     *
     * @return IV
     */
    private static byte[] generateIV() throws Exception {
        try {
            // SM4的块大小是128位（16字节）
            byte[] iv = new byte[16];
            new SecureRandom().nextBytes(iv);
            return iv;
        } catch (Exception e) {
            logger.error("生成IV失败", e);
            throw new Exception("生成IV失败", e);
        }
    }

    public static PublicKey generatePublicKeyByKeyValue(String algorithm, KeySpec keySpec) {
        try {
            return KeyFactory
                    .getInstance(algorithm, BOUNCY_CASTLE_PROVIDER)
                    .generatePublic(keySpec);
        } catch (InvalidKeySpecException e) {
            throw new IllegalArgumentException("无效的密钥规范: " + e.getMessage(), e);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("不支持的算法: " + algorithm, e);
        }
    }

    public static PrivateKey generatePrivateKeyByKeyValue(String algorithm, KeySpec keySpec) {
        try {
            return KeyFactory
                    .getInstance(algorithm, BOUNCY_CASTLE_PROVIDER)
                    .generatePrivate(keySpec);
        } catch (InvalidKeySpecException e) {
            throw new IllegalArgumentException("无效的密钥规范: " + e.getMessage(), e);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("不支持的算法: " + algorithm, e);
        }
    }

    /**
     * 功能：获取 Cipher 实例
     *
     * @param transformations 转换信息
     * @return {@link Cipher}
     */
    private static Cipher getCipherInstance(String transformations) {
        try {

            return Cipher.getInstance(transformations, BOUNCY_CASTLE_PROVIDER);

        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("不支持的算法 ", e);
        } catch (NoSuchPaddingException e) {
            throw new IllegalArgumentException("填充错误", e);
        }
    }

    /**
     * 功能：非对称加密
     *
     * @param publicKeyValue 公钥值
     * @param keyAlgorithm   密钥算法
     * @param message        待加密数据
     * @return 密文
     */
    public static byte[] asymmetricKeyEncrypt(byte[] publicKeyValue, String keyAlgorithm, byte[] message) {
        PublicKey publicKey;
        String algorithm;
        if (EC_ALGORITHM.equalsIgnoreCase(keyAlgorithm)) {
            algorithm = SM2_ALGORITHM;
        } else {
            algorithm = keyAlgorithm;
        }
        if ((SM2_ALGORITHM.equalsIgnoreCase(algorithm))) {
            final ECParameterSpec spec = ECNamedCurveTable.getParameterSpec(SM2_CURVES_NAME);
            final ECPublicKeySpec ecPublicKeySpec = new ECPublicKeySpec(spec.getCurve().decodePoint(publicKeyValue), spec);
            publicKey = generatePublicKeyByKeyValue(EC_ALGORITHM, ecPublicKeySpec);
        } else if (RSA_ALGORITHM.equalsIgnoreCase(algorithm)) {
            X509EncodedKeySpec x509EncodedKeySpec = new X509EncodedKeySpec(publicKeyValue);
            publicKey = generatePublicKeyByKeyValue(RSA_ALGORITHM, x509EncodedKeySpec);
        } else {
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), String.format("not support algorithm  %s", keyAlgorithm));
        }
        Cipher cipher = getCipherInstance(algorithm);
        return encryptForAsymmetric(cipher, publicKey, message);
    }

    /**
     * 功能：非对称解密
     *
     * @param privateKeyValue 公钥值
     * @param keyAlgorithm    密钥算法
     * @param cipherText      密文
     * @return 明文
     */
    public static byte[] asymmetricKeyDecrypt(byte[] privateKeyValue, String keyAlgorithm, byte[] cipherText) {
        PrivateKey privateKey;
        String algorithm;
        if (EC_ALGORITHM.equalsIgnoreCase(keyAlgorithm)) {
            algorithm = SM2_ALGORITHM;
        } else {
            algorithm = keyAlgorithm;
        }
        if ((SM2_ALGORITHM.equalsIgnoreCase(algorithm))) {
            final ECParameterSpec spec = ECNamedCurveTable.getParameterSpec(SM2_CURVES_NAME);
            final ECPrivateKeySpec ecPrivateKeySpec = new ECPrivateKeySpec(new BigInteger(1, privateKeyValue), spec);
            privateKey = generatePrivateKeyByKeyValue(EC_ALGORITHM, ecPrivateKeySpec);
        } else if (RSA_ALGORITHM.equalsIgnoreCase(algorithm)) {
            PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(privateKeyValue);
            privateKey = generatePrivateKeyByKeyValue(RSA_ALGORITHM, pkcs8EncodedKeySpec);
        } else {
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), String.format("not support algorithm  %s", keyAlgorithm));
        }
        Cipher cipher = getCipherInstance(algorithm);
        return decryptForAsymmetric(cipher, privateKey, cipherText);

    }

    private static byte[] encryptForAsymmetric(Cipher cipher, Key key, byte[] message) {
        try {
            cipher.init(Cipher.ENCRYPT_MODE, key);
            return cipher.doFinal(message);
        } catch (InvalidKeyException e) {
            throw new IllegalArgumentException("秘钥异常！ ", e);
        } catch (BadPaddingException e) {
            throw new IllegalArgumentException("填充数据错误！ ", e);
        } catch (IllegalBlockSizeException e) {
            throw new IllegalArgumentException("非法区块大小异常！ ", e);
        }
    }

    private static byte[] decryptForAsymmetric(Cipher cipher, Key key, byte[] cipherText) {
        try {
            cipher.init(Cipher.DECRYPT_MODE, key);
            return cipher.doFinal(cipherText);
        } catch (InvalidKeyException e) {
            throw new IllegalArgumentException("秘钥异常！ ", e);
        } catch (BadPaddingException e) {
            throw new IllegalArgumentException("填充数据错误！ ", e);
        } catch (IllegalBlockSizeException e) {
            throw new IllegalArgumentException("非法区块大小异常！ ", e);
        }
    }

    public static EncryptDTO encryptData(String jsonString, String platformPublicKey, String platformPrivateKey, String thirdPartyPublicKey) throws Exception {
        // 1、构建平台方密钥对
        KeyPair keyPair = buildKeyPairFromBase64(platformPrivateKey, platformPublicKey);
        PrivateKey signPrivateKey = keyPair.getPrivate();
        PublicKey signPublicKey = keyPair.getPublic();

        //2、使用平台方的密钥对进行数字签名
        byte[] bytes = signWithSm3(jsonString, signPrivateKey);

        // 3. 生成SM4密钥和IV向量
        byte[] sm4Key = generateKey();
        byte[] iv = generateIV();

        // 4. SM4加密数据
        String encryptedData = encryptWithSm4(jsonString, sm4Key, iv);

        // 5. SM2加密SM4密钥
        byte[] encryptionSm4Key = asymmetricKeyEncrypt(Base64.getDecoder().decode(thirdPartyPublicKey), "SM2", sm4Key);

        EncryptDTO encryptDTO = new EncryptDTO();
        encryptDTO.setEncryptData(encryptionSm4Key);
        encryptDTO.setSm4CipherText(encryptedData);
        encryptDTO.setIv(iv);
        encryptDTO.setSignaData(bytes);
        encryptDTO.setPlatformPublicKey(Base64.getEncoder().encodeToString(signPublicKey.getEncoded()));
        return encryptDTO;

    }


    public static String decDate(EncryptDTO encryptDTO, String thirdPartyPrivateKey) throws Exception {
        log.info("thirdPartyPrivateKey:" + thirdPartyPrivateKey);
        byte[] encryptData = encryptDTO.getEncryptData();
        byte[] iv = encryptDTO.getIv();
        String sm4CipherText = encryptDTO.getSm4CipherText();
        byte[] signaData = encryptDTO.getSignaData();

        PublicKey publicKey = null;

        try {
            // 尝试直接解析platformPublicKey为X.509格式公钥
            byte[] publicKeyBytes = Base64.getDecoder().decode(encryptDTO.getPlatformPublicKey());
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKeyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("EC", "BC");
            publicKey = keyFactory.generatePublic(keySpec);
        } catch (Exception e) {
            // 如果X.509格式解析失败，尝试其他方式
            logger.warn("X.509公钥解析失败，尝试其他方式: " + e.getMessage());

            try {
                // 尝试作为双重Base64编码处理
                byte[] firstDecoded = Base64.getDecoder().decode(encryptDTO.getPlatformPublicKey());
                String realPublicKeyStr = new String(firstDecoded, "UTF-8");
                byte[] publicKeyBytes = Base64.getDecoder().decode(realPublicKeyStr);

                X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKeyBytes);
                KeyFactory keyFactory = KeyFactory.getInstance("EC", "BC");
                publicKey = keyFactory.generatePublic(keySpec);
            } catch (Exception ex) {
                // 如果还是失败，尝试作为原始EC点坐标处理
                logger.warn("双重Base64解析也失败，尝试作为EC点坐标处理: " + ex.getMessage());
                try {
                    byte[] publicKeyBytes = Base64.getDecoder().decode(encryptDTO.getPlatformPublicKey());
                    ECParameterSpec spec = ECNamedCurveTable.getParameterSpec(SM2_CURVES_NAME);
                    ECPublicKeySpec ecPublicKeySpec = new ECPublicKeySpec(spec.getCurve().decodePoint(publicKeyBytes), spec);
                    KeyFactory keyFactory = KeyFactory.getInstance("EC", "BC");
                    publicKey = keyFactory.generatePublic(ecPublicKeySpec);
                } catch (Exception ex2) {
                    throw new RuntimeException("无法解析公钥格式", ex2);
                }
            }
        }

        // 1. SM2解密获取SM4密钥
        byte[] deserializeSm4Key = asymmetricKeyDecrypt(Base64.getDecoder().decode(thirdPartyPrivateKey), "SM2", encryptData);
        // 2. SM4解密 ->明文
        String decryptedText = decryptWithSm4(sm4CipherText, deserializeSm4Key, iv);

        // 3. 验证签名
        boolean isCorrect = verifyWithSm3(decryptedText, signaData, publicKey);

        System.out.println("验证签名：" + isCorrect);
        if (isCorrect) {
            return decryptedText;
        } else {
            return null;
        }
    }




    public static void main1(String[] args) throws Exception {
        String s = "BOWt12afqs8adQxRoacrJfHt/my4HCxj5b3nTGS6t1sKwULaXpNM5oXDBe3CIImVJ1YqabRaO8NkYlICnONLmFo=";

        PublicKey publicKey = null;

        try {
            // 尝试直接解析platformPublicKey为X.509格式公钥
            byte[] publicKeyBytes = Base64.getDecoder().decode(s);
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKeyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("EC", "BC");
            publicKey = keyFactory.generatePublic(keySpec);
        } catch (Exception e) {
            // 如果X.509格式解析失败，尝试其他方式
            logger.warn("X.509公钥解析失败，尝试其他方式: " + e.getMessage());

            try {
                // 尝试作为双重Base64编码处理
                byte[] firstDecoded = Base64.getDecoder().decode(s);
                String realPublicKeyStr = new String(firstDecoded, "UTF-8");
                byte[] publicKeyBytes = Base64.getDecoder().decode(realPublicKeyStr);

                X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKeyBytes);
                KeyFactory keyFactory = KeyFactory.getInstance("EC", "BC");
                publicKey = keyFactory.generatePublic(keySpec);
            } catch (Exception ex) {
                // 如果还是失败，尝试作为原始EC点坐标处理
                logger.warn("双重Base64解析也失败，尝试作为EC点坐标处理: " + ex.getMessage());
                try {
                    byte[] publicKeyBytes = Base64.getDecoder().decode(s);
                    ECParameterSpec spec = ECNamedCurveTable.getParameterSpec(SM2_CURVES_NAME);
                    ECPublicKeySpec ecPublicKeySpec = new ECPublicKeySpec(spec.getCurve().decodePoint(publicKeyBytes), spec);
                    KeyFactory keyFactory = KeyFactory.getInstance("EC", "BC");
                    publicKey = keyFactory.generatePublic(ecPublicKeySpec);
                } catch (Exception ex2) {
                    throw new RuntimeException("无法解析公钥格式", ex2);
                }
            }
        }

    }

    private static KeyPair buildKeyPairFromBase64(String base64PrivateKey, String base64PublicKey) throws Exception {
        try {
            // 解码私钥
            byte[] privateKeyBytes = Base64.getDecoder().decode(base64PrivateKey);
            byte[] publicKeyBytes = Base64.getDecoder().decode(base64PublicKey);

            // 获取SM2参数
            ECParameterSpec spec = ECNamedCurveTable.getParameterSpec(SM2_CURVES_NAME);

            // 构建私钥
            ECPrivateKeySpec ecPrivateKeySpec = new ECPrivateKeySpec(new BigInteger(1, privateKeyBytes), spec);
            PrivateKey privateKey = KeyFactory.getInstance("EC", "BC").generatePrivate(ecPrivateKeySpec);

            // 构建公钥
            ECPublicKeySpec ecPublicKeySpec = new ECPublicKeySpec(spec.getCurve().decodePoint(publicKeyBytes), spec);
            PublicKey publicKey = KeyFactory.getInstance("EC", "BC").generatePublic(ecPublicKeySpec);

            return new KeyPair(publicKey, privateKey);
        } catch (Exception e) {
            logger.error("构建密钥对失败: base64PrivateKey={}, base64PublicKey={}",
                    base64PrivateKey, base64PublicKey, e);
            throw new Exception("构建密钥对失败", e);
        }
    }


    /**
     * 获取我的公钥和私钥
     *
     * @throws Exception
     */
    private static void getMyKeys() throws Exception {
        // 生成SM2密钥对
        KeyPair keyPair = generateSm2KeyPair();
        // 获取公钥点坐标（用于加密）
        ECPublicKey ecPublicKey = (ECPublicKey) keyPair.getPublic();
        ECPoint publicKeyPoint = ecPublicKey.getQ();

        // 将公钥点编码为字节数组（适用于decodePoint方法）
        byte[] publicKeyBytes = publicKeyPoint.getEncoded(false); // uncompressed format

        // 获取私钥D值（用于解密）
        ECPrivateKey ecPrivateKey = (ECPrivateKey) keyPair.getPrivate();
        BigInteger privateKeyD = ecPrivateKey.getS();
        byte[] privateKeyBytes = privateKeyD.toByteArray();

        // Base64编码以便存储和传输
        String base64PublicKey = Base64.getEncoder().encodeToString(publicKeyBytes);
        String base64PrivateKey = Base64.getEncoder().encodeToString(privateKeyBytes);

        System.out.println("原始格式公钥(用于加密): " + base64PublicKey);
        System.out.println("原始格式私钥(用于解密): " + base64PrivateKey);

        // 验证生成的密钥可以被当前代码正确解析
        try {
            ECParameterSpec spec = ECNamedCurveTable.getParameterSpec("sm2p256v1");
            ECPoint decodedPoint = spec.getCurve().decodePoint(publicKeyBytes);
            System.out.println("公钥点解码成功: " + decodedPoint);
        } catch (Exception e) {
            System.err.println("公钥点解码失败: " + e.getMessage());
        }
    }


    //对接实例
    public static void main(String[] args) throws Exception {
        //数据加密
//        String json = "{\"appid\":\"f90528f3635635d6518e40253a84cf1f\",\"cbdkxxData\":[{\"cbfbm\":\"340828112200190012\",\"cbhtbm\":\"340828112200190012J\",\"cbjyqqdfs\":\"110\",\"cbjyqzbm\":\"340828112200190012J\",\"dkbm\":\"3408281122001900074\",\"dkmc\":\"小垅口\",\"fbfbm\":\"34082811220019\",\"htmj\":20.00,\"htmjm\":0.03,\"lzhtbm\":\"\",\"partitionCode\":\"340828\",\"regionCode\":\"34082811220019\",\"sfqqqg\":\"2\",\"yhtmj\":453.33,\"yhtmjm\":0.68},{\"cbfbm\":\"340828112200190012\",\"cbhtbm\":\"340828112200190012J\",\"cbjyqqdfs\":\"110\",\"cbjyqzbm\":\"340828112200190012J\",\"dkbm\":\"3408281122001900135\",\"dkmc\":\"大茶园\",\"fbfbm\":\"34082811220019\",\"htmj\":472.15,\"htmjm\":0.71,\"lzhtbm\":\"\",\"partitionCode\":\"340828\",\"regionCode\":\"34082811220019\",\"sfqqqg\":\"2\",\"yhtmj\":472.15,\"yhtmjm\":0.71},{\"cbfbm\":\"340828112200190012\",\"cbhtbm\":\"340828112200190012J\",\"cbjyqqdfs\":\"110\",\"cbjyqzbm\":\"340828112200190012J\",\"dkbm\":\"3408281122001900078\",\"dkmc\":\"河垅小茶园\",\"fbfbm\":\"34082811220019\",\"htmj\":240.00,\"htmjm\":0.36,\"lzhtbm\":\"\",\"partitionCode\":\"340828\",\"regionCode\":\"34082811220019\",\"sfqqqg\":\"2\",\"yhtmj\":580.00,\"yhtmjm\":0.87},{\"cbfbm\":\"340828112200190012\",\"cbhtbm\":\"340828112200190012J\",\"cbjyqqdfs\":\"110\",\"cbjyqzbm\":\"340828112200190012J\",\"dkbm\":\"3408281122001900153\",\"dkmc\":\"小茶园\",\"fbfbm\":\"34082811220019\",\"htmj\":200.00,\"htmjm\":0.30,\"lzhtbm\":\"\",\"partitionCode\":\"340828\",\"regionCode\":\"34082811220019\",\"sfqqqg\":\"2\",\"yhtmj\":313.33,\"yhtmjm\":0.47},{\"cbfbm\":\"340828112200190012\",\"cbhtbm\":\"340828112200190012J\",\"cbjyqqdfs\":\"110\",\"cbjyqzbm\":\"340828112200190012J\",\"dkbm\":\"3408281122001900164\",\"dkmc\":\"小茶园\",\"fbfbm\":\"34082811220019\",\"htmj\":111.78,\"htmjm\":0.17,\"lzhtbm\":\"\",\"partitionCode\":\"340828\",\"regionCode\":\"34082811220019\",\"sfqqqg\":\"2\",\"yhtmj\":111.78,\"yhtmjm\":0.17}],\"cbdkzs\":5,\"cbfData\":[{\"cbfbm\":\"340828112200190012\",\"cbfdz\":\"岳西县黄尾镇云峰村光明组\",\"cbfmc\":\"郑光友\",\"cbfzjhm\":\"340828197207066913\",\"cbfzjlx\":\"1\"}],\"cbfbm\":\"340828112200190012\",\"cbfs\":\"110\",\"cbhtbm\":\"340828112200190012J\",\"cbqxq\":\"1995-04-01 00:00:00\",\"cbqxz\":\"2025-03-31 23:59:59\",\"cjqydm\":\"340828112200\",\"cjqymc\":\"云峰村\",\"dkxxData\":[{\"dkbm\":\"3408281122001900074\",\"dkbz\":\"旱地\",\"dkdz\":\"郑光发,郑光卫\",\"dkjson\":\"POLYGON((116.26207814493837 31.07883168620356,116.26215022684194 31.078905856415062,116.26230215463039 31.078801298065446,116.26226167920336 31.07875603928149,116.26223461719482 31.078706531642265,116.26223478758304 31.078626806000877,116.26219824711654 31.07862604586478,116.26216281549073 31.078559601054106,116.26215350045082 31.078644433326556,116.26213046216454 31.07868734631422,116.262094838363 31.078713338596685,116.2620142809805 31.07873306966831,116.26207814493837 31.07883168620356))\",\"dklb\":\"10\",\"dkmc\":\"小垅口\",\"dknz\":\"郑光卫\",\"dkxz\":\"郑光发\",\"dldj\":\"04\",\"htmjm\":\"0.03\",\"scmjm\":\"0.68\",\"sfjbnt\":\"1\",\"sfqqqg\":\"2\",\"tdyt\":\"1\",\"ydkbm\":\"\",\"zjrxm\":\"郑光友\"},{\"dkbm\":\"3408281122001900135\",\"dkbz\":\"胡道明,胡道直\",\"dkdz\":\"山边\",\"dkjson\":\"POLYGON((116.26041702689616 31.077439353974363,116.26044734256936 31.07747001618912,116.26058015039894 31.077519910528938,116.2606417665469 31.077503569370666,116.26073924891307 31.077521839170092,116.26076945728217 31.077474843972862,116.26066905961196 31.077401350336842,116.26042757105213 31.077295423652266,116.26038352769156 31.077334534242777,116.2604454045962 31.077382632921513,116.26041702689616 31.077439353974363))\",\"dklb\":\"10\",\"dkmc\":\"大茶园\",\"dknz\":\"郑光发\",\"dkxz\":\"郑安,胡清卫\",\"dldj\":\"04\",\"htmjm\":\"0.71\",\"scmjm\":\"0.71\",\"sfjbnt\":\"1\",\"sfqqqg\":\"2\",\"tdyt\":\"1\",\"ydkbm\":\"\",\"zjrxm\":\"郑光友\"},{\"dkbm\":\"3408281122001900078\",\"dkbz\":\"刘春节\",\"dkdz\":\"小路\",\"dkjson\":\"POLYGON((116.26145960010274 31.07870512469948,116.26147572934691 31.07877139691733,116.26150114403546 31.078809880185542,116.26153486544418 31.078823689521027,116.26161043633681 31.078824122277787,116.26166646303176 31.07869872781261,116.26177489989638 31.07858242434232,116.26164143162707 31.07854001964938,116.26157360810433 31.078535103663185,116.2614980468079 31.07859580997988,116.26145960010274 31.07870512469948))\",\"dklb\":\"10\",\"dkmc\":\"河垅小茶园\",\"dknz\":\"刘春节,胡纵发\",\"dkxz\":\"旱地\",\"dldj\":\"04\",\"htmjm\":\"0.36\",\"scmjm\":\"0.87\",\"sfjbnt\":\"1\",\"sfqqqg\":\"2\",\"tdyt\":\"1\",\"ydkbm\":\"\",\"zjrxm\":\"郑光友\"},{\"dkbm\":\"3408281122001900153\",\"dkbz\":\"山边\",\"dkdz\":\"郑光成\",\"dkjson\":\"POLYGON((116.25929412902816 31.076757914564816,116.25936459627164 31.076772795950237,116.259369540726 31.076725007499533,116.2593259208936 31.076555934637543,116.25932308528773 31.07645809119207,116.25926332509285 31.07647419887104,116.25922337411848 31.076506212856135,116.25918997975971 31.076514428674155,116.25922491451003 31.076623643875784,116.25925579021705 31.07664524353152,116.25928754968476 31.076721392491816,116.25929412902816 31.076757914564816))\",\"dklb\":\"10\",\"dkmc\":\"小茶园\",\"dknz\":\"胡道明\",\"dkxz\":\"山边\",\"dldj\":\"04\",\"htmjm\":\"0.30\",\"scmjm\":\"0.47\",\"sfjbnt\":\"1\",\"sfqqqg\":\"2\",\"tdyt\":\"1\",\"ydkbm\":\"\",\"zjrxm\":\"郑光友\"},{\"dkbm\":\"3408281122001900164\",\"dkbz\":\"菜地\",\"dkdz\":\"刘春节\",\"dkjson\":\"POLYGON((116.26001236607968 31.076420811892483,116.26005140167092 31.07644362796793,116.26008115038091 31.07644204353647,116.26013935683685 31.076331170415752,116.26013510931843 31.076288885892023,116.26011208107802 31.076266600016883,116.26001236607968 31.076420811892483))\",\"dklb\":\"10\",\"dkmc\":\"小茶园\",\"dknz\":\"路\",\"dkxz\":\"胡清卫\",\"dldj\":\"04\",\"htmjm\":\"0.17\",\"scmjm\":\"0.17\",\"sfjbnt\":\"1\",\"sfqqqg\":\"2\",\"tdyt\":\"1\",\"ydkbm\":\"\",\"zjrxm\":\"郑光友\"}],\"fbfData\":[{\"fbfbm\":\"34082811220019\",\"fbfdz\":\"岳西县黄尾镇云峰村阔滩组\",\"fbffzrxm\":\"汪长新\",\"fbfmc\":\"安徽省安庆市岳西县黄尾镇云峰村光明组\",\"fzrzjhm\":\"340828196312167017\",\"fzrzjlx\":\"1\"}],\"fbfbm\":\"34082811220019\",\"files\":[{\"fileadrss\":\"http://110.185.174.27:18888/htwq/upload/20241127/99592fb8526ef5686c06801ffe30a3c2.pdf\",\"filemc\":\"340828112200190012J.pdf\"}],\"htfj\":\"1842625878932312066\",\"htid\":1830481213386788865,\"htzmjm\":1.57,\"jtcyData\":[{\"cbfbm\":\"340828112200190012\",\"cyxb\":\"1\",\"cyxm\":\"郑光友\",\"cyzjhm\":\"340828197207066913\",\"sfgyr\":\"1\",\"yhzgx\":\"02\"},{\"cbfbm\":\"340828112200190012\",\"cyxb\":\"2\",\"cyxm\":\"胡秀兰\",\"cyzjhm\":\"340828196812086926\",\"sfgyr\":\"1\",\"yhzgx\":\"10\"},{\"cbfbm\":\"340828112200190012\",\"cyxb\":\"1\",\"cyxm\":\"郑燊\",\"cyzjhm\":\"340828199010027012\",\"sfgyr\":\"1\",\"yhzgx\":\"20\"},{\"cbfbm\":\"340828112200190012\",\"cyxb\":\"2\",\"cyxm\":\"王盼盼\",\"cyzjhm\":\"340828199010294620\",\"sfgyr\":\"1\",\"yhzgx\":\"38\"},{\"cbfbm\":\"340828112200190012\",\"cybz\":\"6\",\"cyxb\":\"2\",\"cyxm\":\"郑若溪\",\"cyzjhm\":\"340828201608157022\",\"sfgyr\":\"1\",\"yhzgx\":\"42\"},{\"cbfbm\":\"340828112200190012\",\"cyxb\":\"1\",\"cyxm\":\"郑大江\",\"cyzjhm\":\"340828194608016913\",\"sfgyr\":\"1\",\"yhzgx\":\"51\"}],\"xjqydm\":\"340828112\",\"xjqymc\":\"黄尾镇\",\"xzqdm\":\"340828\",\"xzqmc\":\"岳西县\",\"yhtzmjm\":\"0.00\",\"zjqydm\":\"34082811220019\",\"zjqymc\":\"光明组\"}";
//        EncryptDTO encryptDTO = encryptData(json, myPublicKey, myPrivateKey, publicKey);

        //数据解密
        // String res = decDate(JSONObject.toJSONString(encryptDTO), privateKey);

        // System.out.println("解密结果：" + res);
        //获取密钥对
        getMyKeys();

    }
}
