package cn.gtmap.hlw.core.util.string;

import cn.gtmap.hlw.core.constant.Constants;
import cn.gtmap.hlw.core.enums.error.ErrorEnum;
import cn.gtmap.hlw.core.exception.BizException;
import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.security.SecureRandom;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @version V1.0
 * @Description 字符串工具类 :字符串校验、字符串替换、生成特定字符串（业务编号）等
 * @Author admin
 * @Date 2022/12/29 17:11
 **/
public class StringUtil {

    /**
     * 字符串常量：{@code "null"} <br>
     * 注意：{@code "null" != null}
     */
    public static final String NULL = "null";

    /**
     * 字符串常量：空字符串 {@code ""}
     */
    public static final String EMPTY = "";

    /**
     * 字符串常量：空格符 {@code " "}
     */
    public static final String SPACE = " ";

    /**
     * 汉语中数字大写
     */
    private static final String[] CN_UPPER_NUMBER = {"零", "壹", "贰", "叁", "肆",
            "伍", "陆", "柒", "捌", "玖"};
    /**
     * 汉语中货币单位大写，这样的设计类似于占位符
     */
    private static final String[] CN_UPPER_MONETRAY_UNIT = {"分", "角", "元",
            "拾", "佰", "仟", "万", "拾", "佰", "仟", "亿", "拾", "佰", "仟", "兆", "拾",
            "佰", "仟"};
    /**
     * 特殊字符：整
     */
    private static final String CN_FULL = "整";
    /**
     * 特殊字符：负
     */
    private static final String CN_NEGATIVE = "负";
    /**
     * 金额的精度，默认值为2
     */
    private static final int MONEY_PRECISION = 2;
    /**
     * 特殊字符：零元整
     */
    private static final String CN_ZEOR_FULL = "零元" + CN_FULL;

    /**
     * 在使用正则表达式时,利用好其预编译功能
     */
    private static final Pattern PATTERN_BASE64 = Pattern.compile("^data.(.*?);base64,");

    public static final String BDCQ_BH_LEFT_BRACKET = "(";
    public static final String BDCQ_BH_RIGHT_BRACKET = ")";
    /**
     * 中文括号
     */
    public static final String BDCQ_BH_LEFT_BRACKET_CN = "（";
    public static final String BDCQ_BH_RIGHT_BRACKET_CN = "）";
    /**
     * utf-8编码格式
     */
    public static final String ENCODING_UTF8 = "UTF-8";

    private static final Random RANDOM = new Random();

    /**
     * @return
     * @auto <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @description 生成受理编号
     **/
    public static synchronized String generateSlbh() {
        String slbh;
        SimpleDateFormat sf = new SimpleDateFormat("yyyyMMddHHmmss");
        String nowDate = sf.format(new Date());
        int radInt = RANDOM.nextInt(999);
        NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumIntegerDigits(3);
        nf.setMinimumIntegerDigits(3);
        String radStr = nf.format(radInt);
        slbh = String.format("%s%s", nowDate, radStr);
        return slbh;
    }

    /**
     * 校验字符串入参是否为空
     *
     * @return boolean
     * @Author admin
     * @Date 2023/1/5 13:39
     * @Param [args]
     **/
    public static boolean isContainEmpty(String... args) {
        if (args == null) {
            return false;
        } else {
            String[] str = args;
            int len = args.length;

            for (int i = 0; i < len; ++i) {
                String arg = str[i];
                if (arg == null || "".equals(arg)) {
                    return true;
                }
            }
            return false;
        }
    }

    /**
     * 校验字符串 是否相等 存在一个相等返回true
     *
     * @param value
     * @param args
     * @return boolean
     **/
    public static boolean isEquals(String value, String... args) {
        if (args == null && StringUtils.isEmpty(value)) {
            return true;
        }

        if (StringUtils.isNotEmpty(value)) {
            if (args == null) {
                return false;
            }
            String[] str = args;
            int len = args.length;

            for (int i = 0; i < len; ++i) {
                if (value.equals(str[i])) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @Description 校验字符串是否是base64字符串
     * @Author admin
     * @Date 2023/3/21 10:40
     * @Param str
     **/
    public static boolean isBase64(String str) {
        String base64Pattern = "^([A-Za-z0-9+/]{4})*([A-Za-z0-9+/]{4}|[A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{2}==)$";
        return Pattern.matches(base64Pattern, str);
    }

    /**
     * @author hyk
     * 在使用正则表达式时，利用好其预编译功能，可以有效加快正则匹配速度。
     * 说明：不要在方法体内定义：Pattern pattern = Pattern.compile(规则);
     */
    private static Pattern ZW_PATTERN = Pattern.compile("[\u4e00-\u9fa5]");

    public static String hex32() {
        String str = UUID.randomUUID().toString();
        return str.replace("-", "");
    }


    /**
     * 将容易引起xss漏洞的半角字符直接替换成全角字符
     *
     * @param s
     * @return
     */
    public static String xssEncode(String s) {
        if (s == null || s.isEmpty()) {
            return s;
        }
        StringBuilder sb = new StringBuilder(s.length() + 16);
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            switch (c) {
                case '>':
                    //全角大于号
                    sb.append('＞');
                    break;
                case '<':
                    //全角小于号
                    sb.append('＜');
                    break;
                case '\'':
                    //全角单引号
                    sb.append('‘');
                    break;
                case '\"':
                    //全角双引号
                    sb.append('“');
                    break;
                case '&':
                    //全角
                    sb.append('＆');
                    break;
                case '\\':
                    //全角斜线
                    sb.append('＼');
                    break;
                default:
                    sb.append(c);
                    break;
            }
        }
        return sb.toString();
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:liwenwu@gtmap.cn">liwenwu</a>
     * @version 2.0,
     * @description 判断是否包含中文
     */
    public static boolean isChinese(String str) {
        boolean flg = false;
        if (StringUtils.isNotBlank(str)) {
            Matcher matcher = ZW_PATTERN.matcher(str);
            if (matcher.find()) {
                flg = true;
            }
        }
        return flg;
    }

    /**
     * @param obj
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description
     */
    public static String toString(Object obj) {
        return obj != null ? obj.toString() : "";
    }


    /**
     * @param length 数字位数
     * @return:String
     * @author <a href="mailto:wangzhiwen@gtmap.cn">dingweiwei</a>
     * @description 生成N位随机数
     */
    public static String generateVerificationCode(int length) {
        // 可选的验证码字符
        String characters = "0123456789";
        // 生成指定长度的验证码
        StringBuilder code = new StringBuilder();
        for (int i = 0; i < length; i++) {
            // 从字符集中随机选择一个字符
            char randomChar = characters.charAt(RANDOM.nextInt(characters.length()));
            code.append(randomChar);
        }
        // 返回生成的验证码
        return code.toString();
    }

    /**
     * @param a
     * @param b
     * @return:boolean
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description 比较字符是否在字符串中
     */
    public static boolean indexOfStrs(String[] a, String b) {
        boolean flag = false;
        if (a != null) {
            for (String c : a) {
                if (StringUtils.equals(c, b)) {
                    flag = true;
                    break;
                }
            }
        }
        return flag;
    }

    /**
     * 生成证照简称
     *
     * @param str
     * @return 字符串
     * @auto <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     */
    public static String removeChineseAndBrackets(String str) {
        if (StringUtils.isNotBlank(str)) {
            // 使用正则表达式匹配中文和括号，替换为空字符串
            str = str.replaceAll("[\u4e00-\u9fa5()（）]", "");
        }
        return str;
    }

    /**
     * Double 转string 去除科学记数法显示
     *
     * @param d
     * @return string
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     */
    public static String double6ToStr(Double d) {
        if (d == null) {
            return "";
        }
        NumberFormat nf = NumberFormat.getInstance();
        nf.setGroupingUsed(false);
        nf.setMinimumFractionDigits(6);
        return (nf.format(d));
    }

    /**
     * 把输入的金额转换为汉语中人民币的大写
     *
     * @param numberOfMoney 输入的金额
     * @return 对应的汉语大写
     */
    public static String number2CnMontrayUnit(BigDecimal numberOfMoney) {
        StringBuffer sb = new StringBuffer();
        int signum = numberOfMoney.signum();
        // 零元整的情况
        if (signum == 0) {
            return CN_ZEOR_FULL;
        }
        //这里会进行金额的四舍五入
        long number = numberOfMoney.movePointRight(MONEY_PRECISION)
                .setScale(0, 4).abs().longValue();
        // 得到小数点后两位值
        long scale = number % 100;
        int numUnit = 0;
        int numIndex = 0;
        boolean getZero = false;
        // 判断最后两位数，一共有四中情况：00 = 0, 01 = 1, 10, 11
        if (scale <= 0) {
            numIndex = 2;
            number = number / 100;
            getZero = true;
        }
        if ((scale > 0) && (scale % 10 <= 0)) {
            numIndex = 1;
            number = number / 10;
            getZero = true;
        }
        int zeroSize = 0;
        while (true) {
            if (number <= 0) {
                break;
            }
            // 每次获取到最后一个数
            numUnit = (int) (number % 10);
            if (numUnit > 0) {
                if ((numIndex == 9) && (zeroSize >= 3)) {
                    sb.insert(0, CN_UPPER_MONETRAY_UNIT[6]);
                }
                if ((numIndex == 13) && (zeroSize >= 3)) {
                    sb.insert(0, CN_UPPER_MONETRAY_UNIT[10]);
                }
                sb.insert(0, CN_UPPER_MONETRAY_UNIT[numIndex]);
                sb.insert(0, CN_UPPER_NUMBER[numUnit]);
                getZero = false;
                zeroSize = 0;
            } else {
                ++zeroSize;
                if (!(getZero)) {
                    sb.insert(0, CN_UPPER_NUMBER[numUnit]);
                }
                if (numIndex == 2) {
                    if (number > 0) {
                        sb.insert(0, CN_UPPER_MONETRAY_UNIT[numIndex]);
                    }
                } else if (((numIndex - 2) % 4 == 0) && (number % 1000 > 0)) {
                    sb.insert(0, CN_UPPER_MONETRAY_UNIT[numIndex]);
                }
                getZero = true;
            }
            // 让number每次都去掉最后一个数
            number = number / 10;
            ++numIndex;
        }
        // 如果signum == -1，则说明输入的数字为负数，就在最前面追加特殊字符：负
        if (signum == -1) {
            sb.insert(0, CN_NEGATIVE);
        }
        // 输入的数字小数点后两位为"00"的情况，则要在最后追加特殊字符：整
        if (scale <= 0) {
            sb.append(CN_FULL);
        }
        return sb.toString();
    }

    /**
     * 把输入的万元金额转换为元
     *
     * @param amountInWanStr 万元
     * @return 元
     */
    public static double convertWanToYuan(String amountInWanStr) {
        double amountInWan = Double.parseDouble(amountInWanStr);
        return amountInWan * 10000;
    }

    /**
     * 把输入的万元金额转换为元
     *
     * @param amountInWanStr 万元
     * @return 元
     */
    public static Double convertWanToYuan(Double amountInWanStr) {
        if (amountInWanStr != null) {
            return amountInWanStr * 10000;
        }
        return null;
    }

    /**
     * 把输入的万元金额转换为元
     *
     * @param amountInWanStr 万元
     * @return 元
     */
    public static String convertWanToYuanStr(String amountInWanStr) {
        if (StringUtils.isNotBlank(amountInWanStr)) {
            double amountInWan = Double.parseDouble(amountInWanStr);
            double mul = NumberUtil.mul(amountInWan, 10000);
            NumberFormat nf = NumberFormat.getInstance();
            nf.setGroupingUsed(false);
            return nf.format(mul);
        }
        return "";
    }

    /**
     * 把输入的万元金额转换为元
     *
     * @param amountInWan 万元
     * @return 元
     */
    public static String convertWanToYuanStr(Double amountInWan) {
        if (amountInWan != null) {
            double value = amountInWan.doubleValue();
            double mul = NumberUtil.mul(value, 10000);
            NumberFormat nf = NumberFormat.getInstance();
            nf.setGroupingUsed(false);
            return nf.format(mul);
        }
        return "";
    }

    /**
     * 将输入的金额（以元为单位）转换为万元。
     *
     * @param amount 以元为单位的金额字符串
     * @return 转换后的金额，以万元为单位的字符串
     */
    public static String convertYuanToWanStr(String amount) {
        if (StringUtils.isBlank(amount)) {
            return amount;
        }
        try {
            // 将字符串转换为浮点数
            double yuanAmount = Double.parseDouble(amount);
            // 转换为万元，即除以10000
            double wanYuanAmount = yuanAmount / 10000.0;
            // 使用DecimalFormat来格式化，最多保留6位小数，去除多余的零
            DecimalFormat df = new DecimalFormat("#.######");  // 最多6位小数
            return df.format(wanYuanAmount);
        } catch (NumberFormatException e) {
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "万元转换出现异常");
        }
    }

    /**
     * 把输入的元金额转换为万元
     *
     * @param amountInYuan 元
     * @return 万元
     */
    public static String convertYuanToWanStr(Double amountInYuan) {
        if (amountInYuan != null) {
            double value = amountInYuan.doubleValue();
            double mul = NumberUtil.div(value, 10000);

            // 使用DecimalFormat来格式化，最多保留6位小数，去除多余的零
            DecimalFormat df = new DecimalFormat("#.######");  // 最多6位小数
            return df.format(mul);
        }
        return "";
    }


    /**
     * 将金额从元转换为万元，保留全部小数
     *
     * @param amountInYuan 金额，单位为元
     * @return 转换后的金额，单位为万元
     */
    public static Double convertYuanToWan(Double amountInYuan) {
        // 确保输入不为null
        if (amountInYuan == null) {
            return amountInYuan;
        }
        // 转换为万元并保留全部小数
        return amountInYuan / 10000.0;
    }

    /**
     * 去除json字符窜key里的空格
     *
     * @param json
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     */
    public static String trimJsonKey(JSONObject json) {
        Iterator<String> iterator = json.keySet().iterator();
        JSONObject jsonObject = new JSONObject();
        while (iterator.hasNext()) {
            String key = iterator.next();
            Object value = json.get(key);
            if (key.contains(" ")) {
                key = key.replace(" ", "");
            }
            jsonObject.put(key, value);
            if (json.get(key) instanceof JSONObject) {
                String s = trimJsonKey((JSONObject) json.get(key));
                jsonObject.put(key, JSONObject.parseObject(s));
            } else if (json.get(key) instanceof JSONArray) {
                JSONArray array = (JSONArray) json.get(key);
                JSONArray jsonArray = new JSONArray();
                for (Object item : array) {
                    if (item instanceof JSONObject) {
                        String s = trimJsonKey((JSONObject) item);
                        jsonArray.add(JSONObject.parseObject(s));
                    }
                }
                jsonObject.put(key, jsonArray);
            }
        }
        return jsonObject.toJSONString();
    }

    /**
     * 创建StringBuilder对象
     *
     * @param sb   初始StringBuilder
     * @param strs 初始字符串列表
     * @return StringBuilder对象
     */
    public static StringBuilder appendBuilder(StringBuilder sb, CharSequence... strs) {
        for (final CharSequence str : strs) {
            sb.append(str);
        }
        return sb;
    }

    /**
     * 正则表达式去掉base64格式字符串的前缀
     *
     * @param base64Str base64字符串
     * @return 字符串
     */
    public static String removeBase64Pre(String base64Str) {
        if (StringUtils.isNotBlank(base64Str)) {
            Matcher matcher = PATTERN_BASE64.matcher(base64Str);
            if (matcher.find()) {
                return base64Str.replace(matcher.group(), "");
            }
        }
        return base64Str;
    }

    /**
     * @author <a href="mailto:sunchao@gtmap.cn">sunchao</a>
     * @description base64解密
     */
    public static String decodeBase64(String base64Str) {
        String str = "";
        if (StringUtils.isNotBlank(base64Str)) {
            str = new String(Base64.decodeBase64(base64Str));
        }
        if (str.indexOf(BDCQ_BH_LEFT_BRACKET_CN) != -1) {
            str = str.replace(BDCQ_BH_LEFT_BRACKET_CN, BDCQ_BH_LEFT_BRACKET);
        }
        if (str.indexOf(BDCQ_BH_RIGHT_BRACKET_CN) != -1) {
            str = str.replace(BDCQ_BH_RIGHT_BRACKET_CN, BDCQ_BH_RIGHT_BRACKET);
        }
        return str;
    }

    /**
     * @author <a href="mailto:sunchao@gtmap.cn">sunchao</a>
     * @description base64加密
     */
    public static String encodeBase64(String str) {
        String base64Str = "";
        if (StringUtils.isNotBlank(str)) {
            base64Str = Base64.encodeBase64String(str.getBytes());
        }
        return base64Str;
    }

    /**
     * 生成16位不重复的随机数，含数字+大小写
     *
     * @return
     */
    public static String getGUID() {
        StringBuilder uid = new StringBuilder();
        //产生16位的强随机数
        Random rd = new SecureRandom();
        for (int i = 0; i < 16; i++) {
            //产生0-2的3位随机数
            int type = rd.nextInt(3);
            switch (type) {
                case 0:
                    //0-9的随机数
                    uid.append(rd.nextInt(10));
                    break;
                case 1:
                    //ASCII在65-90之间为大写,获取大写随机
                    uid.append((char) (rd.nextInt(25) + 65));
                    break;
                case 2:
                    //ASCII在97-122之间为小写，获取小写随机
                    uid.append((char) (rd.nextInt(25) + 97));
                    break;
                default:
                    break;
            }
        }
        return uid.toString();
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:chenjia@gtmap.cn">chenjia</a>
     * @description 判断是否是小数
     */
    public static boolean isDecimal(String str) {
        if (str == null) {
            return false;
        }
        // 正则表达式，匹配包括负号和小数点的数字，以及科学计数法表示的数字
        return str.matches("-?\\d+\\.\\d+([eE][-+]?\\d+)?");
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:chenjia@gtmap.cn">xusong</a>
     * @description 把base64转为byte
     */
    public static byte[] decodeBase64StrToByte(String base64Str) {
        byte[] bytes = null;
        if (StringUtils.isNotBlank(base64Str)) {
            bytes = Base64.decodeBase64(base64Str);
        }
        return bytes;
    }


    /**
     * @param str
     * @return
     * @author <a href="mailto:xusong@gtmap.cn">xusong</a>
     * @description 字符串转byte数组 编码UTF-8
     */
    public static byte[] strToByteUtf8(String str) {
        return strToByte(str, ENCODING_UTF8);
    }

    /**
     * @param str
     * @return
     * @author <a href="mailto:xusong@gtmap.cn">xusong</a>
     * @description 字符串转byte数组
     */
    public static byte[] strToByte(String str, String encode) {
        byte[] b = null;
        if (StringUtils.isNotBlank(str)) {
            try {
                b = str.getBytes(encode);
            } catch (UnsupportedEncodingException e) {
            }
        }
        return b;
    }

    /**
     * @param b
     * @return
     * @author <a href="mailto:xusong@gtmap.cn">xusong</a>
     * @description byte数组转字符串 编码UTF-8
     */
    public static String byteToStrUtf8(byte[] b) {
        return byteToStr(b, ENCODING_UTF8);
    }


    /**
     * @param b
     * @return
     * @author <a href="mailto:xusong@gtmap.cn">xusong</a>
     * @description byte数组转字符串
     */
    public static String byteToStr(byte[] b, String encode) {
        String str = null;
        if (null != b) {
            try {
                str = new String(b, encode);
            } catch (UnsupportedEncodingException e) {
            }
        }
        return str;
    }

    /**
     * 将集合中的元素使用指定的分隔符连接成一个字符串。
     * 如果集合为空或只包含空字符串，则返回空字符串。
     *
     * @param collection 要连接的集合，不应为null。
     * @param delimiter  用于连接字符串的分隔符，可以为null。
     * @return 连接后的字符串，如果集合为空或只包含空字符串，则返回空字符串。
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     */
    public static String join(Collection<?> collection, String delimiter) {
        if (CollectionUtils.isEmpty(collection)) {
            return "";
        }
        // 过滤掉空字符串
        Collection<?> filteredCollection = collection.stream()
                .filter(obj -> obj != null && !obj.toString().isEmpty())
                .collect(Collectors.toList());
        return StringUtils.join(filteredCollection, delimiter);
    }

    /**
     * 分割字符串
     *
     * @param str
     * @return
     */
    public static List<String> getSeparatorString(String str) {
        List<String> array = null;
        if (str != null) {
            //去除多余空格
            String realStr = str.trim().replaceAll("\\s+", " ");
            if (realStr.contains(Constants.SEMICOLON)) {
                array = Lists.newArrayList(realStr.split(Constants.SEMICOLON));
            } else if (realStr.contains(Constants.SEMICOLON_CN)) {
                array = Lists.newArrayList(realStr.split(Constants.SEMICOLON_CN));
            } else if (realStr.contains(Constants.COMMA)) {
                array = Lists.newArrayList(realStr.split(Constants.COMMA));
            } else if (realStr.contains(Constants.COMMA_CN)) {
                array = Lists.newArrayList(realStr.split(Constants.COMMA_CN));
            } else if (realStr.contains(Constants.SPACE)) {
                array = Lists.newArrayList(realStr.split(Constants.SPACE));
            } else if (realStr.contains(Constants.CAESURA_SIGN)) {
                array = Lists.newArrayList(realStr.split(Constants.CAESURA_SIGN));
            } else {
                array = new ArrayList<>();
                array.add(realStr);
            }
        }
        return array;
    }

    /**
     * 提取参数
     *
     * @param data
     * @param paramName
     * @return
     */

    public static String extractParam(String data, String paramName) {
        // 找到paramid=后面的位置
        int index = data.indexOf(paramName);
        if (index == -1) {
            // 如果没有找到paramid=，返回null或者适当的错误处理
            return null;
        }

        // 截取paramid=后面的值
        String paramid = data.substring(index + paramName.length());

        // 如果还有&符号，截取到&为止
        int endIndex = paramid.indexOf("&");
        if (endIndex != -1) {
            paramid = paramid.substring(0, endIndex);
        }

        return paramid;
    }

}
