
package cn.gtmap.hlw.core.util.date;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

/**
 * @Description bean工具类
 * @Author admin
 * @Date 2023/4/24 18:21
 */
public class DateUtils {

    private static final Logger log = LoggerFactory.getLogger(DateUtils.class);

    /**
     * 格式为yyyy-MM-dd HH:mm:ss.SSS
     */
    public static final String DATE_FORMAT_FULL = "yyyy-MM-dd HH:mm:ss.SSS";
    /**
     * 格式为yyyy-MM-dd HH:mm:ss
     */
    public static final String DATE_FORMAT_LONG = "yyyy-MM-dd HH:mm:ss";
    /**
     * 格式为yyyy年MM月dd日 HH:mm:ss
     */
    public static final String DATE_FORMAT_LONG_LYR = "yyyy年MM月dd日 HH:mm:ss";
    /**
     * 格式为yyyy-MM-dd
     */
    public static final String DATE_FORMAT_SHORT = "yyyy-MM-dd";

    public static final String DATE_FORMAT_HHMMSS = "HHmmss";
    public static final String DATE_FORMAT_HH_MM_SS = "HH:mm:ss";

    /**
     * 格式为：yyyyMMddHHmmss
     */
    public static final String PURE_DATETIME_PATTERN = "yyyyMMddHHmmss";
    /**
     * 格式为：yyyyMMddHHmmssSSS
     */
    public static final String PURE_LONG_DATETIME_PATTERN = "yyyyMMddHHmmssSSS";
    /**
     * 标准日期格式：yyyyMMdd
     */
    public static final String PURE_DATE_PATTERN = "yyyyMMdd";
    /**
     * 格式为yyyy年MM月dd日
     */
    public static final String DATE_FORMAT_YMD = "yyyy年MM月dd日";

    /**
     * 文件夹格式
     */
    public static final String DATE_FORMAT_FILE = "yyyy" + File.separator + "MM" + File.separator + "dd";

    //数字对应的汉字
    static String[] chineseNumbers = {"零", "一", "二", "三", "四", "五", "六", "七", "八", "九"};
    //日期中文
    static String[] chineseUnits = {"年", "月", "日"};

    /**
     * 将日期格式化为"yyyy-MM-dd"格式的日期
     */
    public static Date date2SimpleDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        //清除时、分、秒、毫秒
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 将日期格式化为指定格式的日期
     */
    public static Date date2FormatDate(Date date, String formatStr) {
        String tmp = dateToStr(date, formatStr);
        return strToDate(tmp, formatStr);
    }

    /**
     * 返回格式化的日期字符串
     *
     * @param date   日期
     * @param format 格式
     * @return 格式化的日期字符串
     */
    public static String dateToStr(Date date, String format) {
        if (date == null) {
            return null;
        }
        DateFormat df = new SimpleDateFormat(format);
        return df.format(date);
    }

    /**
     * 返回格式化的日期字符串
     *
     * @param format 格式
     * @return 格式化的日期字符串
     */
    public static String dateToStr(String format) {
        return dateToStr(new Date(), format);
    }

    /**
     * 返回格式化的日期字符串
     *
     * @param date 日期
     * @return 格式化的日期字符串
     */
    public static String dateToStr(Date date) {
        return dateToStr(date, DATE_FORMAT_LONG);
    }

    /**
     * 返回格式化的日期字符串(带T)
     *
     * @param date
     * @return
     */
    public static String dateToString(Date date) {
        String dateStr = dateToStr(date, "yyyy-MM-dd&HH:mm:ss");
        if (null != dateStr) {
            return dateStr.replace('&', 'T');
        }
        return null;
    }

    /**
     * 将字符串转换为日期
     *
     * @param dateStr 日期字符串
     * @param format  格式
     * @return 日期
     * @throws ParseException
     */
    public static Date strToDate(String dateStr, String format) {
        if (dateStr == null) {
            return null;
        }
        DateFormat df = new SimpleDateFormat(format);
        try {
            return df.parse(dateStr);
        } catch (ParseException e) {
            log.debug("strToDate", e);
        }
        return null;
    }

    /**
     * 将字符串转换为日期
     *
     * @param dateStr 日期字符串
     * @return 日期
     * @throws ParseException
     */
    public static Date strToDate(String dateStr) {
        return strToDate(dateStr, DATE_FORMAT_LONG);
    }

    /**
     * 判断字符串是否为日期类型("yyyy-MM-dd")
     *
     * @param dateStr 字符串
     * @return true是日期类型，否则返回false
     */
    public static boolean isDate(String dateStr) {
        if (dateStr == null || !dateStr.matches("^\\d{4}-\\d{2}-\\d{2}$")) {
            return false;
        }
        String[] parts = dateStr.split("-");
        int year = Integer.parseInt(parts[0]);
        int month = Integer.parseInt(parts[1]);
        int day = Integer.parseInt(parts[2]);
        return isValidDate(year, month, day);
    }

    private static boolean isValidDate(int year, int month, int day) {
        if (month < 1 || month > 12 || day < 1 || day > 31) {
            return false;
        }
        if (month == 2) {
            return isLeapYear(year) ? day <= 29 : day <= 28;
        }
        if (month == 4 || month == 6 || month == 9 || month == 11) {
            return day <= 30;
        }
        return true; // 1,3,5,7,8,10,12月有31天
    }

    private static boolean isLeapYear(int year) {
        return (year % 4 == 0 && year % 100 != 0) || (year % 400 == 0);
    }

    /**
     * 判断字符串是否为日期带时间类型("yyyy-MM-dd HH:mm:ss")
     *
     * @param dateStr 字符串
     * @return true是日期带时间类型，否则返回false
     */
    public static boolean isDateTime(String dateStr) {
        return dateStr.matches("^((([0-9]{3}[1-9]|[0-9]{2}[1-9][0-9]{1}|[0-9]{1}[1-9][0-9]{2}|[1-9][0-9]{3})-(((0[13578]|1[02])-(0[1-9]|[12][0-9]|3[01]))|((0[469]|11)-(0[1-9]|[12][0-9]|30))|(02-(0[1-9]|[1][0-9]|2[0-8]))))|((([0-9]{2})(0[48]|[2468][048]|[13579][26])|((0[48]|[2468][048]|[3579][26])00))-02-29)) (([01]{1}\\d{1}|2[0-3]{1}):[0-5]{1}\\d{1}:[0-5]{1}\\d{1})$");
    }

    /**
     * 判断字符串是否为时间类型("HH:mm:ss")
     *
     * @param dateStr 字符串
     * @return true是时间类型，否则返回false
     */
    public static boolean isTime(String dateStr) {
        return dateStr.matches("^([01]{1}\\d{1}|2[0-3]{1}):[0-5]{1}\\d{1}:[0-5]{1}\\d{1}$");
    }

    /**
     * 日期增加或减去某个天数
     *
     * @param dt  需要操作的时间
     * @param num 增加或减去的天数
     * @return 返回操作后的时间
     */
    public static Date getBeforeOrAfterDate(Date dt, int num) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(dt);
        cal.add(Calendar.DATE, num);
        return cal.getTime();
    }

    /**
     * 日期增加或减去某个时间单位数
     *
     * @param dt    需要操作的时间
     * @param num   增加或减去的天数
     * @param field 时间单位类型(年，月，日，星期，分钟等)
     * @return 返回操作后的时间
     */
    public static Date getBeforeOrAfterDate(Date dt, int num, int field) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(dt);
        cal.add(field, num);
        return cal.getTime();
    }

    /**
     * 日期增加或减去某个天数
     *
     * @param dt  需要操作的时间字符串
     * @param num 增加或减去的天数
     * @return 返回操作后的时间字符串
     */
    public static String getBeforeOrAfterDate(String dt, int num) {
        if (isDate(dt)) {
            Date nextDate = getBeforeOrAfterDate(strToDate(dt, DATE_FORMAT_SHORT), num);
            return dateToStr(nextDate, DATE_FORMAT_SHORT);
        } else if (isDateTime(dt)) {
            Date nextDate = getBeforeOrAfterDate(strToDate(dt, DATE_FORMAT_LONG), num);
            return dateToStr(nextDate, DATE_FORMAT_LONG);
        }
        return null;
    }

    /**
     * 日期增加或减去某个天数
     *
     * @param dt     需要操作的时间字符串
     * @param num    增加或减去的天数
     * @param format 返回的日期字符串格式
     * @return 返回操作后的时间字符串 ,带格式format
     */
    public static String getBeforeOrAfterDate(String dt, int num, String format) {
        Date nextDate = null;
        if (isDate(dt)) {
            nextDate = getBeforeOrAfterDate(strToDate(dt, DATE_FORMAT_SHORT), num);
        } else if (isDateTime(dt)) {
            nextDate = getBeforeOrAfterDate(strToDate(dt, DATE_FORMAT_LONG), num);
        }
        return dateToStr(nextDate, format);
    }

    /**
     * 返回日期是星期几
     * Calendar.SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY
     *
     * @param dt
     * @return
     */
    public static int getDayOfWeek(Date dt) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(dt);
        return cal.get(Calendar.DAY_OF_WEEK);
    }

    /**
     * 返回打印模板需要的当前时间
     *
     * @return
     */
    public static String getPrintDate() {
        // 创建一个日期对象，表示当前时间
        Date now = new Date();
        // 定义所需的日期格式
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy年MM月dd日HH时mm分", Locale.CHINA);
        return sdf.format(now);
    }

    /**
     * 返回打印模板需要的当前时间
     *
     * @return
     */
    public static String getPrintDateNYR() {
        // 创建一个日期对象，表示当前时间
        Date now = new Date();
        // 定义所需的日期格式
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy年MM月dd日", Locale.CHINA);
        return sdf.format(now);
    }

    /**
     * 返回指定日期的月份
     *
     * @param date 日期
     * @return 月份
     */
    public static String getMonthByDate(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return String.valueOf(calendar.get(Calendar.MONTH) + 1);
    }

    /**
     * 根据年月日获取时间
     *
     * @param year  年
     * @param month 月
     * @param day   日
     * @return 返回指定时间
     */
    public static Date getDate(int year, int month, int day) {
        Calendar calendar = Calendar.getInstance();
        calendar.clear();
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, month);
        calendar.set(Calendar.DAY_OF_MONTH, day);
        return calendar.getTime();
    }

    public static String convertLongToFormattedString(long timestamp, String format) {
        Date date = new Date(timestamp);
        SimpleDateFormat dateFormat = new SimpleDateFormat(format);
        return dateFormat.format(date);
    }

    /**
     * 日期格式化
     *
     * @param inputDateStr 字符串日期
     * @param inputFormat  字符串日期格式
     * @param outputFormat 目标字符串格式
     * @return 格式化后的字符串日期
     */
    public static String convertDateFormat(String inputDateStr, String inputFormat, String outputFormat) {
        if (StringUtils.isBlank(inputDateStr)) {
            return inputDateStr;
        }
        SimpleDateFormat inputDateFormat = new SimpleDateFormat(inputFormat);
        SimpleDateFormat outputDateFormat = new SimpleDateFormat(outputFormat);
        Date date;
        try {
            date = inputDateFormat.parse(inputDateStr);
        } catch (ParseException e) {
            return inputDateStr;
        }
        return outputDateFormat.format(date);
    }

    public static String DateToChinese(Date date) {
        StringBuilder dateStr = new StringBuilder();
        LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        // 获取年、月、日
        int year = localDate.getYear();
        int month = localDate.getMonthValue();
        int day = localDate.getDayOfMonth();

        // 将年、月、日分别转换为汉字
        dateStr.append(yearToChinese(year, chineseNumbers)).append(chineseUnits[0]);
        dateStr.append(monthToChinese(month, chineseNumbers)).append(chineseUnits[1]);
        dateStr.append(dayToChinese(day, chineseNumbers)).append(chineseUnits[2]);
        return dateStr.toString();
    }

    private static String yearToChinese(int year, String[] chineseNumbers) {
        StringBuilder result = new StringBuilder();
        while (year > 0) {
            result.insert(0, chineseNumbers[year % 10]);
            year /= 10;
        }
        return result.toString();
    }

    // 月转换为汉字
    private static String monthToChinese(int month, String[] chineseNumbers) {
        if (month < 1 || month > 12) {
            throw new IllegalArgumentException("Invalid month: " + month);
        }
        if (month < 10) {
            return chineseNumbers[month % 10];
        } else {
            return "十" + ((month % 10) != 0 ? chineseNumbers[month % 10] : "");
        }
    }

    // 日转换为汉字
    private static String dayToChinese(int day, String[] chineseNumbers) {
        if (day < 1 || day > 31) {
            throw new IllegalArgumentException("Invalid day: " + day);
        }
        if (day < 10) {
            return chineseNumbers[day % 10];
        } else if (day > 9 && day < 20) {
            return "十" + ((day % 10) != 0 ? chineseNumbers[day % 10] : "");
        } else {
            return chineseNumbers[day / 10] + "十" + ((day % 10) != 0 ? chineseNumbers[day % 10] : "");
        }
    }
}
