package cn.gtmap.hlw.core.util.file;

import cn.gtmap.hlw.core.constant.Constants;
import cn.gtmap.hlw.core.enums.error.ErrorEnum;
import cn.gtmap.hlw.core.exception.BizException;
import cn.gtmap.hlw.core.util.date.DateUtils;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.StrUtil;
import com.itextpdf.text.Document;
import com.itextpdf.text.pdf.PdfCopy;
import com.itextpdf.text.pdf.PdfImportedPage;
import com.itextpdf.text.pdf.PdfReader;
import org.apache.commons.lang3.StringUtils;
import org.apache.pdfbox.multipdf.PDFMergerUtility;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.web.multipart.MultipartFile;
import sun.misc.BASE64Decoder;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Base64;
import java.util.Date;
import java.util.List;

/**
 * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
 * @version 1.0, 2023/7/13
 * @description 文件操作工具类
 */
public class FileUtils {

    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);

    /**
     * 每次处理1MB的数据
     */
    private static final int BUFFER_SIZE = 1024 * 1024;

    /**
     * @param uploadFilePath 文件路径
     * @param fileName       文件名称
     * @param file           文件流
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/7/13
     * @description 上传文件到指定路径
     */
    public static File uploadFile(String uploadFilePath, String fileName, byte[] file) {
        if (StringUtils.isNotBlank(uploadFilePath)) {
            //创建目录
            FileUtil.mkdir(uploadFilePath);
            //上传文件
            return FileUtil.writeBytes(file, uploadFilePath + File.separator + fileName);
        }
        return null;
    }

    /**
     * @param uploadFilePath 文件路径
     * @param base64Str      文件base64字符串
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/9/2
     * @description 上传base64字符串文件到指定路径
     */
    public static void uploadBase64File(String uploadFilePath, String fileName, String base64Str) {
        // 去掉头，否则生成文件有问题
        base64Str = base64Str.replace("data:image/jpeg;base64,", "").replace("data:image/png;base64,", "")
                .replace("data:image/gif;base64,", "").replace("data:image/jpg;base64,", "")
                .replace("data:image/png;base64,", "");
        BASE64Decoder decoder = new BASE64Decoder();
        try {
            byte[] bytes = decoder.decodeBuffer(base64Str);
            uploadFile(uploadFilePath, fileName, bytes);
        } catch (Exception e) {
            logger.error("uploadBase64File  ERROR：{}", e.getMessage());
            throw new BizException(ErrorEnum.SERVICE_ERROR, "上传base64字符串文件到指定路径出现异常");
        }
    }

    /**
     * @param filepath 文件路径
     * @param fjmc     附件名称
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/8/23
     * @description 获取附件的base64格式字符
     */
    public static String getFjBase64(String filepath, String fjmc) {

        if (StringUtils.isNotBlank(fjmc)) {
            filepath = filepath + "\\" + fjmc;
        }
        File file = new File(filepath);

        if (!file.exists()) {
            return "";
        }
        // 读取文件内容为字节数组
        byte[] fileContent;
        try {
            fileContent = readBytes(file);
        } catch (IOException e) {
            throw new BizException(ErrorEnum.SERVICE_ERROR, "查看附件信息异常");
        }
        // 将字节数组编码为Base64字符串
        String base64String = Base64.getEncoder().encodeToString(fileContent);
        base64String = base64String.replaceAll("data/image/png/base64/", "");
        return base64String;
    }


    /**
     * @param filePath 文件路径
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/8/6
     * @description 获取文件的 MIME 类型
     */
    public static String getMimeType(String filePath) {
        String mimeType = FileUtil.getMimeType(filePath);
        if (StringUtils.isBlank(mimeType)) {
            // 补充一些常用的mimeType
            if (StrUtil.endWithIgnoreCase(filePath, Constants.PDF)) {
                mimeType = "application/pdf";
            } else if (StrUtil.endWithIgnoreCase(filePath, Constants.OFD)) {
                mimeType = "application/ofd";
            } else if (StrUtil.endWithIgnoreCase(filePath, Constants.JPEG)) {
                mimeType = "image/jpeg";
            } else if (StrUtil.endWithIgnoreCase(filePath, Constants.JPG)) {
                mimeType = "image/jpeg";
            } else if (StrUtil.endWithIgnoreCase(filePath, Constants.PNG)) {
                mimeType = "image/png";
            } else if (StrUtil.endWithIgnoreCase(filePath, Constants.GIF)) {
                mimeType = "image/gif";
            }
        }
        return mimeType;
    }


    /**
     * @param uploadPath 文件路径
     * @param slbh       受理编号
     * @param xmid       项目id
     * @param fjid       附件id
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/8/6
     * @description 生成附件上传路径
     */
    public static String getFilePath(String uploadPath, String slbh, String xmid, String fjid) {
        StringBuilder sb = new StringBuilder();
        String dateToStr = DateUtils.dateToStr(new Date(), DateUtils.DATE_FORMAT_FILE);
        sb.append(uploadPath).
                append(File.separatorChar).
                append(dateToStr).
                append(File.separatorChar).
                append(slbh).
                append(File.separatorChar).
                append(xmid).
                append(File.separator).
                append(fjid);
        return sb.toString();
    }

    /**
     * @param file 文件流
     * @return MultipartFile
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/7/18
     * @description file 转 MultipartFile
     */
    public static MultipartFile getMultipartFile(File file) throws RuntimeException {
        FileInputStream inputStream = null;
        MultipartFile multipartFile = null;
        try {
            inputStream = new FileInputStream(file);
            multipartFile = new MockMultipartFile(file.getName(), file.getName(),
                    Files.probeContentType(file.toPath()), inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    logger.error("文件流转换异常", e);
                }
            }
        }
        return multipartFile;
    }

    /**
     * @param inputStream 输入流
     * @return MultipartFile
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/7/19
     * @description inputStream 转 byte[]
     */
    public static byte[] convertToByteArray(InputStream inputStream) throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int bytesRead;
        while ((bytesRead = inputStream.read(buffer)) != -1) {
            outputStream.write(buffer, 0, bytesRead);
        }
        inputStream.close();
        return outputStream.toByteArray();
    }

    /**
     * @param byteArray
     * @return String
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/7/19
     * @description byte[]转base64
     */
    public static String byteToBase64(byte[] byteArray) {
        // byte[]转base64
        return Base64.getEncoder().encodeToString(byteArray);
    }

    /**
     * @param inputStream 输入流
     * @return MultipartFile
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/8/4
     * @description inputStream 转 base64字符串
     */
    public static String encodeToBase64(InputStream inputStream) throws IOException {
        // 1. 读取 InputStream 中的字节数据
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int count;
        while ((count = inputStream.read(buffer)) != -1) {
            outputStream.write(buffer, 0, count);
        }
        byte[] byteArray = outputStream.toByteArray();
        // 2. 将字节数组转换为 Base64 编码格式
        String base64String = Base64.getEncoder().encodeToString(byteArray);
        // 3. 返回 Base64 编码后的字符串
        outputStream.close();
        return base64String;
    }

    /**
     * @param base64String base64字符串
     * @param fileName     文件名
     * @return MultipartFile
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2024/5/7
     * @description base64字符串 转 MultipartFile
     */
    public static MultipartFile convertMultipartFile(String base64String, String fileName) {
        // 解码base64字符串
        byte[] imageBytes = Base64.getDecoder().decode(base64String);
        // 创建字节数组输入流
        ByteArrayInputStream inputStream = new ByteArrayInputStream(imageBytes);
        // 使用MockMultipartFile模拟MultipartFile
        MultipartFile multipartFile = null;
        try {
            multipartFile = new MockMultipartFile(fileName, fileName, MediaType.APPLICATION_PDF_VALUE, inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return multipartFile;
    }


    public static byte[] images2pdf(File[] files) {
        //创建空PDF
        PDDocument document = new PDDocument();
        for (File image : files) {
            InputStream input = null;
            try {
                input = new FileInputStream(image);
                //读取图片
                BufferedImage bufferedImage = ImageIO.read(input);
                float width = bufferedImage.getWidth();
                float height = bufferedImage.getHeight();
                //创建PDF的一页
                PDPage page = new PDPage(new PDRectangle(width, height));
                document.addPage(page);
                PDImageXObject pdImageXobject = PDImageXObject.createFromFileByContent(image, document);
                try (PDPageContentStream contentStream = new PDPageContentStream(document, page)) {  // 自动关闭
                    contentStream.drawImage(pdImageXobject, 0, 0);
                }
            } catch (Exception e) {
                logger.error("图片转pdf失败，失败原因：{}", e);
            } finally {
                try {
                    if (input != null) {
                        input.close();
                    }
                } catch (Exception e) {
                    logger.error("图片转pdf时流关闭失败，失败原因：{}", e);
                }
            }
        }
        ByteArrayOutputStream output = null;
        try {
            output = new ByteArrayOutputStream();
            document.save(output);
            return output.toByteArray();
        } catch (Exception e) {

        } finally {
            try {
                if (output != null) {
                    output.close();
                }
            } catch (Exception e) {
                logger.error("图片转pdf时流关闭失败，失败原因：{}", e);
            }
        }
        return null;
    }


    /**
     * @param fileName 文件名
     * @return 文件后缀名
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/11/28
     * @description 获取文件名的后缀名，如果没有后缀名则返回空字符串
     */
    public static String getFileExtension(String fileName) {
        if (fileName == null || fileName.isEmpty()) {
            return "";
        }
        int lastDotIndex = fileName.lastIndexOf(".");
        if (lastDotIndex == -1 || lastDotIndex == fileName.length() - 1) {
            // 如果没有找到点，或者点是最后一个字符，表示没有后缀名
            return "";
        }
        return fileName.substring(lastDotIndex + 1);
    }


    /**
     * @param filePath 文件路径
     * @param newName  新的文件名
     * @return true表示重命名成功，false表示重命名失败
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2023/11/28
     * @description 给文件重命名，仅修改文件名，不修改后缀名
     */
    public static boolean renameFile(String filePath, String newName) {
        File oldFile = new File(filePath);

        // 检查文件是否存在
        if (!oldFile.exists()) {
            return false;
        }

        // 获取文件所在目录和文件后缀
        String parentDir = oldFile.getParent();
        String fileExtension = getFileExtension(oldFile.getName());

        // 构建新的文件路径
        String newFilePath = parentDir + File.separator + newName + "." + fileExtension;

        // 创建新的File对象
        File newFile = new File(newFilePath);

        // 尝试重命名文件
        if (oldFile.renameTo(newFile)) {
            return true;
        }
        return false;
    }

    //todo:驼峰
    public static String MultipartFileToString(MultipartFile multipartFile) {
        InputStreamReader isr;
        BufferedReader br;
        StringBuilder txtResult = new StringBuilder();
        try {
            isr = new InputStreamReader(multipartFile.getInputStream(), StandardCharsets.UTF_8);
            br = new BufferedReader(isr);
            String lineTxt;
            while ((lineTxt = br.readLine()) != null) {
                txtResult.append(lineTxt);
            }
            isr.close();
            br.close();
            return txtResult.toString();
        } catch (IOException e) {
            logger.error("MultipartFileToString", e);
            return "";
        }
    }

    /**
     * 图片到byte数组
     *
     * @param path
     * @return
     */
    public static byte[] image2byte(String path) {
        byte[] data = null;
        try (FileInputStream input = new FileInputStream(path);
             ByteArrayOutputStream output = new ByteArrayOutputStream()) {
            // 选择合适的缓冲区大小
            byte[] buf = new byte[8192];
            int numBytesRead;
            while ((numBytesRead = input.read(buf)) != -1) {
                output.write(buf, 0, numBytesRead);
            }
            data = output.toByteArray();
        } catch (IOException e) {
            logger.error("image2byte", e);
        }
        return data;
    }


    /**
     * @param base64Str Base64字符串
     * @return File   文件对象
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @version 1.0, 2024/08/22
     * @description 使用分块的方式，将Base64字符串转换为文件
     */
    public static File base64ToFile(String base64Str) {
        logger.info("base64分块转换开始，时间：{}", DateUtils.dateToStr(new Date()));
        // 创建目标文件
        File outputFile;
        try {
            outputFile = File.createTempFile("temp", Constants.ZIP);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        try (FileOutputStream fos = new FileOutputStream(outputFile)) {
            // 计算Base64字符串的长度
            int length = base64Str.length();
            int offset = 0;

            while (offset < length) {
                // 计算当前块的结束位置
                int end = Math.min(offset + BUFFER_SIZE, length);
                // 获取当前块的Base64子串
                String base64Chunk = base64Str.substring(offset, end);
                // 解码当前块
                byte[] dataChunk = Base64.getDecoder().decode(base64Chunk);
                // 写入文件
                fos.write(dataChunk);
                // 更新偏移量
                offset = end;
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        logger.info("base64分块转换结束，时间：{}", DateUtils.dateToStr(new Date()));
        return outputFile;
    }

    /**
     * @param bytes pdf字节流list
     * @return byte[]
     * @description 获取合并后的PDF文件
     * @version 2.0, 2020-8-11
     */
    public static byte[] getCombinePdf(List<byte[]> bytes) {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        Document document = new Document();
        try {
            PdfCopy copy = new PdfCopy(document, byteArrayOutputStream);
            document.open();
            for (byte[] bs : bytes) {
                PdfReader reader = new PdfReader(bs);
                int pageTotal = reader.getNumberOfPages();
                for (int pageNo = 1; pageNo <= pageTotal; pageNo++) {
                    document.newPage();
                    PdfImportedPage page = copy.getImportedPage(reader, pageNo);
                    copy.addPage(page);
                }
                reader.close();
            }
            document.close();
            byte[] pdfs = byteArrayOutputStream.toByteArray();
            byteArrayOutputStream.close();
            copy.close();
            return pdfs;
        } catch (Exception e) {
            logger.error("pdf文件合并失败, 失败原因：" + e.getMessage());
        }
        return null;
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:chenjia@gtmap.cn">chenjia</a>
     * @description 合并pdf文件
     */
    public static void hbpdf(List<String> pdfFiles) {
        Document document = null;
        FileOutputStream fileOutputStream = null;
        PdfReader pdfReader = null;
        String filePath = StringUtils.replace(pdfFiles.get(0), ".pdf", "1.pdf");
        try {
            // 新建Document，设置纸张大小，根据随便一个PDF文档大小设置
            pdfReader = new PdfReader(pdfFiles.get(0));
            document = new Document(pdfReader.getPageSize(1));
            fileOutputStream = new FileOutputStream(filePath);
            PdfCopy copy = new PdfCopy(document, fileOutputStream);
            document.open();

            // 遍历需要保证取出按照文档顺序
            for (int index = 0; index < pdfFiles.size(); index++) {
                // 每一份PDF
                PdfReader reader = new PdfReader(pdfFiles.get(index));
                // 处理每一页（需要注意索引从1开始）
                for (int i = 1; i <= reader.getNumberOfPages(); i++) {
                    document.newPage();
                    copy.addPage(copy.getImportedPage(reader, i));
                }
                reader.close();
            }
            copy.close();
        } catch (Exception e) {
            // throw new AppException("合并PDF失败，原因：" + e.toString());
        } finally {
            // 依次关闭相关连接，避免应用文件句柄占用
            if (null != pdfReader) {
                pdfReader.close();
            }

            if (null != document) {
                document.close();
            }

            if (null != fileOutputStream) {
                try {
                    fileOutputStream.close();
                } catch (IOException e) {
                }
            }
        }
        try {
            //复制pdf文件到原路径
            org.apache.commons.io.FileUtils.copyFile(new File(filePath), new File(pdfFiles.get(0)));
            //删除临时文件
            boolean delete = new File(filePath).delete();
            if (!delete) {
                logger.info("Failed to delete file: " + filePath);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 合并多个PDF的Base64字符串，并返回合并后的PDF的Base64字符串
     *
     * @param base64Pdfs 多个PDF的Base64字符串数组
     * @return 合并后的PDF的Base64字符串
     * @throws IOException 如果发生IO异常
     */
    public static String mergePdfs(List<String> base64Pdfs) throws IOException {
        PDFMergerUtility mergerUtility = new PDFMergerUtility();

        for (String base64Pdf : base64Pdfs) {
            byte[] pdfBytes = Base64.getDecoder().decode(base64Pdf);
            try (PDDocument document = PDDocument.load(new ByteArrayInputStream(pdfBytes))) {
                mergerUtility.addSource(new ByteArrayInputStream(pdfBytes));
            }
        }

        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            mergerUtility.setDestinationStream(outputStream);
            mergerUtility.mergeDocuments(null);
            byte[] mergedPdfBytes = outputStream.toByteArray();
            return Base64.getEncoder().encodeToString(mergedPdfBytes);
        }
    }

    /**
     * 读取字节
     *
     * @param file
     * @return:byte[]
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     */
    public static byte[] readBytes(File file) throws IOException {
        try (InputStream fis = new FileInputStream(file);
             InputStream is = new BufferedInputStream(fis);
             ByteArrayOutputStream baos = new ByteArrayOutputStream()) {

            byte[] buf = new byte[1024];
            int count;
            while ((count = is.read(buf)) != -1) {
                baos.write(buf, 0, count);
            }
            return baos.toByteArray();
        }
    }
}
