package cn.gtmap.hlw.core.util.file;

import lombok.extern.slf4j.Slf4j;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Base64;

/**
 * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
 * @version 1.0, 2025/2/23
 * @description pdf签名工具类
 */
@Slf4j
public class PdfSigner {

    /**
     * @param inputPdfFile pdf文件对象
     * @param base64Image  签名图片base64
     * @return 合并后的pdf字节流
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @description pdf与签名合并
     */
    public static byte[] addSignatureToPdf(File inputPdfFile, String base64Image, Float x, Float y) throws IOException {
        // 处理可能的Data URI前缀（如"data:image/png;base64,"）
        String pureBase64 = base64Image.contains(",")
                ? base64Image.split(",")[1]
                : base64Image;

        try (PDDocument document = PDDocument.load(inputPdfFile)) {
            // Base64解码
            byte[] imageBytes = Base64.getDecoder().decode(pureBase64);

            // 创建图片对象
            PDImageXObject image = PDImageXObject.createFromByteArray(document, imageBytes, "signature");

            // 获取页数，默认签字图片放到最后一页
            int pageIndex = document.getNumberOfPages() -1;

            // 获取第一页（页码从0开始）
            PDPage page = document.getPage(pageIndex);

            // 计算位置和大小（右下角布局）
            float margin = 20;      // 边距
            float imgWidth = 40;    // 图片宽度
            float imgHeight = (imgWidth / image.getWidth()) * image.getHeight();

            float pageWidth = page.getMediaBox().getWidth();
            if (x == null) {
                // X坐标：右对齐
                x = pageWidth - imgWidth - margin;
            }
            if (y == null) {
                // Y坐标：底部对齐（PDF坐标系原点在左下角）
                y = margin;
            }
            // 将图片绘制到PDF
            try (PDPageContentStream contentStream = new PDPageContentStream(
                    document,
                    page,
                    PDPageContentStream.AppendMode.APPEND,
                    true,
                    true)) {
                contentStream.drawImage(image, x, y, imgWidth, imgHeight);
            } catch (Exception e) {
                log.info("签名合并异常", e);
            }
            // 输出为字节数组
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            document.save(outputStream);
            return outputStream.toByteArray();
        }
    }
}