package cn.gtmap.hlw.core.config;

import cn.gtmap.hlw.core.enums.error.ErrorEnum;
import cn.gtmap.hlw.core.exception.BizException;
import cn.gtmap.hlw.core.util.encryption.sm2.Sm2lib;
import cn.gtmap.hlw.core.util.http.HttpContextUtils;
import com.google.common.base.Joiner;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 过滤器中 对出入参进行操作的时 重写出入参读取
 *
 * @Author admin
 * @Date 2023/3/2 15:46
 * @Version V1.0
 **/
@Slf4j
public class ParameterRequestWrapper extends HttpServletRequestWrapper {

    private static final Logger logger = LoggerFactory.getLogger(ParameterRequestWrapper.class);

    private Map<String, String[]> params = new HashMap<>();

    /**
     * Constructs a request object wrapping the given request.
     *
     * @param request
     * @throws IllegalArgumentException if the request is null
     */
    public ParameterRequestWrapper(HttpServletRequest request) {

        super(request);
        //将参数表，赋予给当前的Map以便于持有request中的参数
        this.params.putAll(request.getParameterMap());
    }

    /**
     * 重载构造方法
     */
    public ParameterRequestWrapper(HttpServletRequest request, Map<String, Object> extendParams) {
        super(request);
        //这里将扩展参数写入参数表
        if (Objects.isNull(extendParams)) {
            extendParams = HttpContextUtils.getParameterMapAll(request);
        }
        addAllParameters(extendParams);
    }

    /**
     * 在获取所有的参数名,必须重写此方法，否则对象中参数值映射不上
     *
     * @return
     */
    @Override
    public Enumeration<String> getParameterNames() {
        return new Vector(params.keySet()).elements();
    }

    /**
     * 重写getParameter方法
     *
     * @param name 参数名
     * @return 返回参数值
     */
    @Override
    public String getParameter(String name) {
        String[] values = params.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values[0];
    }

    @Override
    public String[] getParameterValues(String name) {
        String[] values = params.get(name);
        return values;
    }

    /**
     * 增加多个参数
     *
     * @param otherParams 增加的多个参数
     */
    public void addAllParameters(Map<String, Object> otherParams) {
        for (Map.Entry<String, Object> entry : otherParams.entrySet()) {
            addParameter(entry.getKey(), entry.getValue());
        }
    }

    /**
     * 增加参数
     *
     * @param name  参数名
     * @param value 参数值
     */
    public void addParameter(String name, Object value) {
        if (value != null) {
            if (value instanceof String[]) {
                params.put(name, (String[]) value);
            } else if (value instanceof String) {
                params.put(name, new String[]{(String) value});
            } else {
                params.put(name, new String[]{String.valueOf(value)});
            }
        }
    }

    @Override
    public StringBuffer getRequestURL() {
        StringBuffer originalUrl = super.getRequestURL();

        String[] split = originalUrl.toString().split("/");
        for (int i = split.length - 1; i >= 0; i--) {
            //替换可能存在的加密字段
            String vale = Sm2lib.decode(split[i]);
            split[i] = vale;
        }
        return new StringBuffer(Joiner.on("/").join(split));
    }

    @Override
    public String getRequestURI() {
        String originalUrl = super.getRequestURI();

        String[] split = originalUrl.split("/");
        for (int i = split.length - 1; i >= 0; i--) {
            //替换可能存在的加密字段
            String vale = Sm2lib.decode(split[i]);
            split[i] = vale;
        }
        originalUrl = Joiner.on("/").join(split);
        return originalUrl;
    }

    @Override
    public String getServletPath() {
        String originalUrl = super.getServletPath();
        String[] split = originalUrl.split("/");
        for (int i = split.length - 1; i >= 0; i--) {
            //替换可能存在的加密字段
            String vale = Sm2lib.decode(split[i]);
            split[i] = vale;
        }
        originalUrl = Joiner.on("/").join(split);
        return originalUrl;
    }

    @Override
    public ServletInputStream getInputStream() {
        StringBuilder sb = new StringBuilder();
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {
            inputStream = super.getInputStream();
            reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));

            String line = "";

            while ((line = reader.readLine()) != null) {
                //入参body如果存在加密则进行解密
                line = Sm2lib.decode(line);
                sb.append(line);
            }

            if (StringUtils.isEmpty(sb)) {
                return null;
            }

            final ByteArrayInputStream finalByteArrayInputStream = new ByteArrayInputStream(sb.toString().getBytes(StandardCharsets.UTF_8));
            ServletInputStream servletInputStream = new ServletInputStream() {
                @Override
                public boolean isFinished() {
                    return false;
                }

                @Override
                public boolean isReady() {
                    return false;
                }

                @Override
                public void setReadListener(ReadListener readListener) {
                }

                @Override
                public int read() throws IOException {
                    return finalByteArrayInputStream.read();
                }
            };

            return servletInputStream;
        } catch (IOException e) {
            throw new BizException(ErrorEnum.SERVICE_ERROR, "request解析Body请求信息异常");
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException e) {
                log.error("关闭流失败", e); // 只记录，不抛出
            }
        }
    }

    public String getInput() {
        StringBuilder sb = new StringBuilder();
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {
            inputStream = super.getInputStream();
            reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            String line = "";

            while ((line = reader.readLine()) != null) {
                //入参body如果存在加密则进行解密
                line = Sm2lib.decode(line);
                sb.append(line);
            }

            if (StringUtils.isEmpty(sb)) {
                return null;
            }
            return sb.toString();

        } catch (IOException e) {
            throw new BizException(ErrorEnum.SERVICE_ERROR, "request解析Body请求信息异常");
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException e) {
                log.error("关闭流失败", e); // 只记录，不抛出
            }
        }
    }
}