package cn.gtmap.hlw.core.util.http;

import cn.gtmap.hlw.core.util.encryption.sm2.Sm2lib;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.HandlerMapping;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.Map;
import java.util.Set;

/**
 * @Description http获取请求参数工具类
 * @Author admin
 * @Date 2022/12/28 10:51
 * @ClassName HttpContextUtils
 * @Version
 **/
public class HttpContextUtils {

    private static final Logger logger = LoggerFactory.getLogger(HttpContextUtils.class);

    /**
     * @return java.util.Map<java.lang.String, java.lang.String>
     * @Description 获取query参数
     * @Author admin
     * @Date 2022/12/28 13:54
     * @Param [request]
     **/
    public static Map<String, Object> getParameterMapAll(HttpServletRequest request) {
        Enumeration<String> parameters = request.getParameterNames();

        Map<String, Object> params = Maps.newHashMap();
        while (parameters.hasMoreElements()) {
            String parameter = parameters.nextElement();
            //入参如果存在加密情况则进行解密
            String value = Sm2lib.decode(parameter);

            params.put(parameter, value);
        }
        return params;
    }

    /**
     * @return java.lang.String
     * @Description 获取请求Body
     * @Author admin
     * @Date 2022/12/28 13:44
     * @Param [request]
     **/
    public static String getBodyString(ServletRequest request) {
        StringBuilder sb = new StringBuilder();
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {
            inputStream = request.getInputStream();
            reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            String line = "";
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
        } catch (IOException e) {
            logger.error("request解析Body请求信息异常:", e);
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }

                if (reader != null) {
                    reader.close();
                }
            } catch (IOException e) {
                logger.error("request解析Body请求信息异常:", e);
            }
        }
        return sb.toString();
    }

    /**
     * @return java.lang.String
     * @Description 获取请求Path路径上的参数
     * @Author admin
     * @Date 2022/12/28 13:43
     * @Param [request]
     **/
    public static Map<String, String> getPathString(HttpServletRequest request) {
        Map<String, String> params = Maps.newHashMap();

        Map<String, String> pathVars = (Map<String, String>) request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
        if (!pathVars.isEmpty()) {
            Set<Map.Entry<String, String>> entrySet = pathVars.entrySet();
            for (Map.Entry<String, String> entry : entrySet) {
                params.put(entry.getKey(), entry.getValue());
            }
        }
        return params;
    }

    /**
     * @param request
     * @return
     * @author admin
     * @Date 2022/12/21 9:53
     * @description 对接口请求参数进行组装 1.url参数 2.接口入参 3.获取RequestBody参数
     */
    public static String getParams(HttpServletRequest request) {
        StringBuilder sb = new StringBuilder();

        //获取接口入参
        Map<String, Object> parameterMap = HttpContextUtils.getParameterMapAll(request);
        if (!parameterMap.isEmpty()) {
            sb.append(parameterMap);
        }

        //获取url中的参数
        Map<String, String> pathMap = HttpContextUtils.getPathString(request);
        if (!pathMap.isEmpty()) {
            sb.append(pathMap);
        }

        //获取RequestBody参数
        String requestBody = HttpContextUtils.getBodyString(request);
        if (StringUtils.isNotEmpty(requestBody)) {
            sb.append(requestBody);
        }

        String str = sb.toString().replaceAll(" ", "");

        //入参过长，进行截取
        return str;
    }
}
