package cn.gtmap.hlw.core.util.file;

import cn.gtmap.estateplat.register.common.util.PublicUtil;
import cn.gtmap.hlw.core.dto.ftp.FtpDeleteParamDTO;
import cn.gtmap.hlw.core.dto.ftp.FtpDownloadParamDTO;
import cn.gtmap.hlw.core.dto.ftp.FtpInitParamDTO;
import cn.gtmap.hlw.core.dto.ftp.FtpUploadParamDTO;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPReply;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.MalformedURLException;

/**
 * 上传/下载ftp服务
 *
 * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
 * @see [相关类/方法]（可选）
 * @since [产品/模块版本] （可选）
 */
public class FtpUtils {

    private static final Logger logger = LoggerFactory.getLogger(FtpUtils.class);

    /**
     * 本地字符编码
     */
    private static String LOCAL_CHARSET = "GBK";

    /**
     * FTP协议里面，规定文件名编码为iso-8859-1
     */
    private static String SERVER_CHARSET = "ISO-8859-1";

    private static final String FILE_CENTER = "/fileCenter";

    private static FTPClient ftpClient = null;

    /**
     * @param
     * @return
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     * @description 初始化ftp服务器
     */
    private static void initFtpClient(FtpInitParamDTO paramDTO) {
        ftpClient = new FTPClient();
        logger.info("initFtpClient.paramDTO:{}", JSONObject.toJSONString(paramDTO));
        String encoding = paramDTO.getServerEncoding();
        String hostname = paramDTO.getServerHost();
        String port = paramDTO.getServerPort();
        String username = paramDTO.getServerUsername();
        String password = paramDTO.getServerPassword();
        ftpClient.setControlEncoding(StringUtils.isNotBlank(encoding) ? encoding : "utf-8");
        try {
            logger.info("connecting...ftp服务器");
            //连接ftp服务器
            ftpClient.connect(hostname, Integer.parseInt(port));
            //登录ftp服务器
            ftpClient.login(username, password);
            //是否成功登录服务器
            int replyCode = ftpClient.getReplyCode();
            if (!FTPReply.isPositiveCompletion(replyCode)) {
                logger.info("connect failed...ftp服务器");
            } else {
                logger.info("connect successful...ftp服务器");
            }
            if (FTPReply.isPositiveCompletion(ftpClient.sendCommand("OPTS UTF8", "ON"))) {
                LOCAL_CHARSET = "UTF-8";
            }
            logger.info("initFtpClient.LOCAL_CHARSET:" + LOCAL_CHARSET);
            ftpClient.setControlEncoding(LOCAL_CHARSET);
        } catch (MalformedURLException e) {
            logger.error("初始化异常，url格式不正确", e);
        } catch (IOException e) {
            logger.error("初始化IO异常", e);
        }
    }

    /**
     * 上传文件
     *
     * @param uploadParamDTO
     * @param paramDTO
     * @return
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     */
    public synchronized static boolean uploadFile(FtpUploadParamDTO uploadParamDTO, FtpInitParamDTO paramDTO) {
        boolean flag = false;
        String path = uploadParamDTO.getPath();
        String originPath = uploadParamDTO.getOriginPath();
        String fileName = uploadParamDTO.getFileName();
        if (StringUtils.isNoneBlank(path, originPath, fileName)) {
            String encoding = paramDTO.getServerEncoding();
            path = path.startsWith("/") ? path : "/" + path;
            InputStream inputStream = null;
            try {
                inputStream = new FileInputStream(new File(originPath));
                initFtpClient(paramDTO);
                ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
                createDirectory(FILE_CENTER + path);
                ftpClient.makeDirectory(path);
                ftpClient.enterLocalPassiveMode();
                ftpClient.changeWorkingDirectory(path);
                logger.info("开始保存文件");
                String encodedFileName = fileName;
                if (StringUtils.isNotBlank(encoding)) {
                    encodedFileName = new String(fileName.getBytes(LOCAL_CHARSET), encoding);
                }
                logger.info("fileName：" + fileName + ",encodedFileName：{}" + encodedFileName);
                ftpClient.storeFile(encodedFileName, inputStream);
                logger.info("结束保存文件");
                inputStream.close();
                ftpClient.logout();
                logger.info("ftp上传文件成功:path:{},fileName:{}", path, fileName);
                flag = true;
            } catch (Exception e) {
                logger.error("ftp上传文件失败:path:{},fileName:{}:error:{}", path, fileName, e);
            } finally {
                if (ftpClient.isConnected()) {
                    try {
                        ftpClient.disconnect();
                    } catch (IOException e) {
                        logger.error("ftp上传断开连接异常:{}", e);
                    }
                }
                if (null != inputStream) {
                    try {
                        inputStream.close();
                    } catch (IOException e) {
                        logger.error("ftp上传流关闭异常:{}", e);
                    }
                }
            }
        }
        return flag;
    }

    /**
     * 上传文件
     *
     * @param uploadParamDTO
     * @param inputStream
     * @param paramDTO
     * @return
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     */
    public static boolean uploadFile(FtpUploadParamDTO uploadParamDTO, InputStream inputStream, FtpInitParamDTO paramDTO) {
        boolean flag = false;
        String path = uploadParamDTO.getPath();
        String fileName = uploadParamDTO.getFileName();
        if (null != inputStream && StringUtils.isNoneBlank(path)) {
            path = path.startsWith("/") ? path : "/" + path;
            //统一对文件名作处理，因为中文不友好
            if (StringUtils.isBlank(fileName)) {
                fileName = PublicUtil.getBusinessOrder();
            }
            try {
                initFtpClient(paramDTO);
                ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
                createDirectory(FILE_CENTER + path);
                ftpClient.makeDirectory(path);
                ftpClient.changeWorkingDirectory(path);
                ftpClient.storeFile(fileName, inputStream);
                inputStream.close();
                ftpClient.logout();
                flag = true;
                logger.info("ftp上传文件成功");
            } catch (Exception e) {
                logger.error("ftp上传文件失败", e);
            } finally {
                if (ftpClient.isConnected()) {
                    try {
                        ftpClient.disconnect();
                    } catch (IOException e) {
                        logger.error("ftp连接关闭失败", e);
                    }
                }
                try {
                    inputStream.close();
                } catch (IOException e) {
                    logger.error("ftp上传文件流关闭失败", e);
                }
            }
        }
        return flag;
    }

    /**
     * @param directory
     * @return
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     * 改变目录路径
     */
    private static boolean changeWorkingDirectory(String directory) {
        boolean flag = true;
        try {
            flag = ftpClient.changeWorkingDirectory(directory);
            if (flag) {
                logger.info("进入文件夹" + directory + " 成功！");
            } else {
                logger.info("进入文件夹" + directory + " 失败！");
            }
        } catch (IOException e) {
            logger.error("ftp改变目录路径失败", e);
        }
        return flag;
    }

    /**
     * @param remote
     * @return
     * @throws IOException
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     * 创建多层目录文件，如果有ftp服务器已存在该文件，则不创建，如果无，则创建
     */
    private static boolean createDirectory(String remote) throws IOException {
        logger.info("remote:" + remote);
        boolean success = true;
        String directory = remote + "/";
        //如果远程目录不存在，则递归创建远程服务器目录
        if (!"/".equalsIgnoreCase(directory) && !changeWorkingDirectory(directory)) {
            int start = 0;
            if (directory.startsWith("/")) {
                start = 1;
            }
            int end = directory.indexOf("/", start);
            StringBuilder path = new StringBuilder();
            while (true) {
                String subDirectory = new String(remote.substring(start, end).getBytes(LOCAL_CHARSET), SERVER_CHARSET);
                path.append("/").append(subDirectory);
                if (!existFile(path.toString())) {
                    if (makeDirectory(subDirectory)) {
                        changeWorkingDirectory(subDirectory);
                        logger.info("创建目录[" + subDirectory + "]成功");
                    } else {
                        logger.info("创建目录[" + subDirectory + "]失败");
                        changeWorkingDirectory(subDirectory);
                    }
                } else {
                    changeWorkingDirectory(subDirectory);
                }
                start = end + 1;
                end = directory.indexOf("/", start);
                // 检查所有目录是否创建完毕
                if (end <= start) {
                    break;
                }
            }
        }
        return success;
    }

    /**
     * @param path
     * @return
     * @throws IOException
     * @author <a href="mailto:liwenwu@gtmap.cn">liwenwu</a>
     * 判断ftp服务器文件是否存在
     */
    private static boolean existFile(String path) throws IOException {
        boolean flag = false;
        //告诉ftpClient开启一个端口用来传输数据。
        ftpClient.enterLocalPassiveMode();
        FTPFile[] ftpFileArr = ftpClient.listFiles(path);
        if (ftpFileArr.length > 0) {
            flag = true;
        }
        return flag;
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:liwenwu@gtmap.cn">liwenwu</a>
     * @version 2.0,
     * @description 创建目录
     */
    private static boolean makeDirectory(String dir) {
        boolean flag = true;
        try {
            flag = ftpClient.makeDirectory(dir);
            if (flag) {
                logger.info("创建文件夹" + dir + " 成功！");
            } else {
                logger.info("创建文件夹" + dir + " 失败！");
            }
        } catch (Exception e) {
            logger.error("ftp创建文件夹失败:", e);
        }
        return flag;
    }

    /**
     * 下载文件
     *
     * @param paramDTO
     * @return
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     */
    public static boolean downloadFile(FtpDownloadParamDTO downloadParamDTO, FtpInitParamDTO paramDTO) {
        boolean flag = false;
        String path = downloadParamDTO.getPath();
        String filename = downloadParamDTO.getFilename();
        String localPath = downloadParamDTO.getLocalPath();
        if (StringUtils.isNoneBlank(path, filename, localPath)) {
            OutputStream os = null;
            try {
                initFtpClient(paramDTO);
                //切换FTP目录
                ftpClient.changeWorkingDirectory(path);
                FTPFile[] ftpFiles = ftpClient.listFiles();
                for (FTPFile file : ftpFiles) {
                    if (filename.equalsIgnoreCase(file.getName())) {
                        File localFile = new File(localPath + "/" + file.getName());
                        os = new FileOutputStream(localFile);
                        ftpClient.retrieveFile(file.getName(), os);
                        os.close();
                    }
                }
                ftpClient.logout();
                flag = true;
                logger.info("下载文件成功");
            } catch (Exception e) {
                logger.error("下载文件失败", e);
            } finally {
                if (ftpClient.isConnected()) {
                    try {
                        ftpClient.disconnect();
                    } catch (IOException e) {
                        logger.error("ftp下载文件连接关闭失败", e);
                    }
                }
                if (null != os) {
                    try {
                        os.close();
                    } catch (IOException e) {
                        logger.error("ftp下载文件流关闭失败", e);
                    }
                }
            }
        }
        return flag;
    }

    /**
     * 删除文件
     *
     * @param deleteParamDTO
     * @param paramDTO
     * @return
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     */
    public static boolean deleteFile(FtpDeleteParamDTO deleteParamDTO, FtpInitParamDTO paramDTO) {
        boolean flag = false;
        String pathname = deleteParamDTO.getPathname();
        String filename = deleteParamDTO.getFilename();
        if (StringUtils.isNoneBlank(pathname, filename)) {
            try {
                initFtpClient(paramDTO);
                //切换FTP目录
                ftpClient.changeWorkingDirectory(pathname);
                ftpClient.dele(filename);
                ftpClient.logout();
                flag = true;
                logger.info("删除文件成功");
            } catch (Exception e) {
                logger.error("ftp删除文件失败", e);
            } finally {
                if (ftpClient.isConnected()) {
                    try {
                        ftpClient.disconnect();
                    } catch (IOException e) {
                        logger.error("ftp删除文件连接关闭失败", e);
                    }
                }
            }
        }
        return flag;
    }

    /**
     * @param fileName
     * @return
     * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
     * 获取文件后缀
     */
    public static String getFileSuffix(String fileName) {
        if (StringUtils.isNoneBlank(fileName) && StringUtils.contains(fileName, ".")) {
            fileName = fileName.substring(fileName.lastIndexOf("."));
        }
        return fileName;
    }
}
