package cn.gtmap.hlw.core.util.encryption.aes;

import cn.gtmap.hlw.core.constant.number.NumberConstant;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.FileInputStream;
import java.nio.charset.StandardCharsets;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.spec.KeySpec;
import java.util.*;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2023/6/19
 * @description
 */
public class AesUtil {
    private static Logger logger = LoggerFactory.getLogger(AesUtil.class);

    /**
     * AES加密(黑河政务网)
     *
     * @param sSrc
     * @return
     * @throws Exception
     */
    public static String encryptHh(String sSrc, String key) {

        try {
            String sKey = secureBytes(key);

            // 判断Key是否为16位
            if (sKey.length() != NumberConstant.INT_SIXTEEN) {
                sKey = secureBytes(sKey);
            }
            byte[] raw = sKey.getBytes(StandardCharsets.US_ASCII);
            SecretKeySpec skeySpec = new SecretKeySpec(raw, "AES");
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.ENCRYPT_MODE, skeySpec);
            byte[] encrypted = cipher.doFinal(sSrc.getBytes(StandardCharsets.UTF_8));
            return byte2hex(encrypted).toLowerCase();
        } catch (Exception e) {
            logger.error("AES加密异常：{}", e);
            return null;
        }
    }

    /**
     * AES解密(黑河政务网)
     *
     * @param sSrc
     * @return
     * @throws Exception
     */
    public static String decryptHh(String sSrc, String key) {
        try {
            String sKey = secureBytes(key);
            // 判断Key是否为16位
            if (sKey.length() != NumberConstant.INT_SIXTEEN) {
                sKey = secureBytes(sKey);
            }
            byte[] raw = sKey.getBytes(StandardCharsets.US_ASCII);
            SecretKeySpec skeySpec = new SecretKeySpec(raw, "AES");
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.DECRYPT_MODE, skeySpec);
            byte[] encrypted1 = hex2byte(sSrc);
            byte[] original = cipher.doFinal(encrypted1);
            return new String(original, StandardCharsets.UTF_8);
        } catch (Exception ex) {
            logger.error("AES解密异常：{}", ex);
        }
        return null;
    }

    public static byte[] hex2byte(String strhex) {
        if (strhex == null) {
            return null;
        }
        int l = strhex.length();
        if (l % 2 == 1) {
            return null;
        }
        byte[] b = new byte[l / 2];
        for (int i = 0; i != l / 2; i++) {
            b[i] = (byte) Integer.parseInt(strhex.substring(i * 2, i * 2 + 2),
                    NumberConstant.INT_SIXTEEN);
        }
        return b;
    }

    /**
     * @param b
     * @return
     */
    public static String byte2hex(byte[] b) {
        StringBuilder hs = new StringBuilder();
        String stmp = "";
        for (int n = 0; n < b.length; n++) {
            stmp = (Integer.toHexString(b[n] & 0XFF));
            if (stmp.length() == 1) {
                hs.append("0" + stmp);
            } else {
                hs.append(stmp);
            }
        }
        return hs.toString().toUpperCase();
    }

    /**
     * 密钥如超过16位，截至16位，不足16位，补/000至16位
     *
     * @param key
     * @return 新密钥
     */
    public static String secureBytes(String key) {
        //todo：StringBulid
        if (key.length() > NumberConstant.INT_SIXTEEN) {
            key = key.substring(0, NumberConstant.INT_SIXTEEN);
        } else if (key.length() < NumberConstant.INT_SIXTEEN) {
            for (int i = (key.length() - 1); i < NumberConstant.INT_FIFTEEN; i++) {
                key += "\000";
            }
        }
        return key;
    }

    /**
     * 使用密钥进行加密
     *
     * @param sSrc
     * @param keyStr
     * @return
     * @throws Exception
     */
    public static String encrypt(String sSrc, String keyStr) {
        try {
            byte[] key = keyStr.getBytes(StandardCharsets.UTF_8);
            SecretKeySpec skeySpec = new SecretKeySpec(key, "AES");
            // "算法/模式/补码方式"
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.ENCRYPT_MODE, skeySpec);
            byte[] encrypted = cipher.doFinal(sSrc.getBytes(StandardCharsets.UTF_8));
            return Base64.encodeBase64String(encrypted);
        } catch (Exception e) {
            logger.error("AesUtil类encrypt方法异常，异常原因：{}", e);
        }
        return null;
    }


    /**
     * @param
     * @return
     * @author <a href="mailto:liwenwu@gtmap.cn">liwenwu</a>
     * @version 2.0,
     * @description 使用密钥进行加密
     */
    public static String encryptNull(String sSrc, String keyStr) {
        if (StringUtils.isBlank(sSrc) || StringUtils.isBlank(keyStr)) {
            return null;
        } else {
            return encrypt(sSrc, keyStr);
        }
    }


    /**
     * 使用密钥进行解密
     *
     * @param sSrc
     * @param keyStr
     * @return
     * @throws Exception
     */
    public static String decrypt(String sSrc, String keyStr) {
        try {
            if (StringUtils.isNotBlank(sSrc)) {
                byte[] key = keyStr.getBytes(StandardCharsets.UTF_8);
                SecretKeySpec skeySpec = new SecretKeySpec(key, "AES");
                Cipher cipher = Cipher.getInstance("AES");
                cipher.init(Cipher.DECRYPT_MODE, skeySpec);
                // 先用base64解码
                byte[] encrypted = Base64.decodeBase64(sSrc);
                byte[] original = cipher.doFinal(encrypted);
                return new String(original, StandardCharsets.UTF_8);
            }
        } catch (Exception e) {
        }
        return null;
    }


    /**
     * @param
     * @return
     * @auto <a href="mailto:zhouwanqing@gtmap.cn">zhouwanqing</a>
     * @description 解码错误直接返回原内容
     **/
    public static String decryptNull(String sSrc, String keyStr) {
        String result = sSrc;
        if (StringUtils.isNotBlank(sSrc) && StringUtils.isNotBlank(keyStr)) {
            result = decrypt(sSrc, keyStr);
            if (StringUtils.isBlank(result)) {
                result = sSrc;
            }
        }
        return result;
    }

    public static String decryptByPage(String key, String value) {
        return decrypt(key, key, key, value);
    }

    public static String decrypt(String salt, String iv, String passphrase, String ciphertext) {
        try {
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
            SecretKey key = generateKey(salt, passphrase);
            byte[] decrypted = doFinal(Cipher.DECRYPT_MODE, key, iv, base64(ciphertext), cipher);
            return new String(decrypted, StandardCharsets.UTF_8);
        } catch (Exception e) {
            logger.error("AesUtil类decrypt方法异常，异常原因：{}", e);
        }
        return null;
    }

    private static SecretKey generateKey(String salt, String passphrase) {
        try {
            SecretKeyFactory factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
            KeySpec spec = new PBEKeySpec(passphrase.toCharArray(), hex(salt), 1000, 128);
            SecretKey key = new SecretKeySpec(factory.generateSecret(spec).getEncoded(), "AES");
            return key;
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return null;
    }

    private static byte[] doFinal(int encryptMode, SecretKey key, String iv, byte[] bytes, Cipher cipher) {
        try {
            cipher.init(encryptMode, key, new IvParameterSpec(hex(iv)));
            return cipher.doFinal(bytes);
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return null;
    }


    public static byte[] hex(String str) {
        try {
            return Hex.decodeHex(str.toCharArray());
        } catch (DecoderException e) {
            throw new IllegalStateException(e);
        }
    }

    public static byte[] base64(String str) {
        return Base64.decodeBase64(str);
    }

    /**
     * @param
     * @return
     * @auto <a href="mailto:zhouwanqing@gtmap.cn">zhouwanqing</a>
     * @description cbc模式
     **/
    public static String decryptByCbc(String content, String key) {
        String result = null;
        try {
            byte[] keyByte = key.getBytes(StandardCharsets.UTF_8);
            byte[] contentByte = content.getBytes(StandardCharsets.UTF_8);
            SecretKeySpec secretKeySpec = new SecretKeySpec(keyByte, "AES");
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
            IvParameterSpec ivParameterSpec = new IvParameterSpec(keyByte);
            cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, ivParameterSpec);
            byte[] resultByte = cipher.doFinal(contentByte);
            if (resultByte != null && resultByte.length > 0) {
                result = new String(resultByte, StandardCharsets.UTF_8);
            }
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return result;
    }


    /**
     * 构建密钥字节码
     *
     * @param keyStr
     * @return
     * @throws Exception
     */
    private static byte[] generalKey(String keyStr) {
        byte[] bytes = new byte[0];
        MessageDigest md = null;
        try {
            bytes = keyStr.getBytes(StandardCharsets.UTF_8);
            md = MessageDigest.getInstance("SHA-256");
            md.update(bytes);
            return md.digest();
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return null;
    }

    /**
     * 构建加解密向量字节码
     *
     * @param keyStr
     * @return
     * @throws Exception
     */
    private static byte[] generalIv(String keyStr) {
        byte[] bytes = new byte[0];
        MessageDigest md = null;
        try {
            bytes = keyStr.getBytes(StandardCharsets.UTF_8);
            md = MessageDigest.getInstance("MD5");
            md.update(bytes);
            return md.digest();
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return null;
    }


    /**
     * @param
     * @return
     * @auto <a href="mailto:zhouwanqing@gtmap.cn">zhouwanqing</a>
     * @description 随机生成密钥
     **/
    public static String getRandomKey() {
        String key = "";
        try {
            KeyGenerator keyGenerator = KeyGenerator.getInstance("AES");
            keyGenerator.init(128);
            SecretKey sk = keyGenerator.generateKey();
            byte[] b = sk.getEncoded();
            key = byteToHexString(b);
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return key;
    }


    public static String byteToHexString(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            String strHex = Integer.toHexString(bytes[i]);
            if (strHex.length() > 3) {
                sb.append(strHex.substring(6));
            } else {
                if (strHex.length() < 2) {
                    sb.append("0" + strHex);
                } else {
                    sb.append(strHex);
                }
            }
        }
        return sb.toString();
    }


    public static String generateParamStr(Map<String, String> params) {
        List<String> paramList = new ArrayList<>();
        for (String key : params.keySet()) {
            if ("mac".equals(key)) {
                continue;
            }
            String val = params.get(key);
            paramList.add(key + "=" + val);
        }

        StringBuilder sb = new StringBuilder();

        if (!CollectionUtils.isEmpty(paramList)) {
            Collections.sort(paramList);
            sb.append(paramList.get(0));
            for (int i = 1; i < paramList.size(); i++) {
                sb.append("&").append(paramList.get(i));
            }
        }
        return sb.toString();
    }

//    public static String generateMac(Map<String, String> params, String certPath, String certPwd) {
//        try {
//            PrivateKey privKey = readPrivateKey(certPath, certPwd);
//            Signature signature = Signature.getInstance("SHA1WithRSA");
//            signature.initSign(privKey);
//            signature.update(generateParamStr(params).getBytes(StandardCharsets.UTF_8));
//            byte[] signed = signature.sign();
//            // 计算 base64encode(signed)，无换行。如无法使用，请自行替换为其它 BASE64类库。
//            @SuppressWarnings("restriction")
//            String mac = new BASE64Encoder().encode(signed).replaceAll(System.getProperty("line.separator"), "");
//            return mac;
//        } catch (Exception e) {
//            logger.error("异常信息：", e);
//            // 读取证书等出错
//            return "SIGNATURE_RSA_CERT_ERROR";
//        }
//    }


    private static PrivateKey readPrivateKey(String strPfx, String strPassword) {
        try {
            KeyStore ks = KeyStore.getInstance("PKCS12");
            FileInputStream fis = new FileInputStream(strPfx);

            char[] nPassword = null;
            if ((strPassword == null) || "".equals(strPassword.trim())) {
                nPassword = null;
            } else {
                nPassword = strPassword.toCharArray();
            }
            ks.load(fis, nPassword);
            //todo:finally 中关闭
            fis.close();

            Enumeration enumas = ks.aliases();
            String keyAlias = null;
            if (enumas.hasMoreElements()) {
                keyAlias = (String) enumas.nextElement();
            }
            PrivateKey prikey = (PrivateKey) ks.getKey(keyAlias, nPassword);
            return prikey;
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return null;
    }

    /**
     * 将NumberConstant.INT_SIXTEEN进制转换为二进制
     *
     * @param hexStr
     * @return
     */
    public static byte[] parseHexStr2Byte(String hexStr) {
        if (hexStr.length() < 1) {
            return null;
        }
        byte[] result = new byte[hexStr.length() / 2];
        for (int i = 0; i < hexStr.length() / 2; i++) {
            int high = Integer.parseInt(hexStr.substring(i * 2, i * 2 + 1), NumberConstant.INT_SIXTEEN);
            int low = Integer.parseInt(hexStr.substring(i * 2 + 1, i * 2 + 2), NumberConstant.INT_SIXTEEN);
            result[i] = (byte) (high * NumberConstant.INT_SIXTEEN + low);
        }
        return result;
    }

    /**
     * 解密
     *
     * @param content  待解密内容
     * @param password 解密密钥
     * @return
     */
    public static byte[] decrypt(byte[] content, String password) {
        try {
            KeyGenerator kgen = KeyGenerator.getInstance("AES");
            kgen.init(128, new SecureRandom(password.getBytes()));
            SecretKey secretKey = kgen.generateKey();
            byte[] enCodeFormat = secretKey.getEncoded();
            SecretKeySpec key = new SecretKeySpec(enCodeFormat, "AES");
            // 创建密码器
            Cipher cipher = Cipher.getInstance("AES");
            // 初始化
            cipher.init(Cipher.DECRYPT_MODE, key);
            // 加密
            return cipher.doFinal(content);
        } catch (Exception e) {
            logger.error("异常信息：", e);
        }
        return null;
    }

    /**
     * AES解密 用于数据库储存
     *
     * @param sSrc
     * @param key
     * @return
     * @throws Exception
     */
    public static String decryptHh(byte[] sSrc, String key) {
        try {
            String sKey = secureBytes(key);
            if (StringUtils.isNotEmpty(sKey)) {
                // 判断Key是否为16位
                if (sKey.length() != NumberConstant.INT_SIXTEEN) {
                    sKey = secureBytes(sKey);
                }
                byte[] raw = sKey.getBytes(StandardCharsets.US_ASCII);
                SecretKeySpec skeySpec = new SecretKeySpec(raw, "AES");
                Cipher cipher = Cipher.getInstance("AES");
                cipher.init(Cipher.DECRYPT_MODE, skeySpec);
                byte[] original = cipher.doFinal(sSrc);
                return new String(original, StandardCharsets.UTF_8);
            }
        } catch (Exception ex) {
            logger.error("异常信息：", ex);
        }
        return "";
    }
}
