package cn.gtmap.hlw.core.exception;

import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;

/**
 * 业务断言工具类
 * - 替代Appache和Spring的Assert
 * - 用于 1、客户端请求参数复杂断言，适用于Bean中的参数校验
 * - 用于 2、 服务器内部业务断言，适用于Controller和Service、
 * -   校验失败时, 抛出异常{@link BizException}
 * -   校验失败表示服务器内部错误, 返回HTTP 500
 *
 * @author admin
 * @version V1.0
 * @Date 2022/12/29 17:11
 */
public class BaseAssert {

    /**
     * 断言非null
     * 若是List size不能=0
     *
     * @param t
     * @param code 错误码
     * @param msg  需要打印日志msg
     * @throws BizException
     */
    public static <T> void notNull(T t, IError code, Supplier<String> msg) {
        boolean status = ((t instanceof List) && CollectionUtils.isEmpty((List) t));
        if (null == t || status) {
            throw new BizException(code, Objects.nonNull(msg) ? msg.get() : "");
        }
    }

    public static <T> void notNull(T t, IError code) {
        boolean status = ((t instanceof List) && CollectionUtils.isEmpty((List) t));
        if (null == t || status) {
            throw new BizException(code);
        }
    }


    /**
     * 断言字符串非空
     * 该断言会会判断字符串是否为空，只包含空白符也会判断为空
     *
     * @param s    断言字符串
     * @param code 具体业务错误提示
     * @param msg  需要打印日志msg
     * @throws BizException
     */
    public static void notBlank(String s, IError code, Supplier<String> msg) {
        if (StringUtils.isBlank(s)) {
            throw new BizException(code, Objects.nonNull(msg) ? msg.get() : "");
        }
    }

    public static void notBlank(String s, IError code) {
        if (StringUtils.isBlank(s)) {
            throw new BizException(code);
        }
    }

    /**
     * 断言Object非null && 其中的某个或多个字符串非空
     * 该断言会会判断字符串是否为空，只包含空白符也会判断为空
     *
     * @param t
     * @param s1   断言字符串1
     * @param s2   断言字符串2
     * @param code 具体业务错误提示
     * @param msg
     * @throws BizException
     */
    public static <T> void notNullAndBlank(T t, String s1, String s2, IError code, Supplier<String> msg) {
        if (null == t || StringUtils.isBlank(s1) || StringUtils.isBlank(s2)) {
            throw new BizException(code, Objects.nonNull(msg) ? msg.get() : "");
        }
    }

    /**
     * 断言表达式结果为真
     *
     * @param expression 表达式
     * @param code
     * @param msg        () -> 'xxxx' + xx + xxx ;
     * @throws BizException
     */
    public static void isTrue(boolean expression, IError code, Supplier<String> msg) {
        if (!expression) {
            throw new BizException(code, Objects.nonNull(msg) ? msg.get() : "");
        }
    }

    /**
     * 断言表达式结果为真
     *
     * @param expression
     * @param code
     */
    public static void isTrue(boolean expression, IError code) {
        if (!expression) {
            throw new BizException(code);
        }
    }

    /**
     * 根错误code抛出异常
     *
     * @param code
     * @param msg
     */
    public static void throwErrorByCode(IError code, Supplier<String> msg) {
        throw new BizException(code, Objects.nonNull(msg) ? msg.get() : "");
    }
}
