package cn.gtmap.hlw.core.aop;

import cn.gtmap.hlw.core.annotation.UserAuthFilter;
import cn.gtmap.hlw.core.base.UserInfo;
import cn.gtmap.hlw.core.dto.filter.UserAuthFilterResultDTO;
import cn.gtmap.hlw.core.util.session.SessionUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.util.*;

/**
 * 查询结果过滤
 *
 * @author <a href="mailto:chengong@gtmap.cn">chengong</a>
 * @see [相关类/方法]（可选）
 * @since [产品/模块版本] （可选）
 */
@Aspect
@Component
@Slf4j
public class FilterResultAspect {
    //包装类
    private static final Set<Class<?>> WRAPPERTYPES = new HashSet<>(Arrays.asList(
            java.lang.Byte.class, java.lang.Short.class, java.lang.Integer.class,
            java.lang.Long.class, java.lang.Float.class, java.lang.Double.class,
            java.lang.Character.class, java.lang.Boolean.class,String.class
    ));

    @AfterReturning(pointcut = "@annotation(cn.gtmap.hlw.core.annotation.FilterResult)", returning = "result")
    public void filter(JoinPoint joinPoint, Object result) {
        if (result instanceof List) {
            List<?> resultList = (List<?>) result;
            if(CollectionUtils.isNotEmpty(resultList)) {
                resultList.removeIf(item -> !filterResultResolver(item));
            }
        }
    }

    /**
     * userAuthFilter注解执行逻辑
     * @param item
     * @param field
     * @return
     * @throws IllegalAccessException
     */
    private UserAuthFilterResultDTO userAuthFilter(Object item, Field field) throws IllegalAccessException{
        boolean isNeedFilterFlag = false;
        boolean isChekFlag = true;
        UserInfo user = SessionUtil.getUser();
        if (user != null && field.isAnnotationPresent(UserAuthFilter.class)) {
            UserAuthFilter userAuthFilter = field.getAnnotation(UserAuthFilter.class);
            //有默认值，2自然人默认过滤
            List<String> roles = Lists.newArrayList(userAuthFilter.roles().split(","));
            String value = String.valueOf(field.get(item));
            if (CollectionUtils.isNotEmpty(roles) && roles.contains(user.getRoleId())) {
                isNeedFilterFlag = true;
                isChekFlag = StringUtils.isNotBlank(value) && (StringUtils.equalsAny(value,user.getUserZjid(),user.getOrgTyxydm()));
            }
        }
        UserAuthFilterResultDTO userAuthFilterResultDTO = new UserAuthFilterResultDTO();
        userAuthFilterResultDTO.setIsNeedFilterFlag(isNeedFilterFlag);
        userAuthFilterResultDTO.setIsChekFlag(isChekFlag);
        return userAuthFilterResultDTO;
    }

    /**
     * 迭代实体类
     * @param item
     * @param visited
     * @return
     * @throws IllegalAccessException
     */
    private void shouldKeep(Object item, Set<Object> visited,List<UserAuthFilterResultDTO> userAuthlist)  throws IllegalAccessException{
        //校验过的结束校验，防止死循环
        if (item == null || visited.contains(item)) {
            return;
        }
        visited.add(item);
        Field[] fields = item.getClass().getDeclaredFields();
        for (Field field : fields) {
            field.setAccessible(true);
            //出参用户过滤，只返回当前用户
            UserAuthFilterResultDTO userAuthFilterResultDTO = userAuthFilter(item,field);
            userAuthlist.add(userAuthFilterResultDTO);
            //迭代实体类 集合处理
            if (Collection.class.isAssignableFrom(field.getType())) {
                Collection<?> collection = (Collection<?>) field.get(item);
                if (collection != null) {
                    for (Object element : collection) {
                        shouldKeep(element,visited,userAuthlist);
                    }
                }
            } else if(!field.getType().isPrimitive() && !WRAPPERTYPES.contains(field.getType())){
                //对象处理(非基本类型和包装类)，避免重复迭代
                Object nestedObject = field.get(item);
                shouldKeep(nestedObject,visited,userAuthlist);
            }
        }
    }

    private boolean filterResultResolver(Object item){
        boolean flag = true;
        List<UserAuthFilterResultDTO> userAuthlist = new ArrayList<>();
        try {
            shouldKeep(item, new HashSet<>(),userAuthlist);
        } catch (IllegalAccessException e) {
            log.error("Error during filtering: ", e);
            return false;
        }
        if(CollectionUtils.isNotEmpty(userAuthlist)){
            for (UserAuthFilterResultDTO userAuthFilterResultDTO : userAuthlist) {
                if(userAuthFilterResultDTO.getIsNeedFilterFlag()){
                    flag = false;
                }
                if(userAuthFilterResultDTO.getIsNeedFilterFlag() && userAuthFilterResultDTO.getIsChekFlag()){
                    //满足条件直接跳出
                    return true;
                }
            }
        }
        return flag;
    }
}
