package cn.gtmap.hlw.core.util.http;

import cn.gtmap.estateplat.register.common.util.PublicUtil;
import cn.gtmap.hlw.core.constant.Constants;
import cn.gtmap.hlw.core.enums.error.ErrorEnum;
import cn.gtmap.hlw.core.enums.web.HttpMethodEnum;
import cn.gtmap.hlw.core.exception.BizException;
import cn.gtmap.hlw.core.util.file.FileUtils;
import cn.gtmap.hlw.core.util.string.StringUtil;
import cn.hutool.core.codec.Base64;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.http.*;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.apache.http.ssl.TrustStrategy;
import org.apache.http.util.EntityUtils;
import org.slf4j.MDC;
import org.springframework.web.multipart.MultipartFile;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2023/6/8
 * @description
 */
@Slf4j
public class HttpUtil {
    private static PoolingHttpClientConnectionManager connMgr = new PoolingHttpClientConnectionManager();
    private static RequestConfig requestConfig;
    public static final String CHARSET = "UTF-8";

    public HttpUtil() {
    }

    static {
        connMgr.setMaxTotal(100);
        connMgr.setDefaultMaxPerRoute(connMgr.getMaxTotal());
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        configBuilder.setConnectTimeout(120000);
        configBuilder.setSocketTimeout(300000);
        configBuilder.setConnectionRequestTimeout(120000);
        requestConfig = configBuilder.build();
    }

    /**
     * @param apiUrl  请求地址
     * @param apiName 接口名称
     * @param json    json数据
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description post, 以 application/json 形式
     */
    public static String doPost(String apiUrl, String apiName, String json) {
        long start = System.currentTimeMillis();
        CloseableHttpClient httpClient = HttpClients.custom().setDefaultRequestConfig(requestConfig).build();
        String httpStr = null;
        HttpPost httpPost = new HttpPost(apiUrl);
        CloseableHttpResponse response = null;
        int statusCode = -999;
        String id = StringUtil.hex32();
        log.info("doPost.id:" + id);
        try {
            httpPost.setConfig(requestConfig);
            if (StringUtils.isNotBlank(json)) {
                StringEntity stringEntity = new StringEntity(json, CHARSET);
                //stringEntity.setContentEncoding(CHARSET);
                stringEntity.setContentType("application/json");
                httpPost.setEntity(stringEntity);
            }
            httpPost.addHeader(Constants.TRACE_ID, MDC.get(Constants.TRACE_ID));
            response = httpClient.execute(httpPost);
            HttpEntity entity = response.getEntity();
            statusCode = response.getStatusLine().getStatusCode();
            httpStr = EntityUtils.toString(entity, CHARSET);

        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil post error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil doPost error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            if (!StringUtils.equals(apiName, "es查询")) {
                if (apiUrl.contains("wwsq/tsWwsqFjxx") || apiUrl.contains("storages/multipart/upload")) {
                    log.info("{}:request to:{},param.length:{},response code:{},cost {} ms", new Object[]{id, apiUrl, json.length(), statusCode, end});
                } else {
                    log.info("{}:request to:{},param:{},response code:{},cost {} ms", new Object[]{id, apiUrl, json, statusCode, end});
                }
                if (apiUrl.contains("zzgx/zzxxxz")) {
                    log.info("result to {} : 出参过滤 {}", apiUrl, httpStr.length());
                } else {
                    log.info("{}:result:{}", new Object[]{id, httpStr});
                }
            }
        }
        return httpStr;
    }

    /**
     * @param apiUrl 请求地址
     * @param json   json数据
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description post, 以 application/json 形式
     */
    public static String doPost(String apiUrl, String apiName, String json, String contentType) {
        long start = System.currentTimeMillis();
        CloseableHttpClient httpClient = HttpClients.createDefault();
        String httpStr = null;
        HttpPost httpPost = new HttpPost(apiUrl);
        CloseableHttpResponse response = null;
        int statusCode = -999;
        try {
            httpPost.setConfig(requestConfig);
            if (StringUtils.isNotBlank(json)) {
                StringEntity stringEntity = new StringEntity(json, CHARSET);
                //stringEntity.setContentEncoding(CHARSET);
                stringEntity.setContentType(contentType);
                httpPost.setEntity(stringEntity);
            }
            response = httpClient.execute(httpPost);
            HttpEntity entity = response.getEntity();
            statusCode = response.getStatusLine().getStatusCode();
            httpStr = EntityUtils.toString(entity, CHARSET);
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil post error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil doPost error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, json, statusCode, end});
            log.info("result:{}", new Object[]{httpStr});

        }
        return httpStr;
    }

    /**
     * @param apiUrl    请求地址
     * @param apiName   接口名称
     * @param json      json数据
     * @param headerMap 请求头
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description 带请求头的Post方式
     */
    public static String doPost(String apiUrl, String apiName, String json, Map<String, String> headerMap) {
        long start = System.currentTimeMillis();
        CloseableHttpClient httpClient = HttpClients.createDefault();
        String httpStr = null;
        HttpPost httpPost = new HttpPost(apiUrl);
        CloseableHttpResponse response = null;
        int statusCode = -999;
        try {
            httpPost.setConfig(requestConfig);
            StringEntity stringEntity = new StringEntity(json, CHARSET);
            //stringEntity.setContentEncoding(CHARSET);
            //循环增加header
            if (headerMap != null) {
                Iterator headerIterator = headerMap.entrySet().iterator();
                while (headerIterator.hasNext()) {
                    Map.Entry<String, String> elem = (Map.Entry<String, String>) headerIterator.next();
                    httpPost.addHeader(elem.getKey(), elem.getValue());
                }
            }
            stringEntity.setContentType("application/json");
            httpPost.setEntity(stringEntity);
            response = httpClient.execute(httpPost);
            HttpEntity entity = response.getEntity();
            statusCode = response.getStatusLine().getStatusCode();
            httpStr = EntityUtils.toString(entity, CHARSET);
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil post error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil post error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, json, statusCode, end});
            log.info("result:{}", new Object[]{httpStr});
        }
        return httpStr;
    }

    public static String sendPostRequest(String url, List<MultipartFile> files, Map<String, String> params) throws URISyntaxException, IOException {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        HttpPost httpPost = new HttpPost(url);

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setCharset(StandardCharsets.UTF_8);
        //加上此行代码解决返回中文乱码问题
        builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);

        // 设置请求参数
        //  字节传输http请求头(application/json)
        ContentType contentType = ContentType.create("application/json", StandardCharsets.UTF_8);
        if (params != null && params.size() > 0) {
            for (Map.Entry<String, String> e : params.entrySet()) {
                String value = e.getValue();
                if (StringUtils.isNotBlank(value)) {
                    // 类似浏览器表单提交，对应input的name和value
                    builder.addTextBody(e.getKey(), value, contentType);
                }
            }
        }

        // 设置文件参数
        if (files != null && !files.isEmpty()) {
            builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
            for (MultipartFile file : files) {
                builder.addBinaryBody("file", file.getBytes(), ContentType.MULTIPART_FORM_DATA, file.getOriginalFilename());
            }
            HttpEntity multipart = builder.build();
            httpPost.setEntity(multipart);
        }
        HttpResponse response = httpClient.execute(httpPost);
        HttpEntity responseEntity = response.getEntity();
        String responseString = EntityUtils.toString(responseEntity, StandardCharsets.UTF_8);
        httpClient.close();
        return responseString;
    }


    /**
     * 使用httpclint 发送文件，如果不传输文件，直接设置fileParams=null，
     * 如果不设置请求头参数，直接设置headerParams=null，就可以进行普通参数的POST请求了
     *
     * @param url          请求路径
     * @param files        文件
     * @param otherParams  其他字符串参数
     * @param headerParams 请求头参数
     * @return
     */
    public static String uploadFile(String url, List<MultipartFile> files, Map<String, String> otherParams, Map<String, String> headerParams) {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        String result = "";
        try {
            HttpPost httpPost = new HttpPost(url);
            //设置请求头
            if (headerParams != null && headerParams.size() > 0) {
                for (Map.Entry<String, String> e : headerParams.entrySet()) {
                    String value = e.getValue();
                    String key = e.getKey();
                    if (StringUtils.isNotBlank(value)) {
                        httpPost.setHeader(key, value);
                    }
                }
            }
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setCharset(Charset.forName("utf-8"));
            /**
             * 加上此行代码解决返回中文乱码问题
             */
            builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
            /**
             * 文件传输http请求头(multipart/form-data)
             */
            for (MultipartFile file : files) {
                if (file != null) {
                    String fileName = file.getOriginalFilename();
                    builder.addBinaryBody("file", file.getInputStream(), ContentType.MULTIPART_FORM_DATA, fileName);
                }
            }
            /**
             * 字节传输http请求头(application/json)
             *  addTextBody类似浏览器表单提交，对应input的name和value
             */
            ContentType contentType = ContentType.create("application/json", Charset.forName("UTF-8"));
            if (otherParams != null && otherParams.size() > 0) {
                for (Map.Entry<String, String> e : otherParams.entrySet()) {
                    String value = e.getValue();
                    if (StringUtils.isNotBlank(value)) {
                        builder.addTextBody(e.getKey(), value, contentType);
                    }
                }
            }
            HttpEntity entity = builder.build();
            httpPost.setEntity(entity);
            HttpResponse response = httpClient.execute(httpPost);
            HttpEntity responseEntity = response.getEntity();
            if (responseEntity != null) {
                /**
                 * 将响应内容转换为字符串
                 */
                result = EntityUtils.toString(responseEntity, "UTF-8");
            }
        } catch (Exception e) {
            log.error("使用httpclint 发送文件异常：{}", e);
        } finally {
            if (httpClient != null) {
                try {
                    httpClient.close();
                } catch (IOException e) {
                    log.error("httpclint关闭异常：{}", e);
                }
            }
        }
        return result;
    }

    /**
     * @param apiUrl 请求地址
     * @param params 参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description x-www-form-urlencoded方式post请求
     */
    public static String doPost(String apiUrl, String apiName, Map<String, Object> params) {
        long start = System.currentTimeMillis();
        CloseableHttpResponse response = null;
        String httpStr = null;
        int statusCode = -999;
        //创建http实例
        CloseableHttpClient httpClient = HttpClients.createDefault();
        //创建httpPost实例
        HttpPost httpPost = new HttpPost(apiUrl);
        try {
            httpPost.setConfig(requestConfig);
            List<NameValuePair> pairList = new ArrayList();
            Iterator iterator = params.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, Object> entry = (Map.Entry) iterator.next();
                NameValuePair pair = new BasicNameValuePair((String) entry.getKey(), entry.getValue().toString());
                pairList.add(pair);
            }
            httpPost.setEntity(new UrlEncodedFormEntity(pairList, Charset.forName(CHARSET)));
            response = httpClient.execute(httpPost);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            httpStr = EntityUtils.toString(entity, CHARSET);
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil post error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil post error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(params), statusCode, end});
            log.info("result:{}", new Object[]{httpStr});
        }
        return httpStr;
    }

    /**
     * @param apiUrl 请求地址
     * @param params 参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description form-data方式post请求  返回base64
     */
    public static String doPostForBase64(String apiUrl, String apiName, Map<String, Object> params) {
        long start = System.currentTimeMillis();
        CloseableHttpResponse response = null;
        String httpStr = null;
        int statusCode = -999;
        //创建http实例
        CloseableHttpClient httpClient = HttpClients.createDefault();
        //创建httpPost实例
        HttpPost httpPost = new HttpPost(apiUrl);
        try {
            httpPost.setConfig(requestConfig);
            httpPost.setHeader("Content-Type", "multipart/form-data");
            List<NameValuePair> pairList = new ArrayList();
            Iterator iterator = params.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, Object> entry = (Map.Entry) iterator.next();
                NameValuePair pair = new BasicNameValuePair((String) entry.getKey(), entry.getValue().toString());
                pairList.add(pair);
            }
            httpPost.setEntity(new UrlEncodedFormEntity(pairList, Charset.forName(CHARSET)));
            response = httpClient.execute(httpPost);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                InputStream instream = entity.getContent();
                httpStr = Base64.encode(instream);
            }
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil post error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil post error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(params), statusCode, end});
            if (StringUtils.isNotBlank(httpStr)) {
                log.info("result.size:{}", new Object[]{httpStr.length()});
            }
        }
        return httpStr;
    }

    /**
     * @param apiUrl 请求地址
     * @param params 参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description x-www-form-urlencoded方式post请求
     */
    public static String doHttpsPost(String apiUrl, Map<String, Object> params, Map<String, String> header) {
        long start = System.currentTimeMillis();
        CloseableHttpResponse response = null;
        String httpStr = null;
        int statusCode = -999;
        SSLContext sslContext = null;
        try {
            //使用 loadTrustMaterial() 方法实现一个信任策略，信任所有证书
            sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustStrategy() {
                // 信任所有
                @Override
                public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    return true;
                }
            }).build();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        } catch (KeyManagementException e) {
            throw new RuntimeException(e);
        } catch (KeyStoreException e) {
            throw new RuntimeException(e);
        }
        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext, NoopHostnameVerifier.INSTANCE);
        //创建http实例
        CloseableHttpClient httpClient = HttpClients.custom().setSSLSocketFactory(sslsf).build();
        //创建httpPost实例
        HttpPost httpPost = new HttpPost(apiUrl);
        try {
            httpPost.setConfig(requestConfig);
            httpPost.setHeader("Content-Type", "application/x-www-form-urlencoded");
            // 设置头信息
            if (MapUtils.isNotEmpty(header)) {
                for (Map.Entry<String, String> entry : header.entrySet()) {
                    httpPost.addHeader(entry.getKey(), entry.getValue());
                }
            }
            List<NameValuePair> pairList = new ArrayList();
            Iterator iterator = params.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, Object> entry = (Map.Entry) iterator.next();
                NameValuePair pair = new BasicNameValuePair((String) entry.getKey(), entry.getValue().toString());
                pairList.add(pair);
            }
            httpPost.setEntity(new UrlEncodedFormEntity(pairList, Charset.forName(CHARSET)));
            response = httpClient.execute(httpPost);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            httpStr = EntityUtils.toString(entity, CHARSET);
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil post error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil post error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            if (apiUrl.contains("integrated/pictureInfo")) {
                log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, "入参太长，不输出", statusCode, end});
            } else {
                log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(params), statusCode, end});
            }
            log.info("result:{}", new Object[]{httpStr});
        }
        return httpStr;
    }

    /**
     * @param apiUrl 请求地址
     * @param params 参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description x-www-form-urlencoded方式post请求
     */
    public static String doHttpsGet(String apiUrl, Map<String, Object> params) {
        long start = System.currentTimeMillis();
        CloseableHttpResponse response = null;
        String httpStr = null;
        int statusCode = -999;
        SSLContext sslContext = null;
        try {
            sslContext = SSLContexts.custom().loadTrustMaterial(new TrustSelfSignedStrategy()).build();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        } catch (KeyManagementException e) {
            throw new RuntimeException(e);
        } catch (KeyStoreException e) {
            throw new RuntimeException(e);
        }
        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext, NoopHostnameVerifier.INSTANCE);
        //创建http实例
        CloseableHttpClient httpClient = HttpClients.custom().setSSLSocketFactory(sslsf).build();
        //创建httpPost实例
        try {
            List<NameValuePair> pairList = new ArrayList();
            Iterator iterator = params.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, Object> entry = (Map.Entry) iterator.next();
                NameValuePair pair = new BasicNameValuePair((String) entry.getKey(), entry.getValue().toString());
                pairList.add(pair);
            }
            URIBuilder uriBuilder = new URIBuilder(apiUrl);
            uriBuilder.addParameters(pairList);
            HttpGet httpGet = new HttpGet(uriBuilder.build());
            response = httpClient.execute(httpGet);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            httpStr = EntityUtils.toString(entity, CHARSET);
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil get error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil post error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(params), statusCode, end});
            log.info("result:{}", new Object[]{httpStr});
        }
        return httpStr;
    }


    /**
     * @param apiUrl 请求地址
     * @param json   json数据
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description post, 以 application/json 形式  下载文件流
     */
    public static ByteArrayOutputStream downloadFileStream(String apiUrl, String json) {
        long start = System.currentTimeMillis();
        CloseableHttpClient httpClient = HttpClients.createDefault();
        String httpStr = null;
        HttpPost httpPost = new HttpPost(apiUrl);
        CloseableHttpResponse response = null;
        ByteArrayOutputStream byteOutStream = null;
        int statusCode = -999;
        try {
            httpPost.setConfig(requestConfig);
            StringEntity stringEntity = new StringEntity(json, CHARSET);
            //stringEntity.setContentEncoding(CHARSET);
            stringEntity.setContentType("application/json");
            httpPost.setEntity(stringEntity);
            response = httpClient.execute(httpPost);
            statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == HttpStatus.SC_OK) {
                //请求成功
                HttpEntity entity = response.getEntity();
                if (entity != null && entity.getContent() != null) {
                    //复制输入流
                    byteOutStream = cloneInputStream(entity.getContent());
                }
            }
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil post error:" + ex.getMessage());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ex) {
                    log.error("HttpUtil downloadFileStream error:" + ex.getMessage());
                }
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, json, statusCode, end});
            log.info("result:{}", new Object[]{httpStr});
        }
        return byteOutStream;
    }

    private static ByteArrayOutputStream cloneInputStream(InputStream input) {
        try {
            ByteArrayOutputStream byteOutStream = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            int len;
            while ((len = input.read(buffer)) > -1) {
                byteOutStream.write(buffer, 0, len);
            }
            byteOutStream.flush();
            return byteOutStream;
        } catch (IOException e) {
            log.warn("copy InputStream error，{}", ExceptionUtils.getStackTrace(e));
        }
        return null;
    }

    /**
     * @param url    请求地址
     * @param params 参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description get请求
     */
    public static String doGet(String url, String apiName, Map<String, Object> params) {
        long start = System.currentTimeMillis();
        StringBuffer param = new StringBuffer();
        int i = 0;

        //迭代器优化 --
        if (params != null) {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                String key = entry.getKey();
                // 拼接参数
                if (i == 0) {
                    param.append("?");
                } else {
                    param.append("&");
                }
                i = i + 1;
                param.append(key).append("=").append(params.get(key));
            }
        }

        String apiUrl = url + param;
        String result = null;
        //创建一个httpClient
        CloseableHttpClient httpClient = HttpClients.createDefault();
        int statusCode = -999;
        try {
            HttpGet httpGet = new HttpGet(apiUrl);
            HttpResponse response = httpClient.execute(httpGet);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            result = EntityUtils.toString(entity, CHARSET);
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil get error:" + ex.getMessage());
        } finally {
            if (httpClient != null) {
                //关闭流
                HttpClientUtils.closeQuietly(httpClient);
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(params), statusCode, end});
            log.info("result:{}", new Object[]{result});
        }
        return result;
    }

    /**
     * @param url       请求地址
     * @param params    参数
     * @param headerMap 请求头参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description 带请求头的Get请求
     */
    public static String doGet(String url, String apiName, Map<String, Object> params, Map<String, String> headerMap) {
        long start = System.currentTimeMillis();
        StringBuffer param = new StringBuffer();
        int i = 0;

        if (params != null && params.size() > 0) {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                String key = entry.getKey();
                // 拼接参数
                if (i == 0) {
                    param.append("?");
                } else {
                    param.append("&");
                }
                i = i + 1;
                param.append(key).append("=").append(params.get(key));
            }
        }

        String apiUrl = url + param;
        String result = null;
        CloseableHttpClient httpClient = HttpClients.createDefault();
        int statusCode = -999;
        try {
            HttpGet httpGet = new HttpGet(apiUrl);
            //循环增加header
            if (headerMap != null) {
                Iterator headerIterator = headerMap.entrySet().iterator();
                while (headerIterator.hasNext()) {
                    Map.Entry<String, String> elem = (Map.Entry<String, String>) headerIterator.next();
                    httpGet.addHeader(elem.getKey(), elem.getValue());
                }
            }
            HttpResponse response = httpClient.execute(httpGet);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            result = EntityUtils.toString(entity, CHARSET);
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil get error:" + ex.getMessage());
        } finally {
            if (httpClient != null) {
                HttpClientUtils.closeQuietly(httpClient);
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            MDC.put(Constants.STR_APINAME, apiName);
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(param), statusCode, end});
            log.info("result:{}", new Object[]{result});
        }
        return result;
    }

    /**
     * @param url       请求地址
     * @param params    参数
     * @param headerMap 请求头参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description 带请求头的Get请求 返回文件流 转base64字符串
     */
    public static String doGetForBase64(String url, Map<String, Object> params, Map<String, String> headerMap) {
        long start = System.currentTimeMillis();
        StringBuffer param = new StringBuffer();
        int i = 0;
        if (params != null) {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                String key = entry.getKey();
                // 拼接参数
                if (i == 0) {
                    param.append("?");
                } else {
                    param.append("&");
                }
                i = i + 1;
                param.append(key).append("=").append(params.get(key));
            }
        }

        String apiUrl = url + param;
        String result = null;
        CloseableHttpClient httpClient = HttpClients.createDefault();
        int statusCode = -999;
        try {
            HttpGet httpGet = new HttpGet(apiUrl);
            //循环增加header
            if (headerMap != null) {
                Iterator headerIterator = headerMap.entrySet().iterator();
                while (headerIterator.hasNext()) {
                    Map.Entry<String, String> elem = (Map.Entry<String, String>) headerIterator.next();
                    httpGet.addHeader(elem.getKey(), elem.getValue());
                }
            }
            HttpResponse response = httpClient.execute(httpGet);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                InputStream instream = entity.getContent();
                result = Base64.encode(instream);
            }
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil get error:" + ex.getMessage());
        } finally {
            if (httpClient != null) {
                HttpClientUtils.closeQuietly(httpClient);
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(param), statusCode, end});
            if (url.contains("syd/business/download")) {
                log.info("result to {} : 出参过滤 {}", apiUrl, result.length());
            } else {
                log.info("result:{}", new Object[]{result});
            }
        }
        return result;
    }

    /**
     * @param url       请求地址
     * @param params    参数
     * @param headerMap 请求头参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description 带请求头的Get请求 返回文件流
     */
    public static InputStream doGetForInputStream(String url, Map<String, Object> params, Map<String, String> headerMap) {
        long start = System.currentTimeMillis();
        StringBuffer param = new StringBuffer();
        int i = 0;
        if (params != null) {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                String key = entry.getKey();
                // 拼接参数
                if (i == 0) {
                    param.append("?");
                } else {
                    param.append("&");
                }
                i = i + 1;
                param.append(key).append("=").append(params.get(key));
            }
        }

        String apiUrl = url + param;
        String result = null;
        CloseableHttpClient httpClient = HttpClients.createDefault();
        int statusCode = -999;
        try {
            HttpGet httpGet = new HttpGet(apiUrl);
            //循环增加header
            if (headerMap != null) {
                Iterator headerIterator = headerMap.entrySet().iterator();
                while (headerIterator.hasNext()) {
                    Map.Entry<String, String> elem = (Map.Entry<String, String>) headerIterator.next();
                    httpGet.addHeader(elem.getKey(), elem.getValue());
                }
            }
            HttpResponse response = httpClient.execute(httpGet);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                return entity.getContent();
            }
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil get error:" + ex.getMessage());
        } finally {
            if (httpClient != null) {
                HttpClientUtils.closeQuietly(httpClient);
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(param), statusCode, end});
        }
        return null;
    }

    public static String doHttpsGetForBase64(String url, Map<String, Object> params, Map<String, String> headerMap) {
        long start = System.currentTimeMillis();
        StringBuffer param = new StringBuffer();
        int i = 0;
        if (params != null) {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                String key = entry.getKey();
                // 拼接参数
                if (i == 0) {
                    param.append("?");
                } else {
                    param.append("&");
                }
                i = i + 1;
                param.append(key).append("=").append(params.get(key));
            }
        }

        String apiUrl = url + param;
        String result = null;
        CloseableHttpResponse response = null;
        String httpStr = null;
        int statusCode = -999;
        SSLContext sslContext = null;
        try {
            sslContext = SSLContexts.custom().loadTrustMaterial(new TrustSelfSignedStrategy()).build();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        } catch (KeyManagementException e) {
            throw new RuntimeException(e);
        } catch (KeyStoreException e) {
            throw new RuntimeException(e);
        }
        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext, NoopHostnameVerifier.INSTANCE);
        //创建http实例
        CloseableHttpClient httpClient = HttpClients.custom().setSSLSocketFactory(sslsf).build();
        try {
            HttpGet httpGet = new HttpGet(apiUrl);
            //循环增加header
            if (headerMap != null) {
                Iterator headerIterator = headerMap.entrySet().iterator();
                while (headerIterator.hasNext()) {
                    Map.Entry<String, String> elem = (Map.Entry<String, String>) headerIterator.next();
                    httpGet.addHeader(elem.getKey(), elem.getValue());
                }
            }
            response = httpClient.execute(httpGet);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                InputStream instream = entity.getContent();
                result = Base64.encode(instream);
            }
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil get error:" + ex.getMessage());
        } finally {
            if (httpClient != null) {
                HttpClientUtils.closeQuietly(httpClient);
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{apiUrl, JSON.toJSONString(param), statusCode, end});
            if (url.contains("thirdcheck/downloadEInvoice")) {
                log.info("result.length:{}", result.length());
            } else {
                log.info("result:{}", new Object[]{result});
            }
        }
        return result;
    }

    /**
     * @param url  请求地址
     * @param json 参数
     * @return:String
     * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
     * @description 带body的Get请求 返回base64字符串
     */
    public static String doGet(String url, String json) {
        long start = System.currentTimeMillis();
        String result = null;
        CloseableHttpClient httpClient = HttpClients.createDefault();
        int statusCode = -999;
        try {
            HttpGetWithEntity httpGetWithEntity = new HttpGetWithEntity(url);
            HttpEntity httpEntity = new StringEntity(json, ContentType.APPLICATION_JSON);
            httpGetWithEntity.setEntity(httpEntity);
            HttpResponse response = httpClient.execute(httpGetWithEntity);
            statusCode = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                InputStream instream = entity.getContent();
                result = Base64.encode(instream);
            }
        } catch (Exception ex) {
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            throw new BizException(ErrorEnum.SERVICE_ERROR.getCode(), "HttpUtil get error:" + ex.getMessage());
        } finally {
            if (httpClient != null) {
                HttpClientUtils.closeQuietly(httpClient);
            }
            long end = System.currentTimeMillis() - start;
            MDC.put(Constants.STR_EXECUTETIME, String.valueOf(end));
            log.info("request to:{},param:{},response code:{},cost {} ms", new Object[]{url, json, statusCode, end});
            if (!url.contains("office/pdf/export")) {
                log.info("result:{}", new Object[]{result});
            } else {
                log.info("result.length:{}", result != null ? result.length() : "null");
            }
        }
        return result;
    }

    /**
     * 南方数码交易接口
     *
     * @param jsonStr
     * @param contentType
     * @param url
     * @param jkname
     * @param bh
     * @param ticket
     * @return
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     */
    public static String sendJyxxPost(String jsonStr, String contentType, String url, String jkname, String bh, String ticket) {
        String responseContent = null;
        CloseableHttpClient closeableHttpClient = HttpClients.createDefault();
        if (StringUtils.isBlank(contentType)) {
            contentType = "application/json;charset=UTF-8";
        }
        CloseableHttpResponse httpResponse = null;
        CloseableHttpClient closeHttpClient = HttpClients.createDefault();

        //httpPost方式提交数据
        HttpPost httpPost = new HttpPost(url);
        RequestConfig config = RequestConfig.custom()
                .setSocketTimeout(120000)
                .setConnectTimeout(120000)
                .setConnectionRequestTimeout(120000)
                .build();
        httpPost.setConfig(config);
        httpPost.addHeader("Content-Type", contentType);
        if (StringUtils.isNotBlank(ticket)) {
            httpPost.addHeader("Authorization", "BasicAuth " + ticket);
        }
        if (jsonStr != null) {
            //json字符串转为StringEntity，并防止中文乱码
            StringEntity stringEntity = new StringEntity(jsonStr, "UTF-8");
            //stringEntity.setContentEncoding("UTF-8");
            httpPost.setEntity(stringEntity);
        }
        //执行Post请求 得到Response对象
        try {
            if (closeableHttpClient != null) {
                httpResponse = closeableHttpClient.execute(httpPost);
                org.apache.http.HttpEntity httpEntity = httpResponse.getEntity();
                responseContent = EntityUtils.toString(httpEntity, "UTF-8");
                //数据带了转义符,导致转换异常
                Object parse = null;
                if (url.contains(Constants.BWYD)) {
                    parse = responseContent;
                } else {
                    parse = JSON.parse(responseContent);
                }
                if (parse != null) {
                    responseContent = parse.toString();
                }
            }
        } catch (IOException e) {
            log.error("http请求错误：" + e);
            responseContent = e.getMessage();
        } finally {
            try {
                if (closeHttpClient != null) {
                    closeHttpClient.close();
                }
                if (httpResponse != null) {
                    httpResponse.close();
                }
            } catch (Exception e) {
                log.info(e.getMessage());
            }
        }
        return responseContent;
    }


    /**
     * 南方数码交易接口
     *
     * @param url
     * @param ticket
     * @return
     * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     */
    public static String sendJyxxGet(String url, String ticket) {
        String responseContent = null;
        CloseableHttpClient closeableHttpClient = HttpClients.createDefault();
        CloseableHttpResponse httpResponse = null;
        CloseableHttpClient closeHttpClient = HttpClients.createDefault();

        //httpPost方式提交数据
        HttpGet httpGet = new HttpGet(url);
        RequestConfig config = RequestConfig.custom()
                .setSocketTimeout(120000)
                .setConnectTimeout(120000)
                .setConnectionRequestTimeout(120000)
                .build();
        httpGet.setConfig(config);
        if (StringUtils.isNotBlank(ticket)) {
            httpGet.addHeader("Authorization", "BasicAuth " + ticket);
        }

        //执行Post请求 得到Response对象
        try {
            if (closeableHttpClient != null) {
                httpResponse = closeableHttpClient.execute(httpGet);
                org.apache.http.HttpEntity httpEntity = httpResponse.getEntity();
                responseContent = EntityUtils.toString(httpEntity, "UTF-8");
                //数据带了转义符,导致转换异常
                Object parse = null;
                if (url.contains(Constants.BWYD)) {
                    parse = responseContent;
                } else {
                    parse = JSON.parse(responseContent);
                }
                if (parse != null) {
                    responseContent = parse.toString();
                }
            }
        } catch (IOException e) {
            log.error("http请求错误：" + e);
            responseContent = e.getMessage();
        } finally {
            try {
                if (closeHttpClient != null) {
                    closeHttpClient.close();
                }
                if (httpResponse != null) {
                    httpResponse.close();
                }
            } catch (Exception e) {
                log.info(e.getMessage());
            }
        }
        return responseContent;
    }


    /**
     * @param param
     * @param header
     * @param url
     * @param contentType
     * @param entity
     * @return
     */
    public static String httpClientPost(String url, String contentType, Map<String, String> param, Map<String, String> header, org.apache.http.HttpEntity entity) {
        String result = "";
        String bh = PublicUtil.hex32();
        log.info("bh:{},httpClientPost,接口开始调用时间：{}httpClientPost param: {} URL:{} header:{}", bh, LocalDateTime.now(), JSON.toJSONString(param), url, JSON.toJSONString(header));
        CloseableHttpClient closeableHttpClient = HttpClients.createDefault();
        try {
            HttpPost httpPost = new HttpPost(url);
            if (StringUtils.isNotBlank(contentType)) {
                httpPost.setHeader("Content-Type", contentType);
            }
            // 设置头信息
            if (MapUtils.isNotEmpty(header)) {
                for (Map.Entry<String, String> entry : header.entrySet()) {
                    httpPost.addHeader(entry.getKey(), entry.getValue());
                }
            }
            // 设置请求参数
            if (MapUtils.isNotEmpty(param)) {
                List<NameValuePair> formparams = new ArrayList<NameValuePair>();
                for (Map.Entry<String, String> entry : param.entrySet()) {
                    //给参数赋值
                    formparams.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
                }
                UrlEncodedFormEntity urlEncodedFormEntity = new UrlEncodedFormEntity(formparams, Consts.UTF_8);
                httpPost.setEntity(urlEncodedFormEntity);
            }
            // 设置实体 优先级高
            if (entity != null) {
                httpPost.setEntity(entity);
            }
            HttpResponse httpResponse = closeableHttpClient.execute(httpPost);
            try {
                int statusCode = httpResponse.getStatusLine().getStatusCode();
                if (statusCode == HttpStatus.SC_OK) {
                    org.apache.http.HttpEntity resEntity = httpResponse.getEntity();
                    result = EntityUtils.toString(resEntity);
                }
            } finally {
                if (httpResponse != null) {
                    ((CloseableHttpResponse) httpResponse).close();
                }
            }
        } catch (Exception e) {
            log.error(e.getMessage());
        } finally {
            try {
                closeableHttpClient.close();
            } catch (Exception e) {
                log.info(e.getMessage());
            }
        }
        log.info("httpClientPost请求result:{}", result);
        return result;
    }

    public static String httpRequest(HttpMethodEnum method, String url, String token, String content) {
        String responJson = null;
        CloseableHttpClient httpClient = null;
        CloseableHttpResponse httpResponse = null;
        InputStream inputStream = null;
        KeyStore trustStore = null;
        SSLContext sslcontext = null;
        try {
            trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
            // 设置信任签名
            sslcontext = SSLContexts.custom().loadTrustMaterial(trustStore, new TrustStrategy() {
                @Override
                public boolean isTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
                    return true;
                }
            }).build();
            X509HostnameVerifier hostnameVerifier = new X509HostnameVerifier() {
                @Override
                public void verify(String s, SSLSocket sslSocket) throws IOException {
                }

                @Override
                public void verify(String s, X509Certificate x509Certificate) throws SSLException {
                }

                @Override
                public void verify(String s, String[] strings, String[] strings1) throws SSLException {
                }

                @Override
                public boolean verify(String s, SSLSession sslSession) {
                    return true;
                }
            };
            // 设置协议http和https对应的处理socket链接工厂的对象
            Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
                    .register("http", PlainConnectionSocketFactory.INSTANCE)
                    .register("https", new SSLConnectionSocketFactory(sslcontext, hostnameVerifier))
                    .build();
            PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);
            httpClient = HttpClients.createDefault();
            ;
            //String proto = port == 8320 ? "https://" : "http://";
            String uri = url;
            HttpRequestBase httpReq = getRequestEntity(method, token, uri, content);
            httpResponse = httpClient.execute(httpReq);
            inputStream = httpResponse.getEntity().getContent();
            responJson = convertToString(inputStream);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (null != httpResponse) {
                try {
                    httpResponse.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (null != httpClient) {
                try {
                    httpClient.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return responJson;
    }

    /**
     * @param jsonStr
     * @param contentType
     * @param url
     * @param header
     * @return
     * @author <a href="mailto:liwenwu@gtmap.cn">liwenwu</a>
     * @version 2.0,
     * @description 带ssl带请求头的json请求模式
     */
    public static String httpHeaderPost(String jsonStr, String contentType, String url, Map<String, String> header) {
        return httpHeaderPost(jsonStr, contentType, url, header, null, null);
    }

    public static String httpHeaderPost(String jsonStr, String contentType, String url, Map<String, String> header, String jkname, String id) {
        if (StringUtils.isBlank(id)) {
            id = PublicUtil.hex32();
        }
        log.info("{}:接口开始调用时间：{},httpHeaderPost,jsonStr: {} URL:{} contentType:{} header:{}", id, LocalDateTime.now(), jsonStr, url, contentType, header);
        String responseContent = null;
        CloseableHttpClient closeableHttpClient = null;
        try {
            closeableHttpClient = HttpClients.createDefault();
        } catch (Exception e) {
            log.error("http请求错误：" + e);
        }
        if (StringUtils.isBlank(contentType)) {
            contentType = "application/json;charset=UTF-8";
        }
        CloseableHttpResponse httpResponse = null;
        CloseableHttpClient closeHttpClient = HttpClients.createDefault();
        //httpPost方式提交数据
        HttpPost httpPost = new HttpPost(url);
        httpPost.addHeader("Content-Type", contentType);
        // 设置头信息
        if (MapUtils.isNotEmpty(header)) {
            for (Map.Entry<String, String> entry : header.entrySet()) {
                httpPost.addHeader(entry.getKey(), entry.getValue());
            }
        }
        if (StringUtils.isNotBlank(jsonStr)) {
            //json字符串转为StringEntity，并防止中文乱码
            StringEntity stringEntity = new StringEntity(jsonStr, "UTF-8");
            //stringEntity.setContentEncoding("UTF-8");
            httpPost.setEntity(stringEntity);
        }
        //执行Post请求 得到Response对象
        try {
            if (closeableHttpClient != null) {
                httpResponse = closeableHttpClient.execute(httpPost);
                org.apache.http.HttpEntity httpEntity = httpResponse.getEntity();
                responseContent = EntityUtils.toString(httpEntity, "UTF-8");
            }
        } catch (IOException e) {
            log.error("http请求错误：" + e);
            responseContent = e.getMessage();
        } finally {
            try {
                if (closeHttpClient != null) {
                    closeHttpClient.close();
                }
                if (httpResponse != null) {
                    httpResponse.close();
                }
            } catch (Exception e) {
                log.info(e.getMessage());
            }
        }
        log.info("{}:接口结束调用时间：{},httpHeaderPost,URL:{} responseContent:{}", id, LocalDateTime.now(), url);
        return responseContent;
    }


    private static HttpRequestBase getRequestEntity(HttpMethodEnum method, String token, String uri, String content) throws UnsupportedEncodingException {
        switch (method.getNum()) {
            case 1:
                HttpGet httpGet = new HttpGet(uri + content);
                httpGet.addHeader("Content-type", "application/json");
                httpGet.addHeader("X-Subject-Token", token);
                return httpGet;
            case 2:
                HttpPost httpPost = new HttpPost(uri);
                httpPost.addHeader("Content-type", "application/json");
                httpPost.addHeader("X-Subject-Token", token);
                httpPost.setEntity(new StringEntity(content, "UTF-8"));
                return httpPost;
            case 3:
                HttpPut httpPut = new HttpPut(uri);
                httpPut.addHeader("Content-type", "application/json");
                httpPut.addHeader("X-Subject-Token", token);
                httpPut.setEntity(new StringEntity(content, "UTF-8"));
                return httpPut;
            case 4:
                HttpDelete httpDelete = new HttpDelete(uri + content);
                httpDelete.addHeader("Content-type", "application/json");
                httpDelete.addHeader("X-Subject-Token", token);
                return httpDelete;
            default:
                System.out.println("请求方法不对");
        }
        return null;
    }

    /**
     * @param
     * @return
     * @description 下载文件
     * @version 2.0, 2021/4/12 10:15
     * @author <a href="mailto:wangxuebiao@gtmap.cn">wangxuebiao</a>
     */
    public static byte[] downLoadFile(String fileUrl) {
        URL url;
        byte[] bytes = null;
        InputStream inputStream = null;
        try {
            url = new URL(fileUrl);
            HttpURLConnection http = (HttpURLConnection) url.openConnection();
            http.setConnectTimeout(5000);
            // 设置 User-Agent 避免被拦截
            http.setRequestProperty("User-Agent", "Mozilla/4.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)");
            // 获取文件名
            inputStream = http.getInputStream();
            if (inputStream != null) {
                bytes = FileUtils.convertToByteArray(inputStream);
                inputStream.close();
            }
            return bytes;
        } catch (MalformedURLException e) {
            e.printStackTrace();
        } catch (IOException e) {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e1) {
                    e.printStackTrace();
                }
            }
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    private static String convertToString(InputStream is) {
        if (is == null) {
            return null;
        }
        BufferedReader bf = null;
        try {
            StringBuilder sb = new StringBuilder();
            String temp = "";
            bf = new BufferedReader(new InputStreamReader(is, "UTF-8"));
            while ((temp = bf.readLine()) != null) {
                sb.append(temp);
            }
            return sb.toString();
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            closeStream(bf);
            closeStream(is);
        }
    }

    private static void closeStream(Closeable closeable) {
        if (null != closeable) {
            try {
                closeable.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }
}
