package cn.gtmap.hlw.core.util.property;

import cn.hutool.core.io.FileUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author jiangganzhi <jiangganzhi@gtmap.cn>
 * @date 2022年09月29日
 * @description 配置文件加载工具类
 */
@Slf4j
public class PropertiesLoadUtil {

    /**在使用正则表达式时,利用好其预编译功能*/
    private static final Pattern PATTERN = Pattern.compile("\\$\\{(.+?)\\}");

    /**
     * @author jiangganzhi <jiangganzhi@gtmap.cn>
     * @date 2022/9/29
     * @descreption 加载配置文件配置
     */
    public static void loadProperties() {
        //通过工作路径目录下是否有cfg文件夹判断是否部署后启动项目 如果没有cfg文件夹则读取项目内对应的配置文件
        String projectRootPath = System.getProperty("user.dir") + File.separator + "cfg";
        log.info("配置文件目录根路径：" + projectRootPath);
        if (!FileUtil.exist(projectRootPath)) {
            projectRootPath = PropertiesLoadUtil.class.getClassLoader().getResource("").getPath().substring(1).replace("/", File.separator);
        }
        List<String> propertiesFilePathList = new ArrayList<>();
        List<File> fileList = FileUtil.loopFiles(projectRootPath);
        if (CollectionUtils.isNotEmpty(fileList)) {
            for (File file : fileList) {
                String fileName = file.getName();
                if (StringUtils.endsWith(fileName, ".properties")) {
                    propertiesFilePathList.add(file.getAbsolutePath());
                }
            }
        }
        if (CollectionUtils.isNotEmpty(propertiesFilePathList)) {
            for (String propertiesPath : propertiesFilePathList) {
                log.info("======即将加载" + propertiesPath + "配置文件======");
                load(propertiesPath);
            }
        }
    }

    private static void load(String propertiesPath) {
        Properties properties = new Properties();
        InputStream proStream = null;
        BufferedReader proBufferReader = null;
        try {
            proStream = new BufferedInputStream(new FileInputStream(propertiesPath));
            if (proStream != null) {
                proBufferReader = new BufferedReader(new InputStreamReader(proStream, "UTF-8"));
                properties.load(proBufferReader);

                // 遍历每个属性
                for (String propertyName : properties.stringPropertyNames()) {
                    String propertyValue = properties.getProperty(propertyName);

                    // 匹配占位符，替换为实际的值
                    Matcher matcher = PATTERN.matcher(propertyValue);
                    StringBuffer buffer = new StringBuffer();
                    while (matcher.find()) {
                        String key = matcher.group(1);
                        String value = properties.getProperty(key);
                        if (value == null) {
                            log.error("Missing value for placeholder: " + key);
                        }
                        matcher.appendReplacement(buffer, Matcher.quoteReplacement(value));
                    }
                    matcher.appendTail(buffer);
                    String newValue = buffer.toString();

                    // 将替换后的值重新设置到 Properties 对象中
                    properties.setProperty(propertyName, newValue);
                }

                System.getProperties().putAll(properties);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (proStream != null) {
                    proStream.close();
                }
                if (proBufferReader != null) {
                    proBufferReader.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
}
