package cn.gtmap.hlw.core.config;

import cn.gtmap.hlw.core.base.ApiEncryptBody;
import cn.gtmap.hlw.core.util.encryption.ApiSecurityUtils;
import cn.hutool.core.io.FastByteArrayOutputStream;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.http.ContentType;
import com.alibaba.fastjson.JSONObject;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.multipart.support.StandardServletMultipartResolver;
import org.springframework.web.util.WebUtils;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;
import java.util.TreeMap;


/**
 * @Author jianglin
 * @Date 2024/6/27
 * @Description
 */
@Slf4j
public class EncryptRequestWrapper extends HttpServletRequestWrapper {

    //2024-02-02由于在过滤器中无法直接通过Autowired获取Bean,因此需要通过spring上下文来获取IOC管理的实体类
//    @Autowired
//    private RedisUtil redisUtil;

    protected FastByteArrayOutputStream cachedContent;

    protected String JSPublicKey;

    protected EncryptRequestWrapper(HttpServletRequest request) {
        super(request);
        this.copyBody();
    }

    public static EncryptRequestWrapper newOrGetMultiReadHttpServletRequest(HttpServletRequest request) {
        EncryptRequestWrapper multiReadHttpServletRequest = getMultiReadHttpServletRequest(request);
        if (null != multiReadHttpServletRequest) {
            return multiReadHttpServletRequest;
        } else {
            String contentType = request.getContentType();
            if (contentType != null && contentType.contains(ContentType.MULTIPART.getValue())) {
                // 将转化后的 request 放入过滤链中
                request = new StandardServletMultipartResolver().resolveMultipart(request);
            }
            return new EncryptRequestWrapper(request);
        }
    }

    public static EncryptRequestWrapper getMultiReadHttpServletRequest(HttpServletRequest request) {
        EncryptRequestWrapper nativeRequest = WebUtils.getNativeRequest(request, EncryptRequestWrapper.class);
        if (null != nativeRequest) {
            return nativeRequest;
        } else if (request instanceof EncryptRequestWrapper) {
            return (EncryptRequestWrapper) request;
        } else {
            return null;
        }
    }

    /**
     * 重新读取请求，因为request只可以被读取一次，需要重新设置为可多次读取
     */
    @SneakyThrows
    protected void copyBody() {
        int length = this.getContentLength();
        if (length > 0) {
            cachedContent = IoUtil.read(getRequest().getInputStream());
            String body = new String(cachedContent.toByteArray());
            if (StringUtils.isNotBlank(body)) {
                ApiEncryptBody jsonBody = JSONObject.parseObject(body, ApiEncryptBody.class);
                if (null != jsonBody) {
                    String dataEncrypt = jsonBody.getData();
                    String sm4key = jsonBody.getParamid();
                    JSPublicKey = jsonBody.getFrontKey();
                    String data;
                    data = ApiSecurityUtils.decrypt(sm4key, dataEncrypt);
                    // 如果数据不为空就编译
                    if (StringUtils.isNotBlank(data)) {
                        // 如果参数为空前端传回undefined
                        if ("undefined".equalsIgnoreCase(data) || StringUtils.isBlank(data)) {
                            // 或者其他默认值
                            body = "{}";
                        } else {
                            body = data;
                        }
                    }
                }
                cachedContent.reset();
                cachedContent.write(body.getBytes(), 0, body.getBytes().length);
            }
        } else {
            cachedContent = new FastByteArrayOutputStream();
        }
    }

    @Override
    public int getContentLength() {
        if (null != cachedContent) {
            return cachedContent.size();
        }
        return super.getContentLength();
    }

    public String getBody() {
        String body = null;
        if (cachedContent != null && cachedContent.size() > 0) {
            body = new String(cachedContent.toByteArray());
        }
        return body;
    }

    public String getSortBody() {
        return JSONObject.toJSONString(this.getBody(TreeMap.class));
    }

    public <T> T getBody(Class<T> type) {
        return cn.hutool.json.JSONUtil.toBean(this.getBody(), type);
    }

    /**
     * 这个方法为获取前端给后端用于加密aeskey的rsa公钥
     *
     * @return
     */
    public String getJSPublicKey() {
        return JSPublicKey;
    }

    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(getInputStream()));
    }

    @Override
    public ServletInputStream getInputStream() throws IOException {
        if (this.getContentLength() < 1) {
            return super.getInputStream();
        }
        return new ResettableServletInputStream(new ByteArrayInputStream(cachedContent.toByteArray()));
    }

    private class ResettableServletInputStream extends ServletInputStream {

        private final InputStream sourceStream;

        private boolean finished = false;


        /**
         * Create a DelegatingServletInputStream for the given source stream.
         *
         * @param sourceStream the source stream (never {@code null})
         */
        public ResettableServletInputStream(InputStream sourceStream) {
            Assert.notNull(sourceStream, "Source InputStream must not be null");
            this.sourceStream = sourceStream;
        }

        /**
         * Return the underlying source stream (never {@code null}).
         */
        public final InputStream getSourceStream() {
            return this.sourceStream;
        }


        @Override
        public int read() throws IOException {
            int data = this.sourceStream.read();
            if (data == -1) {
                this.finished = true;
            }
            return data;
        }

        @Override
        public int available() throws IOException {
            return this.sourceStream.available();
        }

        @Override
        public void close() throws IOException {
            super.close();
            this.sourceStream.close();
        }

        @Override
        public boolean isFinished() {
            return finished;
        }

        @Override
        public boolean isReady() {
            return true;
        }

        @Override
        public void setReadListener(ReadListener readListener) {

        }
    }
}
