package cn.gtmap.hlw.core.util.es;

import com.alibaba.fastjson.JSON;
import javafx.util.Pair;
import org.apache.http.Header;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequest;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsRequest;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsResponse;
import org.elasticsearch.action.admin.indices.get.GetIndexRequest;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.IndicesClient;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.List;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2025/3/7
 * @description
 */
@Component
public class ESClient {

    @Autowired
    private RestHighLevelClient restHighLevelClient;

    /**
     * 创建索引库
     *
     * @param index    索引名
     * @param type     映射类型
     * @param settings 设置的分片，备份分片数量
     * @param mappings 索引库的结构
     */
    public boolean createIndex(String index, String type, Settings.Builder settings, XContentBuilder mappings) throws IOException {
        CreateIndexRequest request = new CreateIndexRequest(index)
                .settings(settings)
                .mapping(type, mappings);
        try {
            return restHighLevelClient.indices().create(request, new Header[0]).isAcknowledged();
        } catch (ElasticsearchStatusException e) {
            if (e.status() == RestStatus.BAD_REQUEST && e.getMessage().contains("resource_already_exists_exception")) {
                System.out.println("索引已存在，跳过创建。");
                return true;
            } else {
                throw e; // 其他异常重新抛出
            }
        }
    }

    /**
     * 创建索引
     *
     * @param index
     * @param jsonMapping 索引库结构
     * @return
     * @throws IOException
     */
    public CreateIndexResponse createIndex(String index, String jsonMapping) throws IOException {
        CreateIndexRequest request = new CreateIndexRequest(index);
        request.source(jsonMapping, XContentType.JSON);
        return restHighLevelClient.indices().create(request, new Header[0]);
    }

    /**
     * 删除索引库
     *
     * @param index 索引库名
     */
    public boolean deleteIndex(String index) throws IOException {
        DeleteIndexRequest deleteRequest = new DeleteIndexRequest(index);
        AcknowledgedResponse response = restHighLevelClient.indices().delete(deleteRequest, new Header[0]);
        return response.isAcknowledged();
    }


    /**
     * 创建文档 指定id
     *
     * @param index      索引
     * @param id         文档id
     * @param jsonString 文档内容
     */
    public IndexResponse createDocument(String index,String type, String id, String jsonString) throws IOException {
        IndexRequest request = new IndexRequest(index).type(type)
                .id(id)
                .source(jsonString, XContentType.JSON);
        return restHighLevelClient.index(request, new Header[0]);
    }

    /**
     * 创建文档
     *
     * @param index      索引
     * @param jsonString 文档内容
     */
    public IndexResponse createDocument(String index, String jsonString) throws IOException {
        IndexRequest request = new IndexRequest(index)
                .source(jsonString, XContentType.JSON);
        return restHighLevelClient.index(request, new Header[0]);
    }

    /**
     * 获取文档
     *
     * @param index 索引库
     * @param id    文档id
     */
    public String getDocument(String index, String type, String id) throws IOException {
        GetRequest request = new GetRequest(index, type, id);
        return restHighLevelClient.get(request, new Header[0]).getSourceAsString();
    }


    /**
     * 删除文档
     *
     * @param index 索引库名
     * @param id    文档id
     */
    public boolean deleteDocument(String index, String type, String id) throws IOException {
        DeleteRequest deleteRequest = new DeleteRequest(index, type, id);
        DocWriteResponse.Result result = restHighLevelClient.delete(deleteRequest, new Header[0]).getResult();

        return result == DocWriteResponse.Result.DELETED;
    }

    /**
     * 更新文档
     *
     * @param index      索引库名称
     * @param id         文档id
     * @param jsonString 更新的内容
     */
    public UpdateResponse updateDocument(String index, String type, String id, String jsonString) throws IOException {
        UpdateRequest request = new UpdateRequest(index, type, id)
                .doc(jsonString, XContentType.JSON);
        return restHighLevelClient.update(request, new Header[0]);
    }

    /**
     * 查询文档
     *
     * @param index 索引库
     * @param query 查询条件
     */
    public SearchResponse searchDocuments(String index, SearchSourceBuilder query) throws IOException {
        SearchRequest searchRequest = new SearchRequest(index);
        searchRequest.source(query);
        return restHighLevelClient.search(searchRequest, new Header[0]);
    }

    /**
     * 批量插入,
     *
     * @param index    索引库名
     * @param dataList <id,data>文档列表
     * @return
     */
    //批量插入
    public <T> BulkResponse bulkCreateDocument(String index, List<Pair<String, T>> dataList) throws IOException {
        BulkRequest request = new BulkRequest();
        for (Pair<String, T> pair : dataList) {
            request.add(new IndexRequest(index)
                    .id(pair.getKey())
                    .source(JSON.toJSONString(pair.getValue()), XContentType.JSON));
        }
        return restHighLevelClient.bulk(request, new Header[0]);
    }

    /**
     * 批量删除
     *
     * @param index  索引库名
     * @param idList 需要删除的文档id
     * @return
     */
    public BulkResponse bulkDeleteDocument(String index, String type, List<String> idList) throws IOException {
        BulkRequest request = new BulkRequest();
        for (String id : idList) {
            request.add(new DeleteRequest(index, type, id));
        }
        return restHighLevelClient.bulk(request, new Header[0]);
    }


}