package cn.gtmap.gtc.clients;

import cn.gtmap.gtc.sso.domain.dto.*;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Set;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0, 2018/7/7
 * @description
 */

@FeignClient(name = "account")
@RequestMapping("/account/rest/v1/orgs")
public interface OrganizationManagerClient {

    /**
     * @param orgRecord 组织信息
     * @return
     * @description 创建组织机构
     */
    @PostMapping
    OrganizationDto createOrg(@RequestBody OrganizationDto orgRecord);

    /**
     * @param orgRecords 组织信息
     * @return
     * @description 批量导入组织机构
     */
    @PostMapping("/import")
    BaseResultDto importOrgs(@RequestBody List<OrganizationDto> orgRecords);

    /**
     * @param orgRecord 组织信息
     * @return
     * @description 修改组织机构
     */
    @PutMapping
    OrganizationDto updateOrg(@RequestBody OrganizationDto orgRecord);

    /**
     * @param id 组织id
     * @return
     * @description 删除组织机构
     */
    @DeleteMapping("/{id}")
    boolean deleteOrg(@PathVariable(name = "id") String id);

    /**
     * @param codes 通过编码
     * @return
     * @description 通过编码删除组织机构
     */
    @DeleteMapping("/codes")
    BaseResultDto deleteOrgsByCodes(@RequestParam(name = "codes") String codes);

    /**
     * @param id 组织id
     * @return
     * @description 根据ID查找组织机构
     */
    @GetMapping("/{id}")
    OrganizationDto findOrgById(@PathVariable(name = "id") String id);

    /**
     * @param ids 组织id列表
     * @return
     * @description 根据ID列表查找组织机构
     */
    @PostMapping("/ids")
    List<OrganizationDto> findOrgByIds(@RequestBody List<String> ids);

    /**
     * @param id 组织id
     * @return
     * @description 分页查找组织机构下的用户
     */
    @GetMapping("/{id}/users")
    Page<UserDto> findUsersByOrg(Pageable pageable, @PathVariable(name = "id") String id);

    /**
     * @param id 组织id
     * @return
     * @description 分页查找组织的用户
     */
    @GetMapping("/{id}/page-users")
    Page<UserDto> queryUsersByOrg(Pageable pageable, @PathVariable(name = "id") String id);

    /**
     * @param id 组织id
     * @return
     * @description 查找组织机构下所有用户
     */
    @GetMapping("/{id}/all-users")
    List<UserDto> listUsersByOrg(@PathVariable(name = "id") String id);

    /**
     * @param id 组织id
     * @return
     * @description 组织添加人员列表
     */
    @PostMapping("/{id}/add-users")
    boolean addOrgUsers(@PathVariable(name = "id") String id, @RequestBody List<String> userIds);

    /**
     * @param id 组织id
     * @param userIds 用户Id列表，英文逗号分隔；
     * @return
     * @description 组织删除人员列表
     */
    @DeleteMapping("/{id}/del-users")
    boolean delOrgUsers(@PathVariable(name = "id") String id, @RequestParam(name = "userIds") String userIds);

    /**
     * @param id 组织id
     * @return
     * @description 根据ID查找组织的根节点
     */
    @GetMapping("/{id}/root")
    OrganizationDto queryRootOrg(@PathVariable(name = "id") String id);

    /**
     * @param id 组织id
     * @return
     * @description 查找直接下级组织机构
     * @deprecated {@link OrganizationManagerClient#findChildren}
     */
    @GetMapping("/{id}/junior")
    @Deprecated
    List<OrganizationDto> findJuniorOrgs(@PathVariable(name = "id") String id);

    /**
     * @param id 组织id
     * @param enabled 0:禁用 1：正常 null: 全部
     * @return
     * @description 查找直接下级组织机构
     */
    @GetMapping("/{id}/children")
    List<OrganizationDto> findChildren(@PathVariable(name = "id") String id, @RequestParam(name = "enabled", required = false) Integer enabled);


    /**
     * @param
     * @return
     * @description 查找最顶级组织机构
     */
    @GetMapping("/root")
    List<OrganizationDto> findRootOrgs();

    /**
     * @param id 当前组织id
     * @param parentId 上级组织id
     * @return
     * @description 重设父级组织
     */
    @PostMapping("/{id}/parent")
    boolean relateParent(@PathVariable(name = "id") String id, @RequestParam(name = "parentId") String parentId);

    /**
     * @param id 当前组织id
     * @return
     * @description 禁用组织
     */
    @PostMapping("/{id}/disable")
    boolean disabled(@PathVariable("id") String id);

    /**
     * @param id 当前组织id
     * @return
     * @description 启用组织
     */
    @PostMapping("/{id}/enable")
    boolean enable(@PathVariable("id") String id);

    /**
     * @param id 当前组织id
     * @param code 组织编码
     * @return
     * @description 校验code的唯一性
     */
    @GetMapping("/{code}/code-only")
    boolean validOnlyCode(@RequestParam(name = "id", required = false) String id, @PathVariable(name = "code") String code);

    /**
     * @param id 当前组织id
     * @param enabled 1:正常 0：禁用
     * @return
     * @description 获取组织下角色
     */
    @GetMapping("/{id}/roles")
    List<RoleDto> listRoles(@PathVariable(name = "id") String id, @RequestParam(name = "enabled", required = false) Integer enabled);

    /**
     * @return
     * @description 查询所有组织
     */
    @GetMapping("/list")
    List<OrganizationDto> listOrgs(@RequestParam(name = "enabled", required = false) Integer enabled);

    /**
     * @return
     * @description 查询所有组织及部分用户
     */
    @GetMapping("/org-users")
    List<OrgUserDto> listOrgUsers(@RequestParam(name = "enabled", required = false) Integer enabled);

    /**
     * @return
     * @description 查询所有组织及用户
     */
    @GetMapping("/all-org-users")
    List<OrgUserDto> listAllOrgUsers(@RequestParam(name = "enabled", required = false) Integer enabled);

    /**
     * @return parentId 为空查询根节点，不为空查询下级
     * @description 分级查询组织及用户
     */
    @GetMapping("/grade-org-users")
    List<OrgUserDto> listGradeOrgUsers(@RequestParam(name = "enabled", required = false) Integer enabled,
                                              @RequestParam(name = "parentId", required = false) String parentId);

    /**
     * @return
     * @description 查询用户上级组织
     */
    @GetMapping("/user-parent")
    List<OrganizationDto> listUserParentOrgs(@RequestParam(name = "username") String username);

    /**
     * 查询用户所属组织机构的用户
     * @param pageable
     * @param optUsername
     * @param orgId
     * @param username
     * @param alias
     * @param enabled
     * @return
     */
    @GetMapping(value = "/list-orgs/affiliation-orgs")
    public Page<UserDto> listAffiliationOrgMembers(Pageable pageable,
                                                   @RequestParam(name = "optUsername") String optUsername,
                                                   @RequestParam(name = "orgId", required = false) String orgId,
                                                   @RequestParam(name = "username", required = false) String username,
                                                   @RequestParam(name = "alias", required = false) String alias,
                                                   @RequestParam(name = "enabled", required = false) Integer enabled);

}
