package cn.gtmap.estateplat.log.config;

import cn.gtmap.estateplat.core.ex.AppException;
import cn.gtmap.estateplat.model.server.core.BdcXtLog;
import cn.gtmap.estateplat.utils.ClientInfoUtil;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.gtis.common.util.UUIDGenerator;
import com.gtis.plat.service.SysUserService;
import com.gtis.plat.vo.PfUserVo;
import com.gtis.web.SessionUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.data.domain.Pageable;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.ui.Model;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * @author <a href="mailto:yulei1@gtmap.cn">yulei</a>
 * @version 1.0, 2017/11/13
 * @description
 */
@Aspect
public class ConfigLogAspect {

    private ConfigLogService configLogService;

    private SysUserService sysUserService;

    private ExecutorService executor = Executors.newSingleThreadExecutor();

    private final static String ANONYMOUS_USER_ID = "-999";

    private final static String ANONYMOUS_USER_NAME = "匿名用户";

    //登录用户实际ip
    public static final String LOGIN_USER_IP = "login_user_ip";

    public static final String LOGIN_USER_MAC = "login_user_mac";

    public static final String LOGIN_USER_COMPUTERNAME = "login_user_computername";

    public void setConfigLogService(ConfigLogService configLogService) {
        this.configLogService = configLogService;
    }

    public void setSysUserService(SysUserService sysUserService) {
        this.sysUserService = sysUserService;
    }

    @Pointcut("@annotation(cn.gtmap.estateplat.log.config.ConfigLog)")
    public void configLogServiceAspect() {
    }

    @Before("configLogServiceAspect()")
    public void doBefore(JoinPoint joinPoint) {
        Method method = this.getMethod(joinPoint);
        ConfigLog configLog = method.getAnnotation(ConfigLog.class);
        String userId = StringUtils.isNotBlank(configLog.userId()) ? parseUserId(configLog.userId(), method, joinPoint.getArgs()) : null;
        List<Object> arguments = filterArguments(joinPoint.getArgs());
        if (CollectionUtils.isNotEmpty(arguments)) {
            for (Object argument : arguments) {
                if (argument instanceof Object[]) {
                    Object[] objects = (Object[]) argument;
                    if (objects.length != 0) {
                        for (Object object : objects) {
                            configLog(configLog.name(), getConfigContent(object, configLog), userId);
                        }
                    }
                } else {
                    configLog(configLog.name(), getConfigContent(argument, configLog), userId);
                }
            }
        }
    }

    @After("configLogServiceAspect()")
    public void doAfter(JoinPoint joinPoint) {

    }

    private void configLog(String name, String content, String userId) {
        final BdcXtLog bdcXtLog = new BdcXtLog();
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        bdcXtLog.setLogid(UUIDGenerator.generate());
        bdcXtLog.setCzrq(Calendar.getInstance().getTime());
        if (StringUtils.isNotBlank(SessionUtil.getCurrentUserId())) {
            bdcXtLog.setUserid(SessionUtil.getCurrentUserId());
            bdcXtLog.setUsername(SessionUtil.getCurrentUser().getUsername());
        } else {
            if (StringUtils.isNotBlank(userId)) {
                bdcXtLog.setUserid(userId);
                PfUserVo pfUserVo = sysUserService != null ? sysUserService.getUserVo(userId) : null;
                bdcXtLog.setUsername(pfUserVo != null ? pfUserVo.getUserName() : StringUtils.EMPTY);
            } else {
                bdcXtLog.setUserid(ANONYMOUS_USER_ID);
                bdcXtLog.setUsername(ANONYMOUS_USER_NAME);
            }
        }
        bdcXtLog.setController(name);
        bdcXtLog.setParmjson(content);
        try {
            String ip = ClientInfoUtil.getIpAddr(request);
            bdcXtLog.setIp(ip);
            if (StringUtils.isNotBlank(ip) && !StringUtils.equals(ip, "127.0.0.1")) {
                String mac = ClientInfoUtil.getMACAddress(ip);
                String computername = ClientInfoUtil.getComputerName(ip);
                bdcXtLog.setMac(mac);
                bdcXtLog.setComputername(computername);
            }
        } catch (Exception e) {

        }
        executor.submit(new Runnable() {
            @Override
            public void run() {
                configLogService.saveConfigLog(bdcXtLog);
            }
        });
    }

    private ConfigLog getMethodLog(JoinPoint joinPoint) {
        try {
            String targetName = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            Object[] arguments = joinPoint.getArgs();
            Class targetClass = Class.forName(targetName);
            Method[] methods = targetClass.getMethods();
            for (Method method : methods) {
                if (method.getName().equals(methodName)) {
                    Class[] classes = method.getParameterTypes();
                    if (classes.length == arguments.length) {
                        return method.getAnnotation(ConfigLog.class);
                    }
                }
            }
        } catch (ClassNotFoundException e) {
            throw new AppException(e.getMessage());
        }

        return null;
    }

    private String getConfigContent(Object argument, ConfigLog configLog) {
        Map<String, Object> contentMap = Maps.newHashMap();
        String description = StringUtils.isNotBlank(configLog.description()) ? configLog.description() : "操作#className内容";
        description = StringUtils.replace(description, "#className", argument.getClass().getSimpleName());
        contentMap.put(description, argument);
        return JSON.toJSONString(contentMap);
    }

    /**
     * 过滤掉一些无关的参数
     *
     * @param args
     * @return
     */
    private List<Object> filterArguments(Object[] args) {
        List<Object> arguments = Lists.newArrayList();
        for (Object arg : args) {
            if (!(arg instanceof Model) && !(arg instanceof Pageable) && !(arg instanceof HttpServletResponse) && !(arg instanceof HttpServletRequest)) {
                arguments.add(arg);
            }
        }
        return arguments;
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
     * @version 1.0, 2016/7/29
     * @description 根据SPEL表达式获取注解中的参数值
     */
    private String parseUserId(String userId, Method method, Object[] args) {
        LocalVariableTableParameterNameDiscoverer u = new LocalVariableTableParameterNameDiscoverer();
        String[] paraNameArr = u.getParameterNames(method);

        //使用SPEL进行key的解析
        ExpressionParser parser = new SpelExpressionParser();
        //SPEL上下文
        StandardEvaluationContext context = new StandardEvaluationContext();
        //把方法参数放入SPEL上下文中
        for (int i = 0; i < paraNameArr.length; i++) {
            context.setVariable(paraNameArr[i], args[i]);
        }
        return parser.parseExpression(userId).getValue(context, String.class);
    }

    /**
     * @param joinPoint 注解点
     * @return
     * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
     * @version 1.0, 2016/7/29
     * @description 返回注解的方法对象
     */
    private Method getMethod(JoinPoint joinPoint) {
        Object[] args = joinPoint.getArgs();
        Class[] argTypes = new Class[joinPoint.getArgs().length];
        for (int i = 0; i < args.length; i++) {
            argTypes[i] = args[i].getClass();
        }
        Method method = null;
        try {
            method = joinPoint.getTarget().getClass().getMethod(joinPoint.getSignature().getName(), argTypes);
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (SecurityException e) {
            e.printStackTrace();
        }
        return method;
    }

    private String getRemoteAddress(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || ip.equalsIgnoreCase("unknown")) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || ip.equalsIgnoreCase("unknown")) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || ip.equalsIgnoreCase("unknown")) {
            ip = request.getRemoteAddr();
        }
        return ip;
    }


}
