package cn.gtmap.estateplat.core.support.freemarker.directive;

import cn.gtmap.estateplat.core.support.freemarker.FmUtils;
import cn.gtmap.estateplat.utils.Codecs;
import cn.gtmap.estateplat.utils.RequestUtils;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import freemarker.core.Environment;
import freemarker.template.TemplateDirectiveBody;
import freemarker.template.TemplateDirectiveModel;
import freemarker.template.TemplateException;
import freemarker.template.TemplateModel;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.ServletContextAware;

import javax.servlet.ServletContext;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.Set;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-6-30
 */
public abstract class AbstractResourceDirective implements TemplateDirectiveModel, ServletContextAware {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractResourceDirective.class);
    public static final String BASE = "base";
    public static final String PATH = "path";
    public static final String VERSION = "version";
    private static final Set<String> IGNORES = Sets.newHashSet(BASE, PATH, VERSION);
    private ServletContext servletContext;

    private Map<String, FileItem> crc32Cache = Maps.newConcurrentMap();

    static class FileItem {
        private File file;
        private long lastModified;
        private String crc32;

        FileItem(File file) {
            this.file = file;
            lastModified = file.lastModified();
            crc32 = getCrc32(file);
        }

        private void update() {
            if (file.lastModified() > lastModified) {
                crc32 = getCrc32(file);
            }
        }
    }

    @Override
    public void setServletContext(ServletContext servletContext) {
        this.servletContext = servletContext;
    }

    @Override
    public void execute(Environment env, Map params, TemplateModel[] loopVars, TemplateDirectiveBody body) throws TemplateException, IOException {
        String base = FmUtils.getString(params, BASE, getDefaultBase());
        Object path = FmUtils.getParam(params, PATH);
        if (path == null) {
            throw new TemplateException("path is required", env);
        }
        StringBuilder attrSb = new StringBuilder();
        for (Object obj : params.entrySet()) {
            Map.Entry entry = (Map.Entry) obj;
            String key = (String) entry.getKey();
            if (IGNORES.contains(key)) {
                continue;
            }
            FmUtils.appendUE(attrSb, key, entry.getValue());
        }
        String attrString = attrSb.toString();
        Object version = FmUtils.getParam(params, VERSION);
        if (path instanceof String) {
            out(env, getHtml(format(base, (String) path, version), attrString));
        } else if (path instanceof Iterable) {
            for (Object obj : (Iterable) path) {
                out(env, getHtml(format(base, obj.toString(), version), attrString));
            }
        }
    }

    protected String getDefaultBase() {
        return "/static";
    }

    protected abstract String getHtml(String path, String attrs);

    private String format(String base, String path, Object version) {
        StringBuilder sb = new StringBuilder(base);
        if (path.charAt(0) != '/') {
            sb.append("/");
        }
        sb.append(path);
        path = sb.toString();
        if (version != null) {
            if (version instanceof Boolean) {
                if (!(Boolean) version) {
                    return RequestUtils.buildUrl(path, null);
                }
            } else {
                return RequestUtils.buildUrl(sb.append("?").append(version).toString(), null);
            }
        }
        FileItem fi = crc32Cache.get(path);
        if (fi == null) {
            String filePath = servletContext.getRealPath(path);
            if (filePath != null) {
                File file = new File(filePath);
                if (file.exists()) {
                    fi = new FileItem(file);
                    crc32Cache.put(path, fi);
                }
            }
        } else {
            fi.update();
        }
        if (fi != null) {
            sb.append("?").append(fi.crc32);
        } else {
            LOG.info("Web resouce [{}] not found", path);
        }
        return RequestUtils.buildUrl(sb.toString(), null);
    }

    private static String getCrc32(File file) {
        try {
            return Codecs.encode((int) FileUtils.checksumCRC32(file));
        } catch (IOException e) {
            return "";
        }
    }

    private static void out(Environment env, String html) throws IOException {
        env.getOut().write(html);
        env.getOut().write("\n");
    }
}
