/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util;

import java.io.IOException;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.FixedBitSet;
import org.apache.lucene.util.hash.HashFunction;

public class FuzzySet {
    public static final int FUZZY_SERIALIZATION_VERSION = 1;
    private HashFunction hashFunction;
    private FixedBitSet filter;
    private int bloomSize;
    static final int[] usableBitSetSizes;

    public static int getNearestSetSize(int maxNumberOfBits) {
        int result = usableBitSetSizes[0];
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            if (usableBitSetSizes[i] > maxNumberOfBits) continue;
            result = usableBitSetSizes[i];
        }
        return result;
    }

    public static int getNearestSetSize(int maxNumberOfValuesExpected, float desiredSaturation) {
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            int numSetBitsAtDesiredSaturation = (int)((float)usableBitSetSizes[i] * desiredSaturation);
            int estimatedNumUniqueValues = FuzzySet.getEstimatedNumberUniqueValuesAllowingForCollisions(usableBitSetSizes[i], numSetBitsAtDesiredSaturation);
            if (estimatedNumUniqueValues <= maxNumberOfValuesExpected) continue;
            return usableBitSetSizes[i];
        }
        return -1;
    }

    public static FuzzySet createSetBasedOnMaxMemory(int maxNumBytes, HashFunction hashFunction) {
        int setSize = FuzzySet.getNearestSetSize(maxNumBytes);
        return new FuzzySet(new FixedBitSet(setSize + 1), setSize, hashFunction);
    }

    public static FuzzySet createSetBasedOnQuality(int maxNumUniqueValues, float desiredMaxSaturation, HashFunction hashFunction) {
        int setSize = FuzzySet.getNearestSetSize(maxNumUniqueValues, desiredMaxSaturation);
        return new FuzzySet(new FixedBitSet(setSize + 1), setSize, hashFunction);
    }

    private FuzzySet(FixedBitSet filter, int bloomSize, HashFunction hashFunction) {
        this.filter = filter;
        this.bloomSize = bloomSize;
        this.hashFunction = hashFunction;
    }

    public ContainsResult contains(BytesRef value) {
        int hash = this.hashFunction.hash(value);
        if (hash < 0) {
            hash *= -1;
        }
        return this.mayContainValue(hash);
    }

    public void serialize(DataOutput out) throws IOException {
        out.writeInt(1);
        out.writeString(this.hashFunction.getName());
        out.writeInt(this.bloomSize);
        long[] bits = this.filter.getBits();
        out.writeInt(bits.length);
        for (int i = 0; i < bits.length; ++i) {
            out.writeLong(bits[i]);
        }
    }

    public static FuzzySet deserialize(DataInput in) throws IOException {
        int version = in.readInt();
        if (version != 1) {
            throw new IOException("Error deserializing: set version is not 1");
        }
        HashFunction hashFunction = HashFunction.forName(in.readString());
        int bloomSize = in.readInt();
        int numLongs = in.readInt();
        long[] longs = new long[numLongs];
        for (int i = 0; i < numLongs; ++i) {
            longs[i] = in.readLong();
        }
        FixedBitSet bits = new FixedBitSet(longs, bloomSize + 1);
        return new FuzzySet(bits, bloomSize, hashFunction);
    }

    private ContainsResult mayContainValue(int positiveHash) {
        assert (positiveHash >= 0);
        int pos = positiveHash & this.bloomSize;
        if (this.filter.get(pos)) {
            return ContainsResult.MAYBE;
        }
        return ContainsResult.NO;
    }

    public void addValue(BytesRef value) throws IOException {
        int hash = this.hashFunction.hash(value);
        if (hash < 0) {
            hash *= -1;
        }
        int bloomPos = hash & this.bloomSize;
        this.filter.set(bloomPos);
    }

    public FuzzySet downsize(float targetMaxSaturation) {
        int numBitsSet = this.filter.cardinality();
        FixedBitSet rightSizedBitSet = this.filter;
        int rightSizedBitSetSize = this.bloomSize;
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            int candidateBitsetSize = usableBitSetSizes[i];
            float candidateSaturation = (float)numBitsSet / (float)candidateBitsetSize;
            if (!(candidateSaturation <= targetMaxSaturation)) continue;
            rightSizedBitSetSize = candidateBitsetSize;
            break;
        }
        if (rightSizedBitSetSize < this.bloomSize) {
            rightSizedBitSet = new FixedBitSet(rightSizedBitSetSize + 1);
            int bitIndex = 0;
            do {
                if ((bitIndex = this.filter.nextSetBit(bitIndex)) < 0) continue;
                int downSizedBitIndex = bitIndex & rightSizedBitSetSize;
                rightSizedBitSet.set(downSizedBitIndex);
                ++bitIndex;
            } while (bitIndex >= 0 && bitIndex <= this.bloomSize);
        } else {
            return null;
        }
        return new FuzzySet(rightSizedBitSet, rightSizedBitSetSize, this.hashFunction);
    }

    public int getEstimatedUniqueValues() {
        return FuzzySet.getEstimatedNumberUniqueValuesAllowingForCollisions(this.bloomSize, this.filter.cardinality());
    }

    public static int getEstimatedNumberUniqueValuesAllowingForCollisions(int setSize, int numRecordedBits) {
        double setSizeAsDouble = setSize;
        double numRecordedBitsAsDouble = numRecordedBits;
        double saturation = numRecordedBitsAsDouble / setSizeAsDouble;
        double logInverseSaturation = Math.log(1.0 - saturation) * -1.0;
        return (int)(setSizeAsDouble * logInverseSaturation);
    }

    public float getSaturation() {
        int numBitsSet = this.filter.cardinality();
        return (float)numBitsSet / (float)this.bloomSize;
    }

    static {
        int mask;
        usableBitSetSizes = new int[30];
        int size = mask = 1;
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            FuzzySet.usableBitSetSizes[i] = size = size << 1 | mask;
        }
    }

    public static enum ContainsResult {
        MAYBE,
        NO;

    }
}

