/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.contract;

import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Properties;
import java.util.UUID;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.junit.Assert;
import org.junit.internal.AssumptionViolatedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContractTestUtils
extends Assert {
    private static final Logger LOG = LoggerFactory.getLogger(ContractTestUtils.class);
    public static final String IO_FILE_BUFFER_SIZE = "io.file.buffer.size";
    public static final String IO_CHUNK_BUFFER_SIZE = "io.chunk.buffer.size";
    public static final int DEFAULT_IO_CHUNK_BUFFER_SIZE = 128;
    public static final String IO_CHUNK_MODULUS_SIZE = "io.chunk.modulus.size";
    public static final int DEFAULT_IO_CHUNK_MODULUS_SIZE = 128;

    public static void assertPropertyEquals(Properties props, String key, String expected) {
        String val = props.getProperty(key);
        if (expected == null) {
            ContractTestUtils.assertNull((String)("Non null property " + key + " = " + val), (Object)val);
        } else {
            ContractTestUtils.assertEquals((String)("property " + key + " = " + val), (Object)expected, (Object)val);
        }
    }

    public static void writeAndRead(FileSystem fs, Path path, byte[] src, int len, int blocksize, boolean overwrite, boolean delete) throws IOException {
        fs.mkdirs(path.getParent());
        ContractTestUtils.writeDataset(fs, path, src, len, blocksize, overwrite);
        byte[] dest = ContractTestUtils.readDataset(fs, path, len);
        ContractTestUtils.compareByteArrays(src, dest, len);
        if (delete) {
            ContractTestUtils.rejectRootOperation(path);
            boolean deleted = fs.delete(path, false);
            ContractTestUtils.assertTrue((String)"Deleted", (boolean)deleted);
            ContractTestUtils.assertPathDoesNotExist(fs, "Cleanup failed", path);
        }
    }

    public static void writeDataset(FileSystem fs, Path path, byte[] src, int len, int buffersize, boolean overwrite) throws IOException {
        ContractTestUtils.assertTrue((String)("Not enough data in source array to write " + len + " bytes"), (src.length >= len ? 1 : 0) != 0);
        FSDataOutputStream out = fs.create(path, overwrite, fs.getConf().getInt(IO_FILE_BUFFER_SIZE, 4096), (short)1, (long)buffersize);
        out.write(src, 0, len);
        out.close();
        ContractTestUtils.assertFileHasLength(fs, path, len);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] readDataset(FileSystem fs, Path path, int len) throws IOException {
        FSDataInputStream in = fs.open(path);
        byte[] dest = new byte[len];
        int offset = 0;
        try {
            int nbytes;
            for (int nread = 0; nread < len; nread += nbytes) {
                nbytes = in.read(dest, offset + nread, len - nread);
                if (nbytes >= 0) continue;
                throw new EOFException("End of file reached before reading fully.");
            }
        }
        finally {
            in.close();
        }
        return dest;
    }

    public static void verifyFileContents(FileSystem fs, Path path, byte[] original) throws IOException {
        FileStatus stat = fs.getFileStatus(path);
        String statText = stat.toString();
        ContractTestUtils.assertTrue((String)("not a file " + statText), (boolean)stat.isFile());
        ContractTestUtils.assertEquals((String)("wrong length " + statText), (long)original.length, (long)stat.getLen());
        byte[] bytes = ContractTestUtils.readDataset(fs, path, original.length);
        ContractTestUtils.compareByteArrays(original, bytes, original.length);
    }

    public static void verifyRead(FSDataInputStream stm, byte[] fileContents, int seekOff, int toRead) throws IOException {
        byte[] out = new byte[toRead];
        stm.seek((long)seekOff);
        stm.readFully(out);
        byte[] expected = Arrays.copyOfRange(fileContents, seekOff, seekOff + toRead);
        ContractTestUtils.compareByteArrays(expected, out, toRead);
    }

    public static void compareByteArrays(byte[] original, byte[] received, int len) {
        ContractTestUtils.assertEquals((String)"Number of bytes read != number written", (long)len, (long)received.length);
        int errors = 0;
        int first_error_byte = -1;
        for (int i = 0; i < len; ++i) {
            if (original[i] == received[i]) continue;
            if (errors == 0) {
                first_error_byte = i;
            }
            ++errors;
        }
        if (errors > 0) {
            String message = String.format(" %d errors in file of length %d", errors, len);
            LOG.warn(message);
            int overlap = 10;
            for (int i = Math.max(0, first_error_byte - 10); i < Math.min(first_error_byte + 10, len); ++i) {
                byte actual = received[i];
                byte expected = original[i];
                String letter = ContractTestUtils.toChar(actual);
                String line = String.format("[%04d] %2x %s\n", i, actual, letter);
                if (expected != actual) {
                    line = String.format("[%04d] %2x %s -expected %2x %s\n", i, actual, letter, expected, ContractTestUtils.toChar(expected));
                }
                LOG.warn(line);
            }
            ContractTestUtils.fail((String)message);
        }
    }

    public static String toChar(byte b) {
        if (b >= 32) {
            return Character.toString((char)b);
        }
        return String.format("%02x", b);
    }

    public static String toChar(byte[] buffer) {
        StringBuilder builder = new StringBuilder(buffer.length);
        for (byte b : buffer) {
            builder.append(ContractTestUtils.toChar(b));
        }
        return builder.toString();
    }

    public static byte[] toAsciiByteArray(String s) {
        char[] chars = s.toCharArray();
        int len = chars.length;
        byte[] buffer = new byte[len];
        for (int i = 0; i < len; ++i) {
            buffer[i] = (byte)(chars[i] & 0xFF);
        }
        return buffer;
    }

    public static void cleanup(String action, FileSystem fileSystem, String cleanupPath) {
        if (fileSystem == null) {
            return;
        }
        Path path = new Path(cleanupPath).makeQualified(fileSystem.getUri(), fileSystem.getWorkingDirectory());
        ContractTestUtils.cleanup(action, fileSystem, path);
    }

    public static void cleanup(String action, FileSystem fileSystem, Path path) {
        ContractTestUtils.noteAction(action);
        try {
            ContractTestUtils.rm(fileSystem, path, true, false);
        }
        catch (Exception e) {
            LOG.error("Error deleting in " + action + " - " + path + ": " + e, (Throwable)e);
        }
    }

    public static boolean rm(FileSystem fileSystem, Path path, boolean recursive, boolean allowRootDelete) throws IOException {
        if (fileSystem != null) {
            ContractTestUtils.rejectRootOperation(path, allowRootDelete);
            if (fileSystem.exists(path)) {
                return fileSystem.delete(path, recursive);
            }
        }
        return false;
    }

    public static void rejectRootOperation(Path path, boolean allowRootOperation) throws IOException {
        if (path.isRoot() && !allowRootOperation) {
            throw new IOException("Root directory operation rejected: " + path);
        }
    }

    public static void rejectRootOperation(Path path) throws IOException {
        ContractTestUtils.rejectRootOperation(path, false);
    }

    public static void noteAction(String action) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==============  " + action + " =============");
        }
    }

    public static void downgrade(String message, Throwable failure) {
        LOG.warn("Downgrading test " + message, failure);
        AssumptionViolatedException ave = new AssumptionViolatedException((Object)failure, null);
        throw ave;
    }

    public static void unsupported(String message) {
        ContractTestUtils.skip(message);
    }

    public static void skip(String message) {
        LOG.info("Skipping: {}", (Object)message);
        throw new AssumptionViolatedException(message);
    }

    public static void fail(String text, Throwable thrown) {
        AssertionError e = new AssertionError((Object)text);
        ((Throwable)((Object)e)).initCause(thrown);
        throw e;
    }

    public static void assertFileHasLength(FileSystem fs, Path path, int expected) throws IOException {
        FileStatus status = fs.getFileStatus(path);
        ContractTestUtils.assertEquals((String)("Wrong file length of file " + path + " status: " + status), (long)expected, (long)status.getLen());
    }

    public static void assertIsDirectory(FileSystem fs, Path path) throws IOException {
        FileStatus fileStatus = fs.getFileStatus(path);
        ContractTestUtils.assertIsDirectory(fileStatus);
    }

    public static void assertIsDirectory(FileStatus fileStatus) {
        ContractTestUtils.assertTrue((String)("Should be a directory -but isn't: " + fileStatus), (boolean)fileStatus.isDirectory());
    }

    public static byte[] writeTextFile(FileSystem fs, Path path, String text, boolean overwrite) throws IOException {
        byte[] bytes = new byte[]{};
        if (text != null) {
            bytes = ContractTestUtils.toAsciiByteArray(text);
        }
        ContractTestUtils.createFile(fs, path, overwrite, bytes);
        return bytes;
    }

    public static void createFile(FileSystem fs, Path path, boolean overwrite, byte[] data) throws IOException {
        FSDataOutputStream stream = fs.create(path, overwrite);
        if (data != null && data.length > 0) {
            stream.write(data);
        }
        stream.close();
    }

    public static void touch(FileSystem fs, Path path) throws IOException {
        ContractTestUtils.createFile(fs, path, true, null);
    }

    public static void assertDeleted(FileSystem fs, Path file, boolean recursive) throws IOException {
        ContractTestUtils.assertDeleted(fs, file, recursive, false);
    }

    public static void assertDeleted(FileSystem fs, Path file, boolean recursive, boolean allowRootOperations) throws IOException {
        ContractTestUtils.rejectRootOperation(file, allowRootOperations);
        ContractTestUtils.assertPathExists(fs, "about to be deleted file", file);
        boolean deleted = fs.delete(file, recursive);
        String dir = ContractTestUtils.ls(fs, file.getParent());
        ContractTestUtils.assertTrue((String)("Delete failed on " + file + ": " + dir), (boolean)deleted);
        ContractTestUtils.assertPathDoesNotExist(fs, "Deleted file", file);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String readBytesToString(FileSystem fs, Path path, int length) throws IOException {
        try (FSDataInputStream in = fs.open(path);){
            byte[] buf = new byte[length];
            in.readFully(0L, buf);
            String string = ContractTestUtils.toChar(buf);
            return string;
        }
    }

    public static String fileStatsToString(FileStatus[] stats, String separator) {
        StringBuilder buf = new StringBuilder(stats.length * 128);
        for (int i = 0; i < stats.length; ++i) {
            buf.append(String.format("[%02d] %s", i, stats[i])).append(separator);
        }
        return buf.toString();
    }

    public static String ls(FileSystem fileSystem, Path path) throws IOException {
        FileStatus[] stats;
        if (path == null) {
            return "/";
        }
        String pathtext = "ls " + path;
        try {
            stats = fileSystem.listStatus(path);
        }
        catch (FileNotFoundException e) {
            return pathtext + " -file not found";
        }
        catch (IOException e) {
            return pathtext + " -failed: " + e;
        }
        return ContractTestUtils.dumpStats(pathtext, stats);
    }

    public static String dumpStats(String pathname, FileStatus[] stats) {
        return pathname + ContractTestUtils.fileStatsToString(stats, "\n");
    }

    public static void assertIsFile(FileSystem fileSystem, Path filename) throws IOException {
        ContractTestUtils.assertPathExists(fileSystem, "Expected file", filename);
        FileStatus status = fileSystem.getFileStatus(filename);
        ContractTestUtils.assertIsFile(filename, status);
    }

    public static void assertIsFile(Path filename, FileStatus status) {
        String fileInfo = filename + "  " + status;
        ContractTestUtils.assertFalse((String)("File claims to be a directory " + fileInfo), (boolean)status.isDirectory());
        ContractTestUtils.assertFalse((String)("File claims to be a symlink " + fileInfo), (boolean)status.isSymlink());
    }

    public static byte[] dataset(int len, int base, int modulo) {
        byte[] dataset = new byte[len];
        for (int i = 0; i < len; ++i) {
            dataset[i] = (byte)(base + i % modulo);
        }
        return dataset;
    }

    public static void assertPathExists(FileSystem fileSystem, String message, Path path) throws IOException {
        if (!fileSystem.exists(path)) {
            ContractTestUtils.ls(fileSystem, path.getParent());
            throw new FileNotFoundException(message + ": not found " + path + " in " + path.getParent());
        }
    }

    public static void assertPathDoesNotExist(FileSystem fileSystem, String message, Path path) throws IOException {
        try {
            FileStatus status = fileSystem.getFileStatus(path);
            ContractTestUtils.fail((String)(message + ": unexpectedly found " + path + " as  " + status));
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    public static void assertListStatusFinds(FileSystem fs, Path dir, Path subdir) throws IOException {
        FileStatus[] stats = fs.listStatus(dir);
        boolean found = false;
        StringBuilder builder = new StringBuilder();
        for (FileStatus stat : stats) {
            builder.append(stat.toString()).append('\n');
            if (!stat.getPath().equals((Object)subdir)) continue;
            found = true;
        }
        ContractTestUtils.assertTrue((String)("Path " + subdir + " not found in directory " + dir + ":" + builder), (boolean)found);
    }

    public static boolean isOSX() {
        return System.getProperty("os.name").contains("OS X");
    }

    public static void validateFileContent(byte[] concat, byte[][] bytes) {
        int idx = 0;
        boolean mismatch = false;
        byte[][] arr$ = bytes;
        int len$ = arr$.length;
        for (int i$ = 0; i$ < len$; ++i$) {
            byte[] bb;
            for (byte b : bb = arr$[i$]) {
                if (b == concat[idx++]) continue;
                mismatch = true;
                break;
            }
            if (mismatch) break;
        }
        ContractTestUtils.assertFalse((String)("File content of file is not as expected at offset " + idx), (boolean)mismatch);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void verifyReceivedData(FileSystem fs, Path path, long expectedSize, int bufferLen, int modulus) throws IOException {
        byte[] testBuffer = new byte[bufferLen];
        long totalBytesRead = 0L;
        int nextExpectedNumber = 0;
        try (FSDataInputStream inputStream = fs.open(path);){
            int bytesRead;
            while ((bytesRead = inputStream.read(testBuffer)) >= 0) {
                totalBytesRead += (long)bytesRead;
                for (int i = 0; i < bytesRead; ++i) {
                    if (testBuffer[i] != nextExpectedNumber) {
                        throw new IOException("Read number " + testBuffer[i] + " but expected " + nextExpectedNumber);
                    }
                    if (++nextExpectedNumber != modulus) continue;
                    nextExpectedNumber = 0;
                }
            }
            if (totalBytesRead != expectedSize) {
                throw new IOException("Expected to read " + expectedSize + " bytes but only received " + totalBytesRead);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static long generateTestFile(FileSystem fs, Path path, long size, int bufferLen, int modulus) throws IOException {
        byte[] testBuffer = new byte[bufferLen];
        for (int i = 0; i < testBuffer.length; ++i) {
            testBuffer[i] = (byte)(i % modulus);
        }
        long bytesWritten = 0L;
        try (FSDataOutputStream outputStream = fs.create(path, false);){
            while (bytesWritten < size) {
                long diff = size - bytesWritten;
                if (diff < (long)testBuffer.length) {
                    outputStream.write(testBuffer, 0, (int)diff);
                    bytesWritten += diff;
                    continue;
                }
                outputStream.write(testBuffer);
                bytesWritten += (long)testBuffer.length;
            }
            long l = bytesWritten;
            return l;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void createAndVerifyFile(FileSystem fs, Path parent, long fileSize) throws IOException {
        int testBufferSize = fs.getConf().getInt(IO_CHUNK_BUFFER_SIZE, 128);
        int modulus = fs.getConf().getInt(IO_CHUNK_MODULUS_SIZE, 128);
        String objectName = UUID.randomUUID().toString();
        Path objectPath = new Path(parent, objectName);
        ContractTestUtils.assertEquals((long)fileSize, (long)ContractTestUtils.generateTestFile(fs, objectPath, fileSize, testBufferSize, modulus));
        ContractTestUtils.assertPathExists(fs, "not created successful", objectPath);
        try {
            ContractTestUtils.verifyReceivedData(fs, objectPath, fileSize, testBufferSize, modulus);
        }
        finally {
            fs.delete(objectPath, false);
        }
    }
}

