/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.URI;
import java.util.Arrays;
import java.util.Random;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BufferedFSInputStream;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileSystemTestHelper;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RawLocalFileSystem;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.internal.util.reflection.Whitebox;

public class TestLocalFileSystem {
    private static final String TEST_ROOT_DIR = System.getProperty("test.build.data", "build/test/data") + "/work-dir/localfs";
    private final File base = new File(TEST_ROOT_DIR);
    private final Path TEST_PATH = new Path(TEST_ROOT_DIR, "test-file");
    private Configuration conf;
    private LocalFileSystem fileSys;

    private void cleanupFile(FileSystem fs, Path name) throws IOException {
        Assert.assertTrue((boolean)fs.exists(name));
        fs.delete(name, true);
        Assert.assertTrue((!fs.exists(name) ? 1 : 0) != 0);
    }

    @Before
    public void setup() throws IOException {
        this.conf = new Configuration(false);
        this.conf.set("fs.file.impl", LocalFileSystem.class.getName());
        this.fileSys = FileSystem.getLocal((Configuration)this.conf);
        this.fileSys.delete(new Path(TEST_ROOT_DIR), true);
    }

    @After
    public void after() throws IOException {
        FileUtil.setWritable((File)this.base, (boolean)true);
        FileUtil.fullyDelete((File)this.base);
        Assert.assertTrue((!this.base.exists() ? 1 : 0) != 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=10000L)
    public void testWorkingDirectory() throws IOException {
        Path origDir = this.fileSys.getWorkingDirectory();
        Path subdir = new Path(TEST_ROOT_DIR, "new");
        try {
            Assert.assertTrue((!this.fileSys.exists(subdir) ? 1 : 0) != 0);
            Assert.assertTrue((boolean)this.fileSys.mkdirs(subdir));
            Assert.assertTrue((boolean)this.fileSys.isDirectory(subdir));
            this.fileSys.setWorkingDirectory(subdir);
            Path dir1 = new Path("dir1");
            Assert.assertTrue((boolean)this.fileSys.mkdirs(dir1));
            Assert.assertTrue((boolean)this.fileSys.isDirectory(dir1));
            this.fileSys.delete(dir1, true);
            Assert.assertTrue((!this.fileSys.exists(dir1) ? 1 : 0) != 0);
            Path file1 = new Path("file1");
            Path file2 = new Path("sub/file2");
            String contents = FileSystemTestHelper.writeFile((FileSystem)this.fileSys, file1, 1);
            this.fileSys.copyFromLocalFile(file1, file2);
            Assert.assertTrue((boolean)this.fileSys.exists(file1));
            Assert.assertTrue((boolean)this.fileSys.isFile(file1));
            this.cleanupFile((FileSystem)this.fileSys, file2);
            this.fileSys.copyToLocalFile(file1, file2);
            this.cleanupFile((FileSystem)this.fileSys, file2);
            this.fileSys.rename(file1, file2);
            Assert.assertTrue((!this.fileSys.exists(file1) ? 1 : 0) != 0);
            Assert.assertTrue((boolean)this.fileSys.exists(file2));
            this.fileSys.rename(file2, file1);
            FSDataInputStream stm = this.fileSys.open(file1);
            byte[] buffer = new byte[3];
            int bytesRead = stm.read(buffer, 0, 3);
            Assert.assertEquals((Object)contents, (Object)new String(buffer, 0, bytesRead));
            stm.close();
        }
        finally {
            this.fileSys.setWorkingDirectory(origDir);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=1000L)
    public void testSyncable() throws IOException {
        FileSystem fs = this.fileSys.getRawFileSystem();
        Path file = new Path(TEST_ROOT_DIR, "syncable");
        FSDataOutputStream out = fs.create(file);
        boolean bytesWritten = true;
        byte[] expectedBuf = new byte[]{48, 49, 50, 51};
        try {
            out.write(expectedBuf, 0, 1);
            out.hflush();
            this.verifyFile(fs, file, 1, expectedBuf);
            out.write(expectedBuf, 1, expectedBuf.length - 1);
            out.hsync();
            this.verifyFile(fs, file, expectedBuf.length, expectedBuf);
        }
        finally {
            out.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyFile(FileSystem fs, Path file, int bytesToVerify, byte[] expectedBytes) throws IOException {
        FSDataInputStream in = fs.open(file);
        try {
            byte[] readBuf = new byte[bytesToVerify];
            in.readFully(readBuf, 0, bytesToVerify);
            for (int i = 0; i < bytesToVerify; ++i) {
                Assert.assertEquals((long)expectedBytes[i], (long)readBuf[i]);
            }
        }
        finally {
            in.close();
        }
    }

    @Test(timeout=10000L)
    public void testCopy() throws IOException {
        Path src = new Path(TEST_ROOT_DIR, "dingo");
        Path dst = new Path(TEST_ROOT_DIR, "yak");
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, src, 1);
        Assert.assertTrue((boolean)FileUtil.copy((FileSystem)this.fileSys, (Path)src, (FileSystem)this.fileSys, (Path)dst, (boolean)true, (boolean)false, (Configuration)this.conf));
        Assert.assertTrue((!this.fileSys.exists(src) && this.fileSys.exists(dst) ? 1 : 0) != 0);
        Assert.assertTrue((boolean)FileUtil.copy((FileSystem)this.fileSys, (Path)dst, (FileSystem)this.fileSys, (Path)src, (boolean)false, (boolean)false, (Configuration)this.conf));
        Assert.assertTrue((this.fileSys.exists(src) && this.fileSys.exists(dst) ? 1 : 0) != 0);
        Assert.assertTrue((boolean)FileUtil.copy((FileSystem)this.fileSys, (Path)src, (FileSystem)this.fileSys, (Path)dst, (boolean)true, (boolean)true, (Configuration)this.conf));
        Assert.assertTrue((!this.fileSys.exists(src) && this.fileSys.exists(dst) ? 1 : 0) != 0);
        this.fileSys.mkdirs(src);
        Assert.assertTrue((boolean)FileUtil.copy((FileSystem)this.fileSys, (Path)dst, (FileSystem)this.fileSys, (Path)src, (boolean)false, (boolean)false, (Configuration)this.conf));
        Path tmp = new Path(src, dst.getName());
        Assert.assertTrue((this.fileSys.exists(tmp) && this.fileSys.exists(dst) ? 1 : 0) != 0);
        Assert.assertTrue((boolean)FileUtil.copy((FileSystem)this.fileSys, (Path)dst, (FileSystem)this.fileSys, (Path)src, (boolean)false, (boolean)true, (Configuration)this.conf));
        Assert.assertTrue((boolean)this.fileSys.delete(tmp, true));
        this.fileSys.mkdirs(tmp);
        try {
            FileUtil.copy((FileSystem)this.fileSys, (Path)dst, (FileSystem)this.fileSys, (Path)src, (boolean)true, (boolean)true, (Configuration)this.conf);
            Assert.fail((String)"Failed to detect existing dir");
        }
        catch (IOException e) {
            // empty catch block
        }
    }

    @Test(timeout=1000L)
    public void testHomeDirectory() throws IOException {
        Path home = new Path(System.getProperty("user.home")).makeQualified((FileSystem)this.fileSys);
        Path fsHome = this.fileSys.getHomeDirectory();
        Assert.assertEquals((Object)home, (Object)fsHome);
    }

    @Test(timeout=1000L)
    public void testPathEscapes() throws IOException {
        Path path = new Path(TEST_ROOT_DIR, "foo%bar");
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, path, 1);
        FileStatus status = this.fileSys.getFileStatus(path);
        Assert.assertEquals((Object)path.makeQualified((FileSystem)this.fileSys), (Object)status.getPath());
        this.cleanupFile((FileSystem)this.fileSys, path);
    }

    @Test(timeout=1000L)
    public void testCreateFileAndMkdirs() throws IOException {
        Path test_dir = new Path(TEST_ROOT_DIR, "test_dir");
        Path test_file = new Path(test_dir, "file1");
        Assert.assertTrue((boolean)this.fileSys.mkdirs(test_dir));
        int fileSize = new Random().nextInt(0x100000) + 1;
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, test_file, fileSize);
        FileStatus status = this.fileSys.getFileStatus(test_file);
        Assert.assertEquals((long)fileSize, (long)status.getLen());
        ContentSummary summary = this.fileSys.getContentSummary(test_dir);
        Assert.assertEquals((long)fileSize, (long)summary.getLength());
        Path bad_dir = new Path(test_file, "another_dir");
        try {
            this.fileSys.mkdirs(bad_dir);
            Assert.fail((String)"Failed to detect existing file in path");
        }
        catch (ParentNotDirectoryException e) {
            // empty catch block
        }
        try {
            this.fileSys.mkdirs(null);
            Assert.fail((String)"Failed to detect null in mkdir arg");
        }
        catch (IllegalArgumentException e) {
            // empty catch block
        }
    }

    @Test(timeout=1000L)
    public void testBasicDelete() throws IOException {
        Path dir1 = new Path(TEST_ROOT_DIR, "dir1");
        Path file1 = new Path(TEST_ROOT_DIR, "file1");
        Path file2 = new Path(TEST_ROOT_DIR + "/dir1", "file2");
        Path file3 = new Path(TEST_ROOT_DIR, "does-not-exist");
        Assert.assertTrue((boolean)this.fileSys.mkdirs(dir1));
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, file1, 1);
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, file2, 1);
        Assert.assertFalse((String)"Returned true deleting non-existant path", (boolean)this.fileSys.delete(file3));
        Assert.assertTrue((String)"Did not delete file", (boolean)this.fileSys.delete(file1));
        Assert.assertTrue((String)"Did not delete non-empty dir", (boolean)this.fileSys.delete(dir1));
    }

    @Test(timeout=1000L)
    public void testStatistics() throws Exception {
        int fileSchemeCount = 0;
        for (FileSystem.Statistics stats : FileSystem.getAllStatistics()) {
            if (!stats.getScheme().equals("file")) continue;
            ++fileSchemeCount;
        }
        Assert.assertEquals((long)1L, (long)fileSchemeCount);
    }

    @Test(timeout=1000L)
    public void testHasFileDescriptor() throws IOException {
        Path path = new Path(TEST_ROOT_DIR, "test-file");
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, path, 1);
        RawLocalFileSystem rawLocalFileSystem = new RawLocalFileSystem();
        rawLocalFileSystem.getClass();
        BufferedFSInputStream bis = new BufferedFSInputStream((FSInputStream)new RawLocalFileSystem.LocalFSFileInputStream(rawLocalFileSystem, path), 1024);
        Assert.assertNotNull((Object)bis.getFileDescriptor());
        bis.close();
    }

    @Test(timeout=1000L)
    public void testListStatusWithColons() throws IOException {
        Assume.assumeTrue((!Shell.WINDOWS ? 1 : 0) != 0);
        File colonFile = new File(TEST_ROOT_DIR, "foo:bar");
        colonFile.mkdirs();
        FileStatus[] stats = this.fileSys.listStatus(new Path(TEST_ROOT_DIR));
        Assert.assertEquals((String)"Unexpected number of stats", (long)1L, (long)stats.length);
        Assert.assertEquals((String)"Bad path from stat", (Object)colonFile.getAbsolutePath(), (Object)stats[0].getPath().toUri().getPath());
    }

    @Test
    public void testListStatusReturnConsistentPathOnWindows() throws IOException {
        Assume.assumeTrue((boolean)Shell.WINDOWS);
        String dirNoDriveSpec = TEST_ROOT_DIR;
        if (dirNoDriveSpec.charAt(1) == ':') {
            dirNoDriveSpec = dirNoDriveSpec.substring(2);
        }
        File file = new File(dirNoDriveSpec, "foo");
        file.mkdirs();
        FileStatus[] stats = this.fileSys.listStatus(new Path(dirNoDriveSpec));
        Assert.assertEquals((String)"Unexpected number of stats", (long)1L, (long)stats.length);
        Assert.assertEquals((String)"Bad path from stat", (Object)new Path(file.getPath()).toUri().getPath(), (Object)stats[0].getPath().toUri().getPath());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=10000L)
    public void testReportChecksumFailure() throws IOException {
        this.base.mkdirs();
        Assert.assertTrue((this.base.exists() && this.base.isDirectory() ? 1 : 0) != 0);
        File dir1 = new File(this.base, "dir1");
        File dir2 = new File(dir1, "dir2");
        dir2.mkdirs();
        Assert.assertTrue((dir2.exists() && FileUtil.canWrite((File)dir2) ? 1 : 0) != 0);
        String dataFileName = "corruptedData";
        Path dataPath = new Path(new File(dir2, "corruptedData").toURI());
        Path checksumPath = this.fileSys.getChecksumFile(dataPath);
        FSDataOutputStream fsdos = this.fileSys.create(dataPath);
        try {
            fsdos.writeUTF("foo");
        }
        finally {
            fsdos.close();
        }
        Assert.assertTrue((boolean)this.fileSys.pathToFile(dataPath).exists());
        long dataFileLength = this.fileSys.getFileStatus(dataPath).getLen();
        Assert.assertTrue((dataFileLength > 0L ? 1 : 0) != 0);
        Assert.assertTrue((boolean)this.fileSys.pathToFile(checksumPath).exists());
        long checksumFileLength = this.fileSys.getFileStatus(checksumPath).getLen();
        Assert.assertTrue((checksumFileLength > 0L ? 1 : 0) != 0);
        FileUtil.setWritable((File)this.base, (boolean)false);
        FSDataInputStream dataFsdis = this.fileSys.open(dataPath);
        FSDataInputStream checksumFsdis = this.fileSys.open(checksumPath);
        boolean retryIsNecessary = this.fileSys.reportChecksumFailure(dataPath, dataFsdis, 0L, checksumFsdis, 0L);
        Assert.assertTrue((!retryIsNecessary ? 1 : 0) != 0);
        Assert.assertTrue((!this.fileSys.pathToFile(dataPath).exists() ? 1 : 0) != 0);
        Assert.assertTrue((!this.fileSys.pathToFile(checksumPath).exists() ? 1 : 0) != 0);
        File[] dir1files = dir1.listFiles(new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return pathname != null && !pathname.getName().equals("dir2");
            }
        });
        Assert.assertTrue((dir1files != null ? 1 : 0) != 0);
        Assert.assertTrue((dir1files.length == 1 ? 1 : 0) != 0);
        File badFilesDir = dir1files[0];
        File[] badFiles = badFilesDir.listFiles();
        Assert.assertTrue((badFiles != null ? 1 : 0) != 0);
        Assert.assertTrue((badFiles.length == 2 ? 1 : 0) != 0);
        boolean dataFileFound = false;
        boolean checksumFileFound = false;
        for (File badFile : badFiles) {
            if (badFile.getName().startsWith("corruptedData")) {
                Assert.assertTrue((dataFileLength == badFile.length() ? 1 : 0) != 0);
                dataFileFound = true;
                continue;
            }
            if (!badFile.getName().contains("corruptedData.crc")) continue;
            Assert.assertTrue((checksumFileLength == badFile.length() ? 1 : 0) != 0);
            checksumFileFound = true;
        }
        Assert.assertTrue((boolean)dataFileFound);
        Assert.assertTrue((boolean)checksumFileFound);
    }

    @Test(timeout=1000L)
    public void testSetTimes() throws Exception {
        Path path = new Path(TEST_ROOT_DIR, "set-times");
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, path, 1);
        long newModTime = 12345000L;
        FileStatus status = this.fileSys.getFileStatus(path);
        Assert.assertTrue((String)"check we're actually changing something", (newModTime != status.getModificationTime() ? 1 : 0) != 0);
        long accessTime = status.getAccessTime();
        this.fileSys.setTimes(path, newModTime, -1L);
        status = this.fileSys.getFileStatus(path);
        Assert.assertEquals((long)newModTime, (long)status.getModificationTime());
        Assert.assertEquals((long)accessTime, (long)status.getAccessTime());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testBufferedFSInputStream() throws IOException {
        Configuration conf = new Configuration();
        conf.setClass("fs.file.impl", RawLocalFileSystem.class, FileSystem.class);
        conf.setInt("io.file.buffer.size", 4096);
        FileSystem fs = FileSystem.newInstance((Configuration)conf);
        byte[] buf = new byte[10240];
        new Random().nextBytes(buf);
        FSDataOutputStream stream = fs.create(this.TEST_PATH);
        try {
            stream.write(buf);
        }
        finally {
            stream.close();
        }
        Random r = new Random();
        FSDataInputStream stm = fs.open(this.TEST_PATH);
        int[] seeks = new int[10];
        int[] reads = new int[10];
        try {
            for (int i = 0; i < 1000; ++i) {
                int seekOff = r.nextInt(buf.length);
                int toRead = r.nextInt(Math.min(buf.length - seekOff, 32000));
                seeks[i % seeks.length] = seekOff;
                reads[i % reads.length] = toRead;
                this.verifyRead(stm, buf, seekOff, toRead);
            }
        }
        catch (AssertionError afe) {
            StringBuilder sb = new StringBuilder();
            sb.append("Sequence of actions:\n");
            for (int j = 0; j < seeks.length; ++j) {
                sb.append("seek @ ").append(seeks[j]).append("  ").append("read ").append(reads[j]).append("\n");
            }
            System.err.println(sb.toString());
            throw afe;
        }
        finally {
            stm.close();
        }
    }

    @Test
    public void testRenameDirectory() throws IOException {
        Path src = new Path(TEST_ROOT_DIR, "dir1");
        Path dst = new Path(TEST_ROOT_DIR, "dir2");
        this.fileSys.delete(src, true);
        this.fileSys.delete(dst, true);
        Assert.assertTrue((boolean)this.fileSys.mkdirs(src));
        Assert.assertTrue((boolean)this.fileSys.rename(src, dst));
        Assert.assertTrue((boolean)this.fileSys.exists(dst));
        Assert.assertFalse((boolean)this.fileSys.exists(src));
    }

    @Test
    public void testRenameReplaceExistingEmptyDirectory() throws IOException {
        Path src = new Path(TEST_ROOT_DIR, "dir1");
        Path dst = new Path(TEST_ROOT_DIR, "dir2");
        this.fileSys.delete(src, true);
        this.fileSys.delete(dst, true);
        Assert.assertTrue((boolean)this.fileSys.mkdirs(src));
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, new Path(src, "file1"), 1);
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, new Path(src, "file2"), 1);
        Assert.assertTrue((boolean)this.fileSys.mkdirs(dst));
        Assert.assertTrue((boolean)this.fileSys.rename(src, dst));
        Assert.assertTrue((boolean)this.fileSys.exists(dst));
        Assert.assertTrue((boolean)this.fileSys.exists(new Path(dst, "file1")));
        Assert.assertTrue((boolean)this.fileSys.exists(new Path(dst, "file2")));
        Assert.assertFalse((boolean)this.fileSys.exists(src));
    }

    @Test
    public void testRenameMoveToExistingNonEmptyDirectory() throws IOException {
        Path src = new Path(TEST_ROOT_DIR, "dir1/dir2/dir3");
        Path dst = new Path(TEST_ROOT_DIR, "dir1");
        this.fileSys.delete(src, true);
        this.fileSys.delete(dst, true);
        Assert.assertTrue((boolean)this.fileSys.mkdirs(src));
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, new Path(src, "file1"), 1);
        FileSystemTestHelper.writeFile((FileSystem)this.fileSys, new Path(src, "file2"), 1);
        Assert.assertTrue((boolean)this.fileSys.exists(dst));
        Assert.assertTrue((boolean)this.fileSys.rename(src, dst));
        Assert.assertTrue((boolean)this.fileSys.exists(dst));
        Assert.assertTrue((boolean)this.fileSys.exists(new Path(dst, "dir3")));
        Assert.assertTrue((boolean)this.fileSys.exists(new Path(dst, "dir3/file1")));
        Assert.assertTrue((boolean)this.fileSys.exists(new Path(dst, "dir3/file2")));
        Assert.assertFalse((boolean)this.fileSys.exists(src));
    }

    private void verifyRead(FSDataInputStream stm, byte[] fileContents, int seekOff, int toRead) throws IOException {
        byte[] out = new byte[toRead];
        stm.seek((long)seekOff);
        stm.readFully(out);
        byte[] expected = Arrays.copyOfRange(fileContents, seekOff, seekOff + toRead);
        if (!Arrays.equals(out, expected)) {
            String s = "\nExpected: " + StringUtils.byteToHexString((byte[])expected) + "\ngot:      " + StringUtils.byteToHexString((byte[])out) + "\noff=" + seekOff + " len=" + toRead;
            Assert.fail((String)s);
        }
    }

    @Test
    public void testStripFragmentFromPath() throws Exception {
        LocalFileSystem fs = FileSystem.getLocal((Configuration)new Configuration());
        Path pathQualified = this.TEST_PATH.makeQualified(fs.getUri(), fs.getWorkingDirectory());
        Path pathWithFragment = new Path(new URI(pathQualified.toString() + "#glacier"));
        FileSystemTestHelper.createFile((FileSystem)fs, pathWithFragment);
        Path resolved = fs.resolvePath(pathWithFragment);
        Assert.assertEquals((String)"resolvePath did not strip fragment from Path", (Object)pathQualified, (Object)resolved);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAppendSetsPosCorrectly() throws Exception {
        FileSystem fs = this.fileSys.getRawFileSystem();
        Path file = new Path(TEST_ROOT_DIR, "test-append");
        fs.delete(file, true);
        FSDataOutputStream out = fs.create(file);
        try {
            out.write("text1".getBytes());
        }
        finally {
            out.close();
        }
        out = fs.append(file);
        try {
            Assert.assertEquals((long)5L, (long)out.getPos());
            out.write("text2".getBytes());
        }
        finally {
            out.close();
        }
        FSDataInputStream in = fs.open(file);
        try {
            byte[] buf = new byte[in.available()];
            in.readFully(buf);
            Assert.assertEquals((Object)"text1text2", (Object)new String(buf));
        }
        finally {
            in.close();
        }
    }

    @Test
    public void testFileStatusPipeFile() throws Exception {
        RawLocalFileSystem origFs = new RawLocalFileSystem();
        RawLocalFileSystem fs = (RawLocalFileSystem)Mockito.spy((Object)origFs);
        Configuration conf = (Configuration)Mockito.mock(Configuration.class);
        fs.setConf(conf);
        Whitebox.setInternalState((Object)fs, (String)"useDeprecatedFileStatus", (Object)false);
        Path path = new Path("/foo");
        File pipe = (File)Mockito.mock(File.class);
        Mockito.when((Object)pipe.isFile()).thenReturn((Object)false);
        Mockito.when((Object)pipe.isDirectory()).thenReturn((Object)false);
        Mockito.when((Object)pipe.exists()).thenReturn((Object)true);
        FileStatus stat = (FileStatus)Mockito.mock(FileStatus.class);
        ((RawLocalFileSystem)Mockito.doReturn((Object)pipe).when((Object)fs)).pathToFile(path);
        ((RawLocalFileSystem)Mockito.doReturn((Object)stat).when((Object)fs)).getFileStatus(path);
        FileStatus[] stats = fs.listStatus(path);
        Assert.assertTrue((stats != null && stats.length == 1 && stats[0] == stat ? 1 : 0) != 0);
    }
}

