/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.repository.store.graph.v2;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import javax.inject.Inject;
import org.apache.atlas.AtlasErrorCode;
import org.apache.atlas.AtlasException;
import org.apache.atlas.RequestContext;
import org.apache.atlas.exception.AtlasBaseException;
import org.apache.atlas.listener.EntityChangeListener;
import org.apache.atlas.listener.EntityChangeListenerV2;
import org.apache.atlas.model.audit.EntityAuditEventV2;
import org.apache.atlas.model.glossary.AtlasGlossaryTerm;
import org.apache.atlas.model.instance.AtlasClassification;
import org.apache.atlas.model.instance.AtlasEntity;
import org.apache.atlas.model.instance.AtlasEntityHeader;
import org.apache.atlas.model.instance.AtlasRelatedObjectId;
import org.apache.atlas.model.instance.EntityMutationResponse;
import org.apache.atlas.model.instance.EntityMutations;
import org.apache.atlas.repository.Constants;
import org.apache.atlas.repository.converters.AtlasInstanceConverter;
import org.apache.atlas.repository.graph.FullTextMapperV2;
import org.apache.atlas.repository.graph.GraphHelper;
import org.apache.atlas.repository.graphdb.AtlasVertex;
import org.apache.atlas.repository.store.graph.v2.AtlasGraphUtilsV2;
import org.apache.atlas.type.AtlasEntityType;
import org.apache.atlas.type.AtlasTypeRegistry;
import org.apache.atlas.util.AtlasRepositoryConfiguration;
import org.apache.atlas.utils.AtlasPerfMetrics;
import org.apache.atlas.v1.model.instance.Referenceable;
import org.apache.atlas.v1.model.instance.Struct;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class AtlasEntityChangeNotifier {
    private static final Logger LOG = LoggerFactory.getLogger(AtlasEntityChangeNotifier.class);
    private final Set<EntityChangeListener> entityChangeListeners;
    private final Set<EntityChangeListenerV2> entityChangeListenersV2;
    private final AtlasInstanceConverter instanceConverter;
    private final FullTextMapperV2 fullTextMapperV2;
    private final AtlasTypeRegistry atlasTypeRegistry;
    private final boolean isV2EntityNotificationEnabled;

    @Inject
    public AtlasEntityChangeNotifier(Set<EntityChangeListener> entityChangeListeners, Set<EntityChangeListenerV2> entityChangeListenersV2, AtlasInstanceConverter instanceConverter, FullTextMapperV2 fullTextMapperV2, AtlasTypeRegistry atlasTypeRegistry) {
        this.entityChangeListeners = entityChangeListeners;
        this.entityChangeListenersV2 = entityChangeListenersV2;
        this.instanceConverter = instanceConverter;
        this.fullTextMapperV2 = fullTextMapperV2;
        this.atlasTypeRegistry = atlasTypeRegistry;
        this.isV2EntityNotificationEnabled = AtlasRepositoryConfiguration.isV2EntityNotificationEnabled();
    }

    public void onEntitiesMutated(EntityMutationResponse entityMutationResponse, boolean isImport) throws AtlasBaseException {
        if (CollectionUtils.isEmpty(this.entityChangeListeners) || this.instanceConverter == null) {
            return;
        }
        this.pruneResponse(entityMutationResponse);
        List createdEntities = entityMutationResponse.getCreatedEntities();
        List updatedEntities = entityMutationResponse.getUpdatedEntities();
        List partiallyUpdatedEntities = entityMutationResponse.getPartialUpdatedEntities();
        List deletedEntities = entityMutationResponse.getDeletedEntities();
        this.doFullTextMapping(createdEntities);
        this.doFullTextMapping(updatedEntities);
        this.doFullTextMapping(partiallyUpdatedEntities);
        this.notifyListeners(createdEntities, EntityMutations.EntityOperation.CREATE, isImport);
        this.notifyListeners(updatedEntities, EntityMutations.EntityOperation.UPDATE, isImport);
        this.notifyListeners(partiallyUpdatedEntities, EntityMutations.EntityOperation.PARTIAL_UPDATE, isImport);
        this.notifyListeners(deletedEntities, EntityMutations.EntityOperation.DELETE, isImport);
        this.notifyPropagatedEntities();
    }

    public void onClassificationAddedToEntity(AtlasEntity entity, List<AtlasClassification> addedClassifications) throws AtlasBaseException {
        if (this.isV2EntityNotificationEnabled) {
            this.doFullTextMapping(entity.getGuid());
            for (EntityChangeListenerV2 listener : this.entityChangeListenersV2) {
                listener.onClassificationsAdded(entity, addedClassifications);
            }
        } else {
            this.updateFullTextMapping(entity.getGuid(), addedClassifications);
            Referenceable entityRef = this.toReferenceable(entity.getGuid());
            List<Struct> traits = this.toStruct(addedClassifications);
            if (entity == null || CollectionUtils.isEmpty(traits)) {
                return;
            }
            for (EntityChangeListener listener : this.entityChangeListeners) {
                try {
                    listener.onTraitsAdded(entityRef, traits);
                }
                catch (AtlasException e) {
                    throw new AtlasBaseException(AtlasErrorCode.NOTIFICATION_FAILED, (Throwable)e, new String[]{this.getListenerName(listener), "TraitAdd"});
                }
            }
        }
    }

    public void onClassificationUpdatedToEntity(AtlasEntity entity, List<AtlasClassification> updatedClassifications) throws AtlasBaseException {
        if (this.isV2EntityNotificationEnabled) {
            this.doFullTextMapping(entity.getGuid());
            for (EntityChangeListenerV2 listener : this.entityChangeListenersV2) {
                listener.onClassificationsUpdated(entity, updatedClassifications);
            }
        } else {
            this.doFullTextMapping(entity.getGuid());
            Referenceable entityRef = this.toReferenceable(entity.getGuid());
            List<Struct> traits = this.toStruct(updatedClassifications);
            if (entityRef == null || CollectionUtils.isEmpty(traits)) {
                return;
            }
            for (EntityChangeListener listener : this.entityChangeListeners) {
                try {
                    listener.onTraitsUpdated(entityRef, traits);
                }
                catch (AtlasException e) {
                    throw new AtlasBaseException(AtlasErrorCode.NOTIFICATION_FAILED, (Throwable)e, new String[]{this.getListenerName(listener), "TraitUpdate"});
                }
            }
        }
    }

    public void onClassificationDeletedFromEntity(AtlasEntity entity, List<AtlasClassification> deletedClassifications) throws AtlasBaseException {
        if (this.isV2EntityNotificationEnabled) {
            this.doFullTextMapping(entity.getGuid());
            for (EntityChangeListenerV2 listener : this.entityChangeListenersV2) {
                listener.onClassificationsDeleted(entity, deletedClassifications);
            }
        } else {
            this.doFullTextMapping(entity.getGuid());
            Referenceable entityRef = this.toReferenceable(entity.getGuid());
            List<Struct> traits = this.toStruct(deletedClassifications);
            if (entityRef == null || CollectionUtils.isEmpty(deletedClassifications)) {
                return;
            }
            for (EntityChangeListener listener : this.entityChangeListeners) {
                try {
                    listener.onTraitsDeleted(entityRef, traits);
                }
                catch (AtlasException e) {
                    throw new AtlasBaseException(AtlasErrorCode.NOTIFICATION_FAILED, (Throwable)e, new String[]{this.getListenerName(listener), "TraitDelete"});
                }
            }
        }
    }

    public void onTermAddedToEntities(AtlasGlossaryTerm term, List<AtlasRelatedObjectId> entityIds) throws AtlasBaseException {
        if (this.isV2EntityNotificationEnabled) {
            for (EntityChangeListenerV2 listener : this.entityChangeListenersV2) {
                listener.onTermAdded(term, entityIds);
            }
        } else {
            List<Referenceable> entityRefs = this.toReferenceables(entityIds);
            for (EntityChangeListener listener : this.entityChangeListeners) {
                try {
                    listener.onTermAdded(entityRefs, term);
                }
                catch (AtlasException e) {
                    throw new AtlasBaseException(AtlasErrorCode.NOTIFICATION_FAILED, (Throwable)e, new String[]{this.getListenerName(listener), "TermAdd"});
                }
            }
        }
    }

    public void onTermDeletedFromEntities(AtlasGlossaryTerm term, List<AtlasRelatedObjectId> entityIds) throws AtlasBaseException {
        if (this.isV2EntityNotificationEnabled) {
            for (EntityChangeListenerV2 listener : this.entityChangeListenersV2) {
                listener.onTermDeleted(term, entityIds);
            }
        } else {
            List<Referenceable> entityRefs = this.toReferenceables(entityIds);
            for (EntityChangeListener listener : this.entityChangeListeners) {
                try {
                    listener.onTermDeleted(entityRefs, term);
                }
                catch (AtlasException e) {
                    throw new AtlasBaseException(AtlasErrorCode.NOTIFICATION_FAILED, (Throwable)e, new String[]{this.getListenerName(listener), "TermDelete"});
                }
            }
        }
    }

    public void notifyPropagatedEntities() throws AtlasBaseException {
        RequestContext context = RequestContext.get();
        Map addedPropagations = context.getAddedPropagations();
        Map removedPropagations = context.getRemovedPropagations();
        this.notifyPropagatedEntities(addedPropagations, EntityAuditEventV2.EntityAuditActionV2.PROPAGATED_CLASSIFICATION_ADD);
        this.notifyPropagatedEntities(removedPropagations, EntityAuditEventV2.EntityAuditActionV2.PROPAGATED_CLASSIFICATION_DELETE);
    }

    private void notifyPropagatedEntities(Map<String, List<AtlasClassification>> entityPropagationMap, EntityAuditEventV2.EntityAuditActionV2 action) throws AtlasBaseException {
        if (MapUtils.isEmpty(entityPropagationMap) || action == null) {
            return;
        }
        RequestContext context = RequestContext.get();
        for (String guid : entityPropagationMap.keySet()) {
            AtlasEntity entity;
            if (context.isDeletedEntity(guid) || (entity = this.instanceConverter.getAndCacheEntity(guid)) == null) continue;
            if (action == EntityAuditEventV2.EntityAuditActionV2.PROPAGATED_CLASSIFICATION_ADD) {
                this.onClassificationAddedToEntity(entity, entityPropagationMap.get(guid));
                continue;
            }
            if (action != EntityAuditEventV2.EntityAuditActionV2.PROPAGATED_CLASSIFICATION_DELETE) continue;
            this.onClassificationDeletedFromEntity(entity, entityPropagationMap.get(guid));
        }
    }

    private String getListenerName(EntityChangeListener listener) {
        return listener.getClass().getSimpleName();
    }

    private void notifyListeners(List<AtlasEntityHeader> entityHeaders, EntityMutations.EntityOperation operation, boolean isImport) throws AtlasBaseException {
        if (CollectionUtils.isEmpty(entityHeaders)) {
            return;
        }
        if (this.isV2EntityNotificationEnabled) {
            this.notifyV2Listeners(entityHeaders, operation, isImport);
        } else {
            this.notifyV1Listeners(entityHeaders, operation, isImport);
        }
    }

    private void notifyV1Listeners(List<AtlasEntityHeader> entityHeaders, EntityMutations.EntityOperation operation, boolean isImport) throws AtlasBaseException {
        List<Referenceable> typedRefInsts = this.toReferenceables(entityHeaders, operation);
        for (EntityChangeListener listener : this.entityChangeListeners) {
            try {
                switch (operation) {
                    case CREATE: {
                        listener.onEntitiesAdded(typedRefInsts, isImport);
                        break;
                    }
                    case UPDATE: 
                    case PARTIAL_UPDATE: {
                        listener.onEntitiesUpdated(typedRefInsts, isImport);
                        break;
                    }
                    case DELETE: {
                        listener.onEntitiesDeleted(typedRefInsts, isImport);
                    }
                }
            }
            catch (AtlasException e) {
                throw new AtlasBaseException(AtlasErrorCode.NOTIFICATION_FAILED, (Throwable)e, new String[]{this.getListenerName(listener), operation.toString()});
            }
        }
    }

    private void notifyV2Listeners(List<AtlasEntityHeader> entityHeaders, EntityMutations.EntityOperation operation, boolean isImport) throws AtlasBaseException {
        List<AtlasEntity> entities = this.toAtlasEntities(entityHeaders, operation);
        for (EntityChangeListenerV2 listener : this.entityChangeListenersV2) {
            switch (operation) {
                case CREATE: {
                    listener.onEntitiesAdded(entities, isImport);
                    break;
                }
                case UPDATE: 
                case PARTIAL_UPDATE: {
                    listener.onEntitiesUpdated(entities, isImport);
                    break;
                }
                case DELETE: {
                    listener.onEntitiesDeleted(entities, isImport);
                }
            }
        }
    }

    private List<Referenceable> toReferenceables(List<AtlasEntityHeader> entityHeaders, EntityMutations.EntityOperation operation) throws AtlasBaseException {
        ArrayList<Referenceable> ret = new ArrayList<Referenceable>(entityHeaders.size());
        if (operation == EntityMutations.EntityOperation.DELETE) {
            for (AtlasEntityHeader entityHeader : entityHeaders) {
                ret.add(new Referenceable(entityHeader.getGuid(), entityHeader.getTypeName(), entityHeader.getAttributes()));
            }
        } else {
            for (AtlasEntityHeader entityHeader : entityHeaders) {
                ret.add(this.toReferenceable(entityHeader.getGuid()));
            }
        }
        return ret;
    }

    private List<Referenceable> toReferenceables(List<AtlasRelatedObjectId> entityIds) throws AtlasBaseException {
        ArrayList<Referenceable> ret = new ArrayList<Referenceable>();
        if (CollectionUtils.isNotEmpty(entityIds)) {
            for (AtlasRelatedObjectId relatedObjectId : entityIds) {
                String entityGuid = relatedObjectId.getGuid();
                ret.add(this.toReferenceable(entityGuid));
            }
        }
        return ret;
    }

    private Referenceable toReferenceable(String entityId) throws AtlasBaseException {
        Referenceable ret = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty((CharSequence)entityId)) {
            ret = this.instanceConverter.getReferenceable(entityId);
        }
        return ret;
    }

    private List<Struct> toStruct(List<AtlasClassification> classifications) throws AtlasBaseException {
        ArrayList<Struct> ret = null;
        if (classifications != null) {
            ret = new ArrayList<Struct>(classifications.size());
            for (AtlasClassification classification : classifications) {
                if (classification == null) continue;
                ret.add(this.instanceConverter.getTrait(classification));
            }
        }
        return ret;
    }

    private List<AtlasEntity> toAtlasEntities(List<AtlasEntityHeader> entityHeaders, EntityMutations.EntityOperation operation) throws AtlasBaseException {
        ArrayList<AtlasEntity> ret = new ArrayList<AtlasEntity>();
        if (CollectionUtils.isNotEmpty(entityHeaders)) {
            for (AtlasEntityHeader entityHeader : entityHeaders) {
                AtlasEntity entity;
                String typeName = entityHeader.getTypeName();
                AtlasEntityType entityType = this.atlasTypeRegistry.getEntityTypeByName(typeName);
                if (entityType == null) continue;
                if (entityType.isInternalType()) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("Skipping internal type = {}", (Object)typeName);
                    continue;
                }
                if (operation == EntityMutations.EntityOperation.DELETE) {
                    entity = new AtlasEntity(entityHeader);
                } else {
                    String entityGuid = entityHeader.getGuid();
                    entity = this.instanceConverter.getAndCacheEntity(entityGuid);
                }
                if (entity == null) continue;
                ret.add(entity);
            }
        }
        return ret;
    }

    private void doFullTextMapping(List<AtlasEntityHeader> entityHeaders) {
        if (CollectionUtils.isEmpty(entityHeaders)) {
            return;
        }
        try {
            if (!AtlasRepositoryConfiguration.isFullTextSearchEnabled()) {
                return;
            }
        }
        catch (AtlasException e) {
            LOG.warn("Unable to determine if FullText is disabled. Proceeding with FullText mapping");
        }
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("fullTextMapping");
        for (AtlasEntityHeader entityHeader : entityHeaders) {
            String guid;
            AtlasVertex vertex;
            if (GraphHelper.isInternalType(entityHeader.getTypeName()) || (vertex = AtlasGraphUtilsV2.findByGuid(guid = entityHeader.getGuid())) == null) continue;
            try {
                String fullText = this.fullTextMapperV2.getIndexTextForEntity(guid);
                AtlasGraphUtilsV2.setEncodedProperty(vertex, Constants.ENTITY_TEXT_PROPERTY_KEY, fullText);
            }
            catch (AtlasBaseException e) {
                LOG.error("FullText mapping failed for Vertex[ guid = {} ]", (Object)guid, (Object)e);
            }
        }
        RequestContext.get().endMetricRecord(metric);
    }

    private void updateFullTextMapping(String entityId, List<AtlasClassification> classifications) {
        try {
            if (!AtlasRepositoryConfiguration.isFullTextSearchEnabled()) {
                return;
            }
        }
        catch (AtlasException e) {
            LOG.warn("Unable to determine if FullText is disabled. Proceeding with FullText mapping");
        }
        if (org.apache.commons.lang3.StringUtils.isEmpty((CharSequence)entityId) || CollectionUtils.isEmpty(classifications)) {
            return;
        }
        AtlasVertex atlasVertex = AtlasGraphUtilsV2.findByGuid(entityId);
        if (atlasVertex == null || GraphHelper.isInternalType(atlasVertex)) {
            return;
        }
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("fullTextMapping");
        try {
            String classificationFullText = this.fullTextMapperV2.getIndexTextForClassifications(entityId, classifications);
            String existingFullText = AtlasGraphUtilsV2.getEncodedProperty(atlasVertex, Constants.ENTITY_TEXT_PROPERTY_KEY, String.class);
            String newFullText = existingFullText + " " + classificationFullText;
            AtlasGraphUtilsV2.setEncodedProperty(atlasVertex, Constants.ENTITY_TEXT_PROPERTY_KEY, newFullText);
        }
        catch (AtlasBaseException e) {
            LOG.error("FullText mapping failed for Vertex[ guid = {} ]", (Object)entityId, (Object)e);
        }
        RequestContext.get().endMetricRecord(metric);
    }

    private void doFullTextMapping(String guid) {
        AtlasEntityHeader entityHeader = new AtlasEntityHeader();
        entityHeader.setGuid(guid);
        this.doFullTextMapping(Collections.singletonList(entityHeader));
    }

    private void pruneResponse(EntityMutationResponse resp) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> pruneResponse()");
        }
        List createdEntities = resp.getCreatedEntities();
        List updatedEntities = resp.getUpdatedEntities();
        List partialUpdatedEntities = resp.getPartialUpdatedEntities();
        List deletedEntities = resp.getDeletedEntities();
        this.purgeDeletedEntities(createdEntities);
        this.purgeDeletedEntities(updatedEntities);
        this.purgeDeletedEntities(partialUpdatedEntities);
        if (deletedEntities != null) {
            for (AtlasEntityHeader entity : deletedEntities) {
                this.purgeEntity(entity.getGuid(), createdEntities);
                this.purgeEntity(entity.getGuid(), updatedEntities);
                this.purgeEntity(entity.getGuid(), partialUpdatedEntities);
            }
        }
        if (createdEntities != null) {
            for (AtlasEntityHeader entity : createdEntities) {
                this.purgeEntity(entity.getGuid(), updatedEntities);
                this.purgeEntity(entity.getGuid(), partialUpdatedEntities);
            }
        }
        if (updatedEntities != null) {
            for (AtlasEntityHeader entity : updatedEntities) {
                this.purgeEntity(entity.getGuid(), partialUpdatedEntities);
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("<== pruneResponse()");
        }
    }

    private void purgeDeletedEntities(List<AtlasEntityHeader> entities) {
        if (entities != null) {
            ListIterator<AtlasEntityHeader> iter = entities.listIterator();
            while (iter.hasNext()) {
                AtlasEntityHeader entity = iter.next();
                if (entity.getStatus() != AtlasEntity.Status.DELETED) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("purgeDeletedEntities(guid={}, status={}): REMOVED", (Object)entity.getGuid(), (Object)entity.getStatus());
                }
                iter.remove();
            }
        }
    }

    private void purgeEntity(String guid, List<AtlasEntityHeader> entities) {
        if (guid != null && entities != null) {
            ListIterator<AtlasEntityHeader> iter = entities.listIterator();
            while (iter.hasNext()) {
                AtlasEntityHeader entity = iter.next();
                if (!StringUtils.equals((String)guid, (String)entity.getGuid())) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("purgeEntity(guid={}): REMOVED", (Object)entity.getGuid());
                }
                iter.remove();
            }
        }
    }
}

