/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.logstreams.state;

import io.zeebe.db.ZeebeDb;
import io.zeebe.db.ZeebeDbFactory;
import io.zeebe.distributedlog.restore.snapshot.SnapshotRestoreInfo;
import io.zeebe.distributedlog.restore.snapshot.impl.DefaultSnapshotRestoreInfo;
import io.zeebe.distributedlog.restore.snapshot.impl.NullSnapshotRestoreInfo;
import io.zeebe.logstreams.impl.Loggers;
import io.zeebe.logstreams.impl.delete.DeletionService;
import io.zeebe.logstreams.impl.delete.NoopDeletionService;
import io.zeebe.logstreams.spi.SnapshotController;
import io.zeebe.logstreams.spi.ValidSnapshotListener;
import io.zeebe.logstreams.state.NoneSnapshotReplication;
import io.zeebe.logstreams.state.ReplicationController;
import io.zeebe.logstreams.state.SnapshotReplication;
import io.zeebe.logstreams.state.StateStorage;
import io.zeebe.util.FileUtil;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.List;
import java.util.function.Consumer;
import org.slf4j.Logger;

public class StateSnapshotController
implements SnapshotController,
ValidSnapshotListener {
    private static final Logger LOG = Loggers.SNAPSHOT_LOGGER;
    private static final String ERROR_MSG_ENSURING_MAX_SNAPSHOT_COUNT = "Unexpected exception occurred on ensuring maximum snapshot count.";
    private final StateStorage storage;
    private final ZeebeDbFactory zeebeDbFactory;
    private ZeebeDb db;
    private final ReplicationController replicationController;
    private DeletionService deletionService = new NoopDeletionService();
    private final int maxSnapshotCount;
    private volatile SnapshotRestoreInfo snapshotRestoreInfo = new NullSnapshotRestoreInfo();

    public StateSnapshotController(ZeebeDbFactory rocksDbFactory, StateStorage storage) {
        this(rocksDbFactory, storage, new NoneSnapshotReplication(), 1);
    }

    public StateSnapshotController(ZeebeDbFactory rocksDbFactory, StateStorage storage, int maxSnapshotCount) {
        this(rocksDbFactory, storage, new NoneSnapshotReplication(), maxSnapshotCount);
    }

    public StateSnapshotController(ZeebeDbFactory zeebeDbFactory, StateStorage storage, SnapshotReplication replication, int maxSnapshotCount) {
        this.storage = storage;
        this.zeebeDbFactory = zeebeDbFactory;
        this.maxSnapshotCount = maxSnapshotCount;
        this.replicationController = new ReplicationController(replication, storage, this);
        this.initializeRestoreInfo();
    }

    @Override
    public void takeSnapshot(long lowerBoundSnapshotPosition) {
        if (this.db == null) {
            throw new IllegalStateException("Cannot create snapshot of not open database.");
        }
        File snapshotDir = this.storage.getSnapshotDirectoryFor(lowerBoundSnapshotPosition);
        this.db.createSnapshot(snapshotDir);
        this.snapshotRestoreInfo = new DefaultSnapshotRestoreInfo(lowerBoundSnapshotPosition, snapshotDir.listFiles().length);
    }

    @Override
    public void takeTempSnapshot() {
        if (this.db == null) {
            throw new IllegalStateException("Cannot create snapshot of not open database.");
        }
        File snapshotDir = this.storage.getTempSnapshotDirectory();
        LOG.debug("Take temporary snapshot and write into {}.", (Object)snapshotDir.getAbsolutePath());
        this.db.createSnapshot(snapshotDir);
    }

    @Override
    public void moveValidSnapshot(long lowerBoundSnapshotPosition) throws IOException {
        if (this.db == null) {
            throw new IllegalStateException("Cannot create snapshot of not open database.");
        }
        File previousLocation = this.storage.getTempSnapshotDirectory();
        if (!previousLocation.exists()) {
            throw new IllegalStateException(String.format("Temporary snapshot directory %s does not exist.", previousLocation.getAbsolutePath()));
        }
        File snapshotDir = this.storage.getSnapshotDirectoryFor(lowerBoundSnapshotPosition);
        if (snapshotDir.exists()) {
            return;
        }
        LOG.debug("Snapshot is valid. Move snapshot from {} to {}.", (Object)previousLocation.getAbsolutePath(), (Object)snapshotDir.getAbsolutePath());
        Files.move(previousLocation.toPath(), snapshotDir.toPath(), new CopyOption[0]);
        this.onNewValidSnapshot();
    }

    @Override
    public void replicateLatestSnapshot(Consumer<Runnable> executor) {
        List<File> snapshots = this.storage.listByPositionDesc();
        if (snapshots != null && !snapshots.isEmpty()) {
            File[] files;
            File latestSnapshotDirectory = snapshots.get(0);
            LOG.debug("Start replicating latest snapshot {}", (Object)latestSnapshotDirectory.toPath());
            long snapshotPosition = Long.parseLong(latestSnapshotDirectory.getName());
            for (File snapshotChunkFile : files = latestSnapshotDirectory.listFiles()) {
                executor.accept(() -> {
                    LOG.debug("Replicate snapshot chunk {}", (Object)snapshotChunkFile.toPath());
                    this.replicationController.replicate(snapshotPosition, files.length, snapshotChunkFile);
                });
            }
        }
    }

    @Override
    public void consumeReplicatedSnapshots() {
        this.replicationController.consumeReplicatedSnapshots();
    }

    public void setDeletionService(DeletionService deletionService) {
        this.deletionService = deletionService;
    }

    @Override
    public SnapshotRestoreInfo getLatestSnapshotRestoreInfo() {
        return this.snapshotRestoreInfo;
    }

    @Override
    public long recover() throws Exception {
        File runtimeDirectory = this.storage.getRuntimeDirectory();
        if (runtimeDirectory.exists()) {
            FileUtil.deleteFolder((String)runtimeDirectory.getAbsolutePath());
        }
        List<File> snapshots = this.storage.listByPositionDesc();
        LOG.debug("Available snapshots: {}", snapshots);
        long lowerBoundSnapshotPosition = -1L;
        Iterator<File> snapshotIterator = snapshots.iterator();
        while (snapshotIterator.hasNext() && lowerBoundSnapshotPosition < 0L) {
            File snapshotDirectory = snapshotIterator.next();
            FileUtil.copySnapshot((File)runtimeDirectory, (File)snapshotDirectory);
            try {
                this.openDb();
                LOG.debug("Recovered state from snapshot '{}'", (Object)snapshotDirectory);
                lowerBoundSnapshotPosition = Long.parseLong(snapshotDirectory.getName());
                this.snapshotRestoreInfo = new DefaultSnapshotRestoreInfo(lowerBoundSnapshotPosition, snapshotDirectory.listFiles().length);
            }
            catch (Exception e) {
                FileUtil.deleteFolder((String)runtimeDirectory.getAbsolutePath());
                if (snapshotIterator.hasNext()) {
                    LOG.warn("Failed to open snapshot '{}'. Delete this snapshot and try the previous one.", (Object)snapshotDirectory, (Object)e);
                    FileUtil.deleteFolder((String)snapshotDirectory.getAbsolutePath());
                    continue;
                }
                LOG.error("Failed to open snapshot '{}'. No snapshots available to recover from. Manual action is required.", (Object)snapshotDirectory, (Object)e);
                throw new RuntimeException("Failed to recover from snapshots", e);
            }
        }
        return lowerBoundSnapshotPosition;
    }

    @Override
    public ZeebeDb openDb() {
        if (this.db == null) {
            File runtimeDirectory = this.storage.getRuntimeDirectory();
            this.db = this.zeebeDbFactory.createDb(runtimeDirectory);
            LOG.debug("Opened database from '{}'.", (Object)runtimeDirectory.toPath());
        }
        return this.db;
    }

    @Override
    public void onNewValidSnapshot() {
        try {
            File latestSnapshot = this.getLastValidSnapshotDirectory();
            this.snapshotRestoreInfo = new DefaultSnapshotRestoreInfo(Long.parseLong(latestSnapshot.getName()), latestSnapshot.listFiles().length);
            this.ensureMaxSnapshotCount();
        }
        catch (IOException e) {
            LOG.error(ERROR_MSG_ENSURING_MAX_SNAPSHOT_COUNT, (Throwable)e);
        }
        if (this.getValidSnapshotsCount() >= this.maxSnapshotCount) {
            this.deletionService.delete(this.getPositionToDelete(this.maxSnapshotCount));
        }
    }

    public void ensureMaxSnapshotCount() throws IOException {
        List<File> snapshots = this.storage.listByPositionAsc();
        if (snapshots.size() > this.maxSnapshotCount) {
            int oldestValidSnapshotIndex = snapshots.size() - this.maxSnapshotCount;
            LOG.debug("Ensure max snapshot count {}, will delete {} snapshot(s).", (Object)this.maxSnapshotCount, (Object)oldestValidSnapshotIndex);
            List<File> snapshotsToRemove = snapshots.subList(0, oldestValidSnapshotIndex);
            for (File snapshot : snapshotsToRemove) {
                FileUtil.deleteFolder((Path)snapshot.toPath());
                LOG.debug("Purged snapshot {}", (Object)snapshot);
            }
            this.cleanUpTemporarySnapshots(snapshots, oldestValidSnapshotIndex);
        } else {
            LOG.debug("Tried to ensure max snapshot count {}, nothing to do snapshot count is {}.", (Object)this.maxSnapshotCount, (Object)snapshots.size());
        }
    }

    private void cleanUpTemporarySnapshots(List<File> snapshots, int oldestValidSnapshotIndex) throws IOException {
        File oldestValidSnapshot = snapshots.get(oldestValidSnapshotIndex);
        long oldestValidSnapshotPosition = Long.parseLong(oldestValidSnapshot.getName());
        LOG.debug("Search for orphaned snapshots below oldest valid snapshot position {}", (Object)oldestValidSnapshotPosition);
        List<File> tmpDirectoriesBelowPosition = this.storage.findTmpDirectoriesBelowPosition(oldestValidSnapshotPosition);
        for (File notCompletedSnapshot : tmpDirectoriesBelowPosition) {
            FileUtil.deleteFolder((Path)notCompletedSnapshot.toPath());
            LOG.debug("Delete not completed (orphaned) snapshot {}", (Object)notCompletedSnapshot);
        }
    }

    @Override
    public long getPositionToDelete(int maxSnapshotCount) {
        return this.storage.listByPositionDesc().stream().skip(maxSnapshotCount - 1).findFirst().map(f -> Long.parseLong(f.getName())).orElse(-1L);
    }

    @Override
    public int getValidSnapshotsCount() {
        return this.storage.list().size();
    }

    @Override
    public long getLastValidSnapshotPosition() {
        return this.storage.listByPositionDesc().stream().map(File::getName).mapToLong(Long::parseLong).findFirst().orElse(-1L);
    }

    @Override
    public File getLastValidSnapshotDirectory() {
        List<File> snapshots = this.storage.listByPositionDesc();
        if (snapshots != null && !snapshots.isEmpty()) {
            return snapshots.get(0);
        }
        return null;
    }

    @Override
    public File getSnapshotDirectoryFor(long snapshotId) {
        return this.storage.getSnapshotDirectoryFor(snapshotId);
    }

    @Override
    public void close() throws Exception {
        if (this.db != null) {
            this.db.close();
            File runtimeDirectory = this.storage.getRuntimeDirectory();
            LOG.debug("Closed database from '{}'.", (Object)runtimeDirectory.toPath());
            this.db = null;
        }
    }

    public boolean isDbOpened() {
        return this.db != null;
    }

    private void initializeRestoreInfo() {
        File lastSnapshot = this.getLastValidSnapshotDirectory();
        if (lastSnapshot != null) {
            int numFiles = lastSnapshot.listFiles().length;
            long lastSnapshotPosition = Long.parseLong(lastSnapshot.getName());
            if (lastSnapshotPosition > -1L && numFiles > 0) {
                this.snapshotRestoreInfo = new DefaultSnapshotRestoreInfo(lastSnapshotPosition, numFiles);
            }
        }
    }
}

