/**
 * Licensed under the Artistic License; you may not use this file
 * except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://displaytag.sourceforge.net/license.html
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.displaytag.localization;

import java.util.Locale;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.jsp.PageContext;
import javax.servlet.jsp.tagext.Tag;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.displaytag.Messages;
import org.springframework.context.MessageSource;
import org.springframework.web.servlet.support.RequestContextUtils;


/**
 * Spring implementation of a resource provider and locale resolver. Since Displaytag locale resolution is modelled on
 * the Spring one, it simply forward <code>resolveLocale</code> calls to the Spring-configured LocaleResolver.
 * @author Fabrizio Giustina
 * @version $Revision: 1140 $ ($Author: fgiust $)
 */
public class I18nSpringAdapter implements LocaleResolver, I18nResourceProvider
{

    /**
     * prefix/suffix for missing entries.
     */
    public static final String UNDEFINED_KEY = "???"; //$NON-NLS-1$

    /**
     * logger.
     */
    private static Log log = LogFactory.getLog(I18nSpringAdapter.class);

    /**
     * @see LocaleResolver#resolveLocale(HttpServletRequest)
     */
    public Locale resolveLocale(HttpServletRequest request)
    {
        return RequestContextUtils.getLocale(request);
    }

    /**
     * @see I18nResourceProvider#getResource(String, String, Tag, PageContext)
     */
    public String getResource(String resourceKey, String defaultValue, Tag tag, PageContext pageContext)
    {
        MessageSource messageSource = RequestContextUtils.getWebApplicationContext(pageContext.getRequest(), pageContext.getServletContext());
        if (messageSource == null)
        {
            log.warn("messageSource not found");
            return null;
        }

        // if resourceKey isn't defined either, use defaultValue
        String key = (resourceKey != null) ? resourceKey : defaultValue;

        String message = null;

        message = messageSource.getMessage(key, null, null, RequestContextUtils
            .getLocale((HttpServletRequest) pageContext.getRequest()));

        // if user explicitely added a titleKey we guess this is an error
        if (message == null && resourceKey != null)
        {
            log.debug(Messages.getString("Localization.missingkey", resourceKey)); //$NON-NLS-1$
            message = UNDEFINED_KEY + resourceKey + UNDEFINED_KEY;
        }

        return message;

    }
}
