/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.ows.kvp;

import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.geoserver.platform.ServiceException;
import org.geotools.util.DateRange;
import org.geotools.util.Range;
import org.geotools.util.logging.Logging;

public class TimeParser {
    static final Logger LOGGER = Logging.getLogger(TimeParser.class);
    private final Integer maxTimes;
    static final TimeZone UTC_TZ = TimeZone.getTimeZone("UTC");
    private static final Pattern pattern = Pattern.compile("(back)(\\d+)([hdw])");
    static final long MILLIS_IN_DAY = 86400000L;
    private static final int DEFAULT_MAX_ELEMENTS_TIMES_KVP = 100;

    public TimeParser() {
        this.maxTimes = null;
    }

    public TimeParser(int maxTimes) {
        this.maxTimes = maxTimes;
    }

    public Collection parse(String value) throws ParseException {
        if (value == null) {
            return Collections.emptyList();
        }
        if ((value = value.trim()).length() == 0) {
            return Collections.emptyList();
        }
        TreeSet result = new TreeSet(new Comparator(){

            public int compare(Object o1, Object o2) {
                boolean o1Date = o1 instanceof Date;
                boolean o2Date = o2 instanceof Date;
                if (o1 == o2) {
                    return 0;
                }
                if (o1Date) {
                    Date dateLeft = (Date)o1;
                    if (o2Date) {
                        return dateLeft.compareTo((Date)o2);
                    }
                    return dateLeft.compareTo(((DateRange)o2).getMinValue());
                }
                DateRange left = (DateRange)o1;
                if (o2Date) {
                    return left.getMinValue().compareTo((Date)o2);
                }
                return left.getMinValue().compareTo(((DateRange)o2).getMinValue());
            }
        });
        String[] listDates = value.split(",");
        int maxValues = this.getMaxTimes();
        for (String date : listDates) {
            if (date.indexOf("/") <= 0) {
                Object o = TimeParser.getFuzzyDate(date);
                if (o instanceof Date) {
                    TimeParser.addDate(result, (Date)o);
                } else {
                    TimeParser.addPeriod(result, (DateRange)o);
                }
            } else {
                Date[] range;
                Object[] period = date.split("/");
                if (period.length == 3) {
                    long time;
                    range = TimeParser.parseTimeDuration((String[])period);
                    long millisIncrement = TimeParser.parsePeriod(period[2]);
                    long startTime = range[0].getTime();
                    long endTime = range[1].getTime();
                    int j = 0;
                    while ((time = (long)j * millisIncrement + startTime) <= endTime) {
                        GregorianCalendar calendar = new GregorianCalendar(UTC_TZ);
                        calendar.setTimeInMillis(time);
                        TimeParser.addDate(result, calendar.getTime());
                        ++j;
                        this.checkMaxTimes(result, maxValues);
                    }
                } else if (period.length == 2) {
                    range = TimeParser.parseTimeDuration((String[])period);
                    TimeParser.addPeriod(result, new DateRange(range[0], range[1]));
                } else {
                    throw new ParseException("Invalid time period: " + Arrays.toString(period), 0);
                }
            }
            this.checkMaxTimes(result, maxValues);
        }
        return new ArrayList(result);
    }

    private int getMaxTimes() {
        if (this.maxTimes != null) {
            return this.maxTimes;
        }
        return 100;
    }

    public void checkMaxTimes(Set result, int maxValues) {
        if (maxValues > 0 && result.size() > maxValues) {
            throw new ServiceException("More than " + maxValues + " times specified in the request, bailing out.", "InvalidParameterValue", "time");
        }
    }

    private static Date[] parseTimeDuration(String[] period) throws ParseException {
        Date[] range = null;
        if (period.length == 2 || period.length == 3) {
            Date begin = null;
            Date end = null;
            if (period[0].toUpperCase().startsWith("P") || period[1].toUpperCase().startsWith("P")) {
                long durationOffset = Long.MIN_VALUE;
                if (period[0].toUpperCase().startsWith("P")) {
                    durationOffset = TimeParser.parsePeriod(period[0]);
                } else {
                    begin = TimeParser.beginning(TimeParser.getFuzzyDate(period[0]));
                }
                if (period[1].toUpperCase().startsWith("P") && !period[1].toUpperCase().startsWith("PRESENT")) {
                    if (durationOffset != Long.MIN_VALUE) {
                        throw new ParseException("Invalid time period containing duration with no paired time value: " + Arrays.toString(period), 0);
                    }
                    durationOffset = TimeParser.parsePeriod(period[1]);
                    GregorianCalendar calendar = new GregorianCalendar();
                    calendar.setTimeInMillis(begin.getTime() + durationOffset);
                    end = calendar.getTime();
                } else {
                    end = TimeParser.end(TimeParser.getFuzzyDate(period[1]));
                    GregorianCalendar calendar = new GregorianCalendar();
                    calendar.setTimeInMillis(end.getTime() - durationOffset);
                    begin = calendar.getTime();
                }
            } else {
                begin = TimeParser.beginning(TimeParser.getFuzzyDate(period[0]));
                end = TimeParser.end(TimeParser.getFuzzyDate(period[1]));
            }
            range = new Date[]{begin, end};
        }
        return range;
    }

    private static Date beginning(Object dateOrDateRange) {
        if (dateOrDateRange instanceof DateRange) {
            return ((DateRange)dateOrDateRange).getMinValue();
        }
        return (Date)dateOrDateRange;
    }

    private static Date end(Object dateOrDateRange) {
        if (dateOrDateRange instanceof DateRange) {
            return ((DateRange)dateOrDateRange).getMaxValue();
        }
        return (Date)dateOrDateRange;
    }

    private static void addPeriod(Collection result, DateRange newRange) {
        Iterator it = result.iterator();
        while (it.hasNext()) {
            Date local;
            Object element = it.next();
            if (element instanceof Date) {
                local = (Date)element;
                if (!newRange.contains((Comparable)local)) continue;
                it.remove();
                continue;
            }
            local = (DateRange)element;
            if (local.contains((Range)newRange)) {
                return;
            }
            if (!newRange.contains((Range)local)) continue;
            it.remove();
        }
        result.add(newRange);
    }

    private static void addDate(Collection result, Date newDate) {
        for (Object element : result) {
            if (!(element instanceof Date ? newDate.equals(element) : ((DateRange)element).contains((Comparable)newDate))) continue;
            return;
        }
        result.add(newDate);
    }

    static Object getFuzzyDate(String value) throws ParseException {
        String computedValue = value;
        if (computedValue.equalsIgnoreCase("current") || computedValue.equalsIgnoreCase("now")) {
            return null;
        }
        if (computedValue.equalsIgnoreCase("present")) {
            Calendar now = Calendar.getInstance();
            now.set(14, 0);
            computedValue = FormatAndPrecision.MILLISECOND.getFormat().format(now.getTime());
        }
        for (FormatAndPrecision f : FormatAndPrecision.values()) {
            ParsePosition pos = new ParsePosition(0);
            Date time = f.getFormat().parse(computedValue, pos);
            if (pos.getIndex() != computedValue.length()) continue;
            DateRange range = f.expand(time);
            if (range.getMinValue().equals(range.getMaxValue())) {
                return range.getMinValue();
            }
            return range;
        }
        throw new ParseException("Invalid date: " + value, 0);
    }

    public static long parsePeriod(String period) throws ParseException {
        int length = period.length();
        if (length != 0 && Character.toUpperCase(period.charAt(0)) != 'P') {
            throw new ParseException("Invalid period increment given: " + period, 0);
        }
        long millis = 0L;
        boolean time = false;
        int lower = 0;
        while (++lower < length) {
            double factor;
            double value;
            int upper;
            block17: {
                char letter;
                block16: {
                    letter = Character.toUpperCase(period.charAt(lower));
                    if (letter == 'T') {
                        time = true;
                        if (++lower >= length) break;
                    }
                    upper = lower;
                    letter = period.charAt(upper);
                    while (!Character.isLetter(letter) || letter == 'e' || letter == 'E') {
                        if (++upper >= length) {
                            throw new ParseException("Missing symbol in \"" + period + "\".", lower);
                        }
                        letter = period.charAt(upper);
                    }
                    letter = Character.toUpperCase(letter);
                    value = Double.parseDouble(period.substring(lower, upper));
                    if (!time) break block16;
                    switch (letter) {
                        case 'S': {
                            factor = 1000.0;
                            break block17;
                        }
                        case 'M': {
                            factor = 60000.0;
                            break block17;
                        }
                        case 'H': {
                            factor = 3600000.0;
                            break block17;
                        }
                        default: {
                            throw new ParseException("Unknown time symbol: " + letter, upper);
                        }
                    }
                }
                switch (letter) {
                    case 'D': {
                        factor = 8.64E7;
                        break;
                    }
                    case 'W': {
                        factor = 6.048E8;
                        break;
                    }
                    case 'M': {
                        factor = 2.592E9;
                        break;
                    }
                    case 'Y': {
                        factor = 3.15576E10;
                        break;
                    }
                    default: {
                        throw new ParseException("Unknown period symbol: " + letter, upper);
                    }
                }
            }
            millis += Math.round(value * factor);
            lower = upper;
        }
        return millis;
    }

    private static enum FormatAndPrecision {
        MILLISECOND("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", 14),
        SECOND("yyyy-MM-dd'T'HH:mm:ss'Z'", 13),
        MINUTE("yyyy-MM-dd'T'HH:mm'Z'", 12),
        HOUR("yyyy-MM-dd'T'HH'Z'", 11),
        DAY("yyyy-MM-dd", 5),
        MONTH("yyyy-MM", 2),
        YEAR("yyyy", 1);

        public final String format;
        public final int precision;

        private FormatAndPrecision(String format, int precision) {
            this.format = format;
            this.precision = precision;
        }

        public SimpleDateFormat getFormat() {
            SimpleDateFormat sdf = new SimpleDateFormat(this.format);
            sdf.setTimeZone(UTC_TZ);
            return sdf;
        }

        public DateRange expand(Date d) {
            GregorianCalendar c = new GregorianCalendar(UTC_TZ);
            c.setTime(d);
            ((Calendar)c).add(this.precision, 1);
            ((Calendar)c).add(14, -1);
            return new DateRange(d, c.getTime());
        }
    }
}

