/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.cloud;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.solr.client.solrj.impl.HttpSolrServer;
import org.apache.solr.client.solrj.request.CoreAdminRequest;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.ClusterState;
import org.apache.solr.common.cloud.ClusterStateUtil;
import org.apache.solr.common.cloud.DocCollection;
import org.apache.solr.common.cloud.Replica;
import org.apache.solr.common.cloud.Slice;
import org.apache.solr.common.cloud.ZkStateReader;
import org.apache.solr.core.ConfigSolr;
import org.apache.solr.update.UpdateShardHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OverseerAutoReplicaFailoverThread
implements Runnable,
Closeable {
    private static Logger log = LoggerFactory.getLogger(OverseerAutoReplicaFailoverThread.class);
    private Integer lastClusterStateVersion;
    private final ExecutorService updateExecutor;
    private volatile boolean isClosed;
    private ZkStateReader zkStateReader;
    private final Cache<String, Long> baseUrlForBadNodes;
    private final int workLoopDelay;
    private final int waitAfterExpiration;

    public OverseerAutoReplicaFailoverThread(ConfigSolr config, ZkStateReader zkStateReader, UpdateShardHandler updateShardHandler) {
        this.zkStateReader = zkStateReader;
        this.workLoopDelay = config.getAutoReplicaFailoverWorkLoopDelay();
        this.waitAfterExpiration = config.getAutoReplicaFailoverWaitAfterExpiration();
        int badNodeExpiration = config.getAutoReplicaFailoverBadNodeExpiration();
        log.info("Starting " + this.getClass().getSimpleName() + " autoReplicaFailoverWorkLoopDelay={} autoReplicaFailoverWaitAfterExpiration={} autoReplicaFailoverBadNodeExpiration={}", this.workLoopDelay, this.waitAfterExpiration, badNodeExpiration);
        this.baseUrlForBadNodes = CacheBuilder.newBuilder().concurrencyLevel(1).expireAfterWrite(badNodeExpiration, TimeUnit.MILLISECONDS).build();
        this.updateExecutor = updateShardHandler.getUpdateExecutor();
    }

    @Override
    public void run() {
        while (!this.isClosed) {
            log.debug("do " + this.getClass().getSimpleName() + " work loop");
            try {
                this.doWork();
            }
            catch (Exception e) {
                SolrException.log(log, this.getClass().getSimpleName() + " had an error it's thread work loop.", e);
            }
            if (this.isClosed) continue;
            try {
                Thread.sleep(this.workLoopDelay);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private void doWork() {
        ClusterState clusterState = this.zkStateReader.getClusterState();
        if (clusterState != null) {
            if (this.lastClusterStateVersion == clusterState.getZkClusterStateVersion() && this.baseUrlForBadNodes.size() == 0L) {
                return;
            }
            this.lastClusterStateVersion = clusterState.getZkClusterStateVersion();
            Set<String> collections = clusterState.getCollections();
            for (String collection : collections) {
                DocCollection docCollection = clusterState.getCollection(collection);
                if (!docCollection.getAutoAddReplicas()) continue;
                if (docCollection.getReplicationFactor() == null) {
                    log.debug("Skipping collection because it has no defined replicationFactor, name={}", (Object)docCollection.getName());
                    continue;
                }
                log.debug("Found collection, name={} replicationFactor=", (Object)collection, (Object)docCollection.getReplicationFactor());
                Collection<Slice> slices = docCollection.getSlices();
                for (Slice slice : slices) {
                    if (!slice.getState().equals(Slice.ACTIVE)) continue;
                    ArrayList<DownReplica> downReplicas = new ArrayList<DownReplica>();
                    int goodReplicas = OverseerAutoReplicaFailoverThread.findDownReplicasInSlice(clusterState, docCollection, slice, downReplicas);
                    log.debug("replicationFactor={} goodReplicaCount={}", (Object)docCollection.getReplicationFactor(), (Object)goodReplicas);
                    if (downReplicas.size() > 0 && goodReplicas < docCollection.getReplicationFactor()) {
                        this.processBadReplicas(collection, downReplicas);
                        continue;
                    }
                    if (goodReplicas <= docCollection.getReplicationFactor()) continue;
                    log.debug("There are too many replicas");
                }
            }
        }
    }

    private void processBadReplicas(String collection, Collection<DownReplica> badReplicas) {
        for (DownReplica badReplica : badReplicas) {
            log.debug("process down replica {}", (Object)badReplica.replica.getName());
            String baseUrl = badReplica.replica.getStr("base_url");
            Long wentBadAtNS = this.baseUrlForBadNodes.getIfPresent(baseUrl);
            if (wentBadAtNS == null) {
                log.warn("Replica {} may need to failover.", (Object)badReplica.replica.getName());
                this.baseUrlForBadNodes.put(baseUrl, System.nanoTime());
                continue;
            }
            long elasped = System.nanoTime() - wentBadAtNS;
            if (elasped < TimeUnit.NANOSECONDS.convert(this.waitAfterExpiration, TimeUnit.MILLISECONDS)) {
                log.debug("Looks troublesome...continue. Elapsed={}", (Object)(elasped + "ns"));
                continue;
            }
            log.debug("We need to add a replica. Elapsed={}", (Object)(elasped + "ns"));
            if (!this.addReplica(collection, badReplica)) continue;
            this.baseUrlForBadNodes.invalidate(baseUrl);
        }
    }

    private boolean addReplica(final String collection, DownReplica badReplica) {
        final String createUrl = OverseerAutoReplicaFailoverThread.getBestCreateUrl(this.zkStateReader, badReplica);
        if (createUrl == null) {
            log.warn("Could not find a node to create new replica on.");
            return false;
        }
        final String dataDir = badReplica.replica.getStr("dataDir");
        final String ulogDir = badReplica.replica.getStr("ulogDir");
        final String coreNodeName = badReplica.replica.getName();
        if (dataDir != null) {
            final String coreName = badReplica.replica.getStr("core");
            log.debug("submit call to {}", (Object)createUrl);
            this.updateExecutor.submit(new Callable<Boolean>(){

                @Override
                public Boolean call() {
                    return OverseerAutoReplicaFailoverThread.this.createSolrCore(collection, createUrl, dataDir, ulogDir, coreNodeName, coreName);
                }
            });
            boolean success = ClusterStateUtil.waitToSeeLive(this.zkStateReader, collection, coreNodeName, createUrl, 30);
            if (!success) {
                log.error("Creating new replica appears to have failed, timed out waiting to see created SolrCore register in the clusterstate.");
                return false;
            }
            return true;
        }
        log.warn("Could not find dataDir or ulogDir in cluster state.");
        return false;
    }

    private static int findDownReplicasInSlice(ClusterState clusterState, DocCollection collection, Slice slice, Collection<DownReplica> badReplicas) {
        int goodReplicas = 0;
        Collection<Replica> replicas = slice.getReplicas();
        if (replicas != null) {
            for (Replica replica : replicas) {
                boolean live = clusterState.liveNodesContain(replica.getNodeName());
                String state = replica.getStr("state");
                boolean okayState = state.equals("down") || state.equals("recovering") || state.equals("active");
                log.debug("Process replica name={} live={} state={}", replica.getName(), live, state);
                if (live && okayState) {
                    ++goodReplicas;
                    continue;
                }
                DownReplica badReplica = new DownReplica();
                badReplica.replica = replica;
                badReplica.slice = slice;
                badReplica.collection = collection;
                badReplicas.add(badReplica);
            }
        }
        log.debug("bad replicas for slice {}", (Object)badReplicas);
        return goodReplicas;
    }

    static String getBestCreateUrl(ZkStateReader zkStateReader, DownReplica badReplica) {
        assert (badReplica != null);
        assert (badReplica.collection != null);
        assert (badReplica.slice != null);
        HashMap<String, Counts> counts = new HashMap<String, Counts>();
        ValueComparator vc = new ValueComparator(counts);
        HashSet<String> liveNodes = new HashSet<String>(zkStateReader.getClusterState().getLiveNodes());
        ClusterState clusterState = zkStateReader.getClusterState();
        if (clusterState != null) {
            Set<String> collections = clusterState.getCollections();
            for (String collection : collections) {
                log.debug("look at collection {} as possible create candidate", (Object)collection);
                DocCollection docCollection = clusterState.getCollection(collection);
                Collection<Slice> slices = docCollection.getSlices();
                for (Slice slice : slices) {
                    if (!slice.getState().equals(Slice.ACTIVE)) continue;
                    log.debug("look at slice {} as possible create candidate", (Object)slice.getName());
                    Collection<Replica> replicas = slice.getReplicas();
                    for (Replica replica : replicas) {
                        boolean alreadyExistsOnNode;
                        Slice s;
                        liveNodes.remove(replica.getNodeName());
                        if (replica.getStr("base_url").equals(badReplica.replica.getStr("base_url"))) continue;
                        String baseUrl = replica.getStr("base_url");
                        log.debug("nodename={} livenodes={}", (Object)replica.getNodeName(), (Object)clusterState.getLiveNodes());
                        boolean live = clusterState.liveNodesContain(replica.getNodeName());
                        log.debug("look at replica {} as possible create candidate, live={}", (Object)replica.getName(), (Object)live);
                        if (!live) continue;
                        Counts cnt = (Counts)counts.get(baseUrl);
                        if (cnt == null) {
                            cnt = new Counts();
                        }
                        if (badReplica.collection.getName().equals(collection)) {
                            cnt.negRankingWeight += 3;
                            ++cnt.collectionShardsOnNode;
                        } else {
                            ++cnt.negRankingWeight;
                        }
                        if (badReplica.collection.getName().equals(collection) && badReplica.slice.getName().equals(slice.getName())) {
                            ++cnt.ourReplicas;
                        }
                        int maxShardsPerNode = docCollection.getMaxShardsPerNode();
                        log.debug("max shards per node={} good replicas={}", (Object)maxShardsPerNode, (Object)cnt);
                        Collection<Replica> badSliceReplicas = null;
                        DocCollection c = clusterState.getCollection(badReplica.collection.getName());
                        if (c != null && (s = c.getSlice(badReplica.slice.getName())) != null) {
                            badSliceReplicas = s.getReplicas();
                        }
                        if ((alreadyExistsOnNode = OverseerAutoReplicaFailoverThread.replicaAlreadyExistsOnNode(zkStateReader.getClusterState(), badSliceReplicas, badReplica, baseUrl)) || cnt.collectionShardsOnNode >= maxShardsPerNode) {
                            counts.remove(replica.getStr("base_url"));
                            continue;
                        }
                        counts.put(replica.getStr("base_url"), cnt);
                    }
                }
            }
        }
        for (String node : liveNodes) {
            counts.put(zkStateReader.getBaseUrlForNodeName(node), new Counts(0, 0));
        }
        if (counts.size() == 0) {
            return null;
        }
        TreeMap<String, Counts> sortedCounts = new TreeMap<String, Counts>(vc);
        sortedCounts.putAll(counts);
        log.debug("empty nodes={}", (Object)liveNodes);
        log.debug("sorted hosts={}", (Object)sortedCounts);
        return (String)sortedCounts.keySet().iterator().next();
    }

    private static boolean replicaAlreadyExistsOnNode(ClusterState clusterState, Collection<Replica> replicas, DownReplica badReplica, String baseUrl) {
        if (replicas != null) {
            log.debug("check if replica already exists on node using replicas {}", OverseerAutoReplicaFailoverThread.getNames(replicas));
            for (Replica replica : replicas) {
                if (replica.getName().equals(badReplica.replica.getName()) || !replica.getStr("base_url").equals(baseUrl) || !clusterState.liveNodesContain(replica.getNodeName()) || !replica.getStr("state").equals("active") && !replica.getStr("state").equals("down") && !replica.getStr("state").equals("recovering")) continue;
                log.debug("replica already exists on node, bad replica={}, existing replica={}, node name={}", badReplica.replica.getName(), replica.getName(), replica.getNodeName());
                return true;
            }
        }
        log.debug("replica does not yet exist on node: {}", (Object)baseUrl);
        return false;
    }

    private static Object getNames(Collection<Replica> replicas) {
        HashSet<String> names = new HashSet<String>(replicas.size());
        for (Replica replica : replicas) {
            names.add(replica.getName());
        }
        return names;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean createSolrCore(String collection, String createUrl, String dataDir, String ulogDir, String coreNodeName, String coreName) {
        HttpSolrServer server = null;
        try {
            log.debug("create url={}", (Object)createUrl);
            server = new HttpSolrServer(createUrl);
            server.setConnectionTimeout(30000);
            server.setSoTimeout(60000);
            CoreAdminRequest.Create createCmd = new CoreAdminRequest.Create();
            createCmd.setCollection(collection);
            createCmd.setCoreNodeName(coreNodeName);
            createCmd.setCoreName(coreName);
            createCmd.setDataDir(dataDir);
            createCmd.setUlogDir(ulogDir);
            server.request(createCmd);
        }
        catch (Exception e) {
            SolrException.log(log, "Exception trying to create new replica on " + createUrl, e);
            boolean bl = false;
            return bl;
        }
        finally {
            if (server != null) {
                server.shutdown();
            }
        }
        return true;
    }

    @Override
    public void close() {
        this.isClosed = true;
    }

    public boolean isClosed() {
        return this.isClosed;
    }

    static class DownReplica {
        Replica replica;
        Slice slice;
        DocCollection collection;

        DownReplica() {
        }

        public String toString() {
            return "DownReplica [replica=" + this.replica.getName() + ", slice=" + this.slice.getName() + ", collection=" + this.collection.getName() + "]";
        }
    }

    private static class Counts {
        int collectionShardsOnNode = 0;
        int negRankingWeight = 0;
        int ourReplicas = 0;

        private Counts() {
        }

        private Counts(int totalReplicas, int ourReplicas) {
            this.negRankingWeight = totalReplicas;
            this.ourReplicas = ourReplicas;
        }

        public String toString() {
            return "Counts [negRankingWeight=" + this.negRankingWeight + ", sameSliceCount=" + this.ourReplicas + "]";
        }
    }

    private static class ValueComparator
    implements Comparator<String> {
        Map<String, Counts> map;

        public ValueComparator(Map<String, Counts> map) {
            this.map = map;
        }

        @Override
        public int compare(String a, String b) {
            if (this.map.get((Object)a).negRankingWeight >= this.map.get((Object)b).negRankingWeight) {
                return 1;
            }
            return -1;
        }
    }
}

