/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io.nativeio;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.SecureIOUtils;
import org.apache.hadoop.io.nativeio.Errno;
import org.apache.hadoop.io.nativeio.NativeIOException;
import org.apache.hadoop.util.NativeCodeLoader;
import org.apache.hadoop.util.Shell;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class NativeIO {
    private static boolean workaroundNonThreadSafePasswdCalls = false;
    private static final Log LOG = LogFactory.getLog(NativeIO.class);
    private static boolean nativeLoaded = false;
    private static final Map<Long, CachedUid> uidCache;
    private static long cacheTimeout;
    private static boolean initialized;

    public static boolean isAvailable() {
        return NativeCodeLoader.isNativeCodeLoaded() && nativeLoaded;
    }

    private static native void initNative();

    private static String stripDomain(String name) {
        int i = name.indexOf(92);
        if (i != -1) {
            name = name.substring(i + 1);
        }
        return name;
    }

    public static String getOwner(FileDescriptor fd) throws IOException {
        NativeIO.ensureInitialized();
        if (Shell.WINDOWS) {
            String owner = Windows.getOwner(fd);
            owner = NativeIO.stripDomain(owner);
            return owner;
        }
        long uid = POSIX.getUIDforFDOwnerforOwner(fd);
        CachedUid cUid = uidCache.get(uid);
        long now = System.currentTimeMillis();
        if (cUid != null && cUid.timestamp + cacheTimeout > now) {
            return cUid.username;
        }
        String user = POSIX.getUserName(uid);
        LOG.info("Got UserName " + user + " for UID " + uid + " from the native implementation");
        cUid = new CachedUid(user, now);
        uidCache.put(uid, cUid);
        return user;
    }

    public static FileInputStream getShareDeleteFileInputStream(File f) throws IOException {
        if (!Shell.WINDOWS) {
            return new FileInputStream(f);
        }
        FileDescriptor fd = Windows.createFile(f.getAbsolutePath(), 0x80000000L, 7L, 3L);
        return new FileInputStream(fd);
    }

    public static FileInputStream getShareDeleteFileInputStream(File f, long seekOffset) throws IOException {
        if (!Shell.WINDOWS) {
            RandomAccessFile rf = new RandomAccessFile(f, "r");
            if (seekOffset > 0L) {
                rf.seek(seekOffset);
            }
            return new FileInputStream(rf.getFD());
        }
        FileDescriptor fd = Windows.createFile(f.getAbsolutePath(), 0x80000000L, 7L, 3L);
        if (seekOffset > 0L) {
            Windows.setFilePointer(fd, seekOffset, 0L);
        }
        return new FileInputStream(fd);
    }

    public static FileOutputStream getCreateForWriteFileOutputStream(File f, int permissions) throws IOException {
        if (!Shell.WINDOWS) {
            try {
                FileDescriptor fd = POSIX.open(f.getAbsolutePath(), 193, permissions);
                return new FileOutputStream(fd);
            }
            catch (NativeIOException nioe) {
                if (nioe.getErrno() == Errno.EEXIST) {
                    throw new SecureIOUtils.AlreadyExistsException(nioe);
                }
                throw nioe;
            }
        }
        try {
            FileDescriptor fd = Windows.createFile(f.getCanonicalPath(), 0x40000000L, 7L, 1L);
            POSIX.chmod(f.getCanonicalPath(), permissions);
            return new FileOutputStream(fd);
        }
        catch (NativeIOException nioe) {
            if (nioe.getErrorCode() == 80L) {
                throw new SecureIOUtils.AlreadyExistsException(nioe);
            }
            throw nioe;
        }
    }

    private static synchronized void ensureInitialized() {
        if (!initialized) {
            cacheTimeout = new Configuration().getLong("hadoop.security.uid.cache.secs", 14400L) * 1000L;
            LOG.info("Initialized cache for UID to User mapping with a cache timeout of " + cacheTimeout / 1000L + " seconds.");
            initialized = true;
        }
    }

    public static void renameTo(File src, File dst) throws IOException {
        if (!nativeLoaded) {
            if (!src.renameTo(dst)) {
                throw new IOException("renameTo(src=" + src + ", dst=" + dst + ") failed.");
            }
        } else {
            NativeIO.renameTo0(src.getAbsolutePath(), dst.getAbsolutePath());
        }
    }

    private static native void renameTo0(String var0, String var1) throws NativeIOException;

    static {
        if (NativeCodeLoader.isNativeCodeLoaded()) {
            try {
                NativeIO.initNative();
                nativeLoaded = true;
            }
            catch (Throwable t) {
                LOG.error("Unable to initialize NativeIO libraries", t);
            }
        }
        uidCache = new ConcurrentHashMap<Long, CachedUid>();
        initialized = false;
    }

    private static class CachedUid {
        final long timestamp;
        final String username;

        public CachedUid(String username, long timestamp) {
            this.timestamp = timestamp;
            this.username = username;
        }
    }

    public static class Windows {
        public static final long GENERIC_READ = 0x80000000L;
        public static final long GENERIC_WRITE = 0x40000000L;
        public static final long FILE_SHARE_READ = 1L;
        public static final long FILE_SHARE_WRITE = 2L;
        public static final long FILE_SHARE_DELETE = 4L;
        public static final long CREATE_NEW = 1L;
        public static final long CREATE_ALWAYS = 2L;
        public static final long OPEN_EXISTING = 3L;
        public static final long OPEN_ALWAYS = 4L;
        public static final long TRUNCATE_EXISTING = 5L;
        public static final long FILE_BEGIN = 0L;
        public static final long FILE_CURRENT = 1L;
        public static final long FILE_END = 2L;

        public static native FileDescriptor createFile(String var0, long var1, long var3, long var5) throws IOException;

        public static native long setFilePointer(FileDescriptor var0, long var1, long var3) throws IOException;

        private static native String getOwner(FileDescriptor var0) throws IOException;

        private static native boolean access0(String var0, int var1);

        public static boolean access(String path, AccessRight desiredAccess) throws IOException {
            return Windows.access0(path, desiredAccess.accessRight());
        }

        static {
            if (NativeCodeLoader.isNativeCodeLoaded()) {
                try {
                    NativeIO.initNative();
                    nativeLoaded = true;
                }
                catch (Throwable t) {
                    LOG.error("Unable to initialize NativeIO libraries", t);
                }
            }
        }

        public static enum AccessRight {
            ACCESS_READ(1),
            ACCESS_WRITE(2),
            ACCESS_EXECUTE(32);

            private final int accessRight;

            private AccessRight(int access) {
                this.accessRight = access;
            }

            public int accessRight() {
                return this.accessRight;
            }
        }
    }

    public static class POSIX {
        public static final int O_RDONLY = 0;
        public static final int O_WRONLY = 1;
        public static final int O_RDWR = 2;
        public static final int O_CREAT = 64;
        public static final int O_EXCL = 128;
        public static final int O_NOCTTY = 256;
        public static final int O_TRUNC = 512;
        public static final int O_APPEND = 1024;
        public static final int O_NONBLOCK = 2048;
        public static final int O_SYNC = 4096;
        public static final int O_ASYNC = 8192;
        public static final int O_FSYNC = 4096;
        public static final int O_NDELAY = 2048;
        public static final int POSIX_FADV_NORMAL = 0;
        public static final int POSIX_FADV_RANDOM = 1;
        public static final int POSIX_FADV_SEQUENTIAL = 2;
        public static final int POSIX_FADV_WILLNEED = 3;
        public static final int POSIX_FADV_DONTNEED = 4;
        public static final int POSIX_FADV_NOREUSE = 5;
        public static final int SYNC_FILE_RANGE_WAIT_BEFORE = 1;
        public static final int SYNC_FILE_RANGE_WRITE = 2;
        public static final int SYNC_FILE_RANGE_WAIT_AFTER = 4;
        private static final Log LOG = LogFactory.getLog(NativeIO.class);
        @VisibleForTesting
        public static CacheTracker cacheTracker = null;
        private static boolean nativeLoaded = false;
        private static boolean fadvisePossible = true;
        private static boolean syncFileRangePossible = true;
        static final String WORKAROUND_NON_THREADSAFE_CALLS_KEY = "hadoop.workaround.non.threadsafe.getpwuid";
        static final boolean WORKAROUND_NON_THREADSAFE_CALLS_DEFAULT = true;
        private static long cacheTimeout = -1L;
        private static final Map<Integer, CachedName> USER_ID_NAME_CACHE;
        private static final Map<Integer, CachedName> GROUP_ID_NAME_CACHE;

        public static boolean isAvailable() {
            return NativeCodeLoader.isNativeCodeLoaded() && nativeLoaded;
        }

        public static native FileDescriptor open(String var0, int var1, int var2) throws IOException;

        private static native Stat fstat(FileDescriptor var0) throws IOException;

        private static native void chmodImpl(String var0, int var1) throws IOException;

        public static void chmod(String path, int mode) throws IOException {
            if (!Shell.WINDOWS) {
                POSIX.chmodImpl(path, mode);
            } else {
                try {
                    POSIX.chmodImpl(path, mode);
                }
                catch (NativeIOException nioe) {
                    if (nioe.getErrorCode() == 3L) {
                        throw new NativeIOException("No such file or directory", Errno.ENOENT);
                    }
                    LOG.warn(String.format("NativeIO.chmod error (%d): %s", nioe.getErrorCode(), nioe.getMessage()));
                    throw new NativeIOException("Unknown error", Errno.UNKNOWN);
                }
            }
        }

        static native void posix_fadvise(FileDescriptor var0, long var1, long var3, int var5) throws NativeIOException;

        static native void sync_file_range(FileDescriptor var0, long var1, long var3, int var5) throws NativeIOException;

        public static void posixFadviseIfPossible(String identifier, FileDescriptor fd, long offset, long len, int flags) throws NativeIOException {
            if (cacheTracker != null) {
                cacheTracker.fadvise(identifier, offset, len, flags);
            }
            if (nativeLoaded && fadvisePossible) {
                try {
                    POSIX.posix_fadvise(fd, offset, len, flags);
                }
                catch (UnsupportedOperationException uoe) {
                    fadvisePossible = false;
                }
                catch (UnsatisfiedLinkError ule) {
                    fadvisePossible = false;
                }
            }
        }

        public static void syncFileRangeIfPossible(FileDescriptor fd, long offset, long nbytes, int flags) throws NativeIOException {
            if (nativeLoaded && syncFileRangePossible) {
                try {
                    POSIX.sync_file_range(fd, offset, nbytes, flags);
                }
                catch (UnsupportedOperationException uoe) {
                    syncFileRangePossible = false;
                }
                catch (UnsatisfiedLinkError ule) {
                    syncFileRangePossible = false;
                }
            }
        }

        private static native long getUIDforFDOwnerforOwner(FileDescriptor var0) throws IOException;

        private static native String getUserName(long var0) throws IOException;

        public static Stat getFstat(FileDescriptor fd) throws IOException {
            Stat stat = null;
            if (!Shell.WINDOWS) {
                stat = POSIX.fstat(fd);
                stat.owner = POSIX.getName(IdCache.USER, stat.ownerId);
                stat.group = POSIX.getName(IdCache.GROUP, stat.groupId);
            } else {
                try {
                    stat = POSIX.fstat(fd);
                }
                catch (NativeIOException nioe) {
                    if (nioe.getErrorCode() == 6L) {
                        throw new NativeIOException("The handle is invalid.", Errno.EBADF);
                    }
                    LOG.warn(String.format("NativeIO.getFstat error (%d): %s", nioe.getErrorCode(), nioe.getMessage()));
                    throw new NativeIOException("Unknown error", Errno.UNKNOWN);
                }
            }
            return stat;
        }

        private static String getName(IdCache domain, int id) throws IOException {
            String name;
            Map<Integer, CachedName> idNameCache = domain == IdCache.USER ? USER_ID_NAME_CACHE : GROUP_ID_NAME_CACHE;
            CachedName cachedName = idNameCache.get(id);
            long now = System.currentTimeMillis();
            if (cachedName != null && cachedName.timestamp + cacheTimeout > now) {
                name = cachedName.name;
            } else {
                String string = name = domain == IdCache.USER ? POSIX.getUserName(id) : POSIX.getGroupName(id);
                if (LOG.isDebugEnabled()) {
                    String type = domain == IdCache.USER ? "UserName" : "GroupName";
                    LOG.debug("Got " + type + " " + name + " for ID " + id + " from the native implementation");
                }
                cachedName = new CachedName(name, now);
                idNameCache.put(id, cachedName);
            }
            return name;
        }

        static native String getUserName(int var0) throws IOException;

        static native String getGroupName(int var0) throws IOException;

        static {
            if (NativeCodeLoader.isNativeCodeLoaded()) {
                try {
                    Configuration conf = new Configuration();
                    workaroundNonThreadSafePasswdCalls = conf.getBoolean(WORKAROUND_NON_THREADSAFE_CALLS_KEY, true);
                    NativeIO.initNative();
                    nativeLoaded = true;
                    cacheTimeout = conf.getLong("hadoop.security.uid.cache.secs", 14400L) * 1000L;
                    LOG.debug("Initialized cache for IDs to User/Group mapping with a  cache timeout of " + cacheTimeout / 1000L + " seconds.");
                }
                catch (Throwable t) {
                    LOG.error("Unable to initialize NativeIO libraries", t);
                }
            }
            USER_ID_NAME_CACHE = new ConcurrentHashMap<Integer, CachedName>();
            GROUP_ID_NAME_CACHE = new ConcurrentHashMap<Integer, CachedName>();
        }

        private static enum IdCache {
            USER,
            GROUP;

        }

        private static class CachedName {
            final long timestamp;
            final String name;

            public CachedName(String name, long timestamp) {
                this.name = name;
                this.timestamp = timestamp;
            }
        }

        public static class Stat {
            private int ownerId;
            private int groupId;
            private String owner;
            private String group;
            private int mode;
            public static final int S_IFMT = 61440;
            public static final int S_IFIFO = 4096;
            public static final int S_IFCHR = 8192;
            public static final int S_IFDIR = 16384;
            public static final int S_IFBLK = 24576;
            public static final int S_IFREG = 32768;
            public static final int S_IFLNK = 40960;
            public static final int S_IFSOCK = 49152;
            public static final int S_IFWHT = 57344;
            public static final int S_ISUID = 2048;
            public static final int S_ISGID = 1024;
            public static final int S_ISVTX = 512;
            public static final int S_IRUSR = 256;
            public static final int S_IWUSR = 128;
            public static final int S_IXUSR = 64;

            Stat(int ownerId, int groupId, int mode) {
                this.ownerId = ownerId;
                this.groupId = groupId;
                this.mode = mode;
            }

            Stat(String owner, String group, int mode) {
                this.owner = !Shell.WINDOWS ? owner : NativeIO.stripDomain(owner);
                this.group = !Shell.WINDOWS ? group : NativeIO.stripDomain(group);
                this.mode = mode;
            }

            public String toString() {
                return "Stat(owner='" + this.owner + "', group='" + this.group + "'" + ", mode=" + this.mode + ")";
            }

            public String getOwner() {
                return this.owner;
            }

            public String getGroup() {
                return this.group;
            }

            public int getMode() {
                return this.mode;
            }
        }

        public static interface CacheTracker {
            public void fadvise(String var1, long var2, long var4, int var6);
        }
    }
}

