/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogPosition;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.SimpleCachedBufferPool;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.WrappedRunnable;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractCommitLogSegmentManager {
    static final Logger logger = LoggerFactory.getLogger(AbstractCommitLogSegmentManager.class);
    private volatile CommitLogSegment availableSegment = null;
    private final WaitQueue segmentPrepared = new WaitQueue();
    private final ConcurrentLinkedQueue<CommitLogSegment> activeSegments = new ConcurrentLinkedQueue();
    private volatile CommitLogSegment allocatingFrom = null;
    final String storageDirectory;
    private final AtomicLong size = new AtomicLong();
    private Thread managerThread;
    protected final CommitLog commitLog;
    private volatile boolean shutdown;
    private static final SimpleCachedBufferPool bufferPool = new SimpleCachedBufferPool(DatabaseDescriptor.getCommitLogMaxCompressionBuffersInPool(), DatabaseDescriptor.getCommitLogSegmentSize());

    AbstractCommitLogSegmentManager(CommitLog commitLog, String storageDirectory) {
        this.commitLog = commitLog;
        this.storageDirectory = storageDirectory;
    }

    void start() {
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws Exception {
                while (!AbstractCommitLogSegmentManager.this.shutdown) {
                    try {
                        assert (AbstractCommitLogSegmentManager.this.availableSegment == null);
                        logger.debug("No segments in reserve; creating a fresh one");
                        AbstractCommitLogSegmentManager.this.availableSegment = AbstractCommitLogSegmentManager.this.createSegment();
                        if (AbstractCommitLogSegmentManager.this.shutdown) {
                            AbstractCommitLogSegmentManager.this.discardAvailableSegment();
                            return;
                        }
                        AbstractCommitLogSegmentManager.this.segmentPrepared.signalAll();
                        Thread.yield();
                        if (AbstractCommitLogSegmentManager.this.availableSegment == null && !AbstractCommitLogSegmentManager.this.atSegmentBufferLimit()) continue;
                        AbstractCommitLogSegmentManager.this.maybeFlushToReclaim();
                        LockSupport.park();
                    }
                    catch (Throwable t) {
                        JVMStabilityInspector.inspectThrowable(t);
                        if (!CommitLog.handleCommitError("Failed managing commit log segments", t)) {
                            return;
                        }
                        Uninterruptibles.sleepUninterruptibly((long)1L, (TimeUnit)TimeUnit.SECONDS);
                    }
                    while (AbstractCommitLogSegmentManager.this.availableSegment != null || AbstractCommitLogSegmentManager.this.atSegmentBufferLimit() && !AbstractCommitLogSegmentManager.this.shutdown) {
                        LockSupport.park();
                    }
                }
            }
        };
        this.shutdown = false;
        this.managerThread = new Thread(NamedThreadFactory.threadLocalDeallocator(runnable), "COMMIT-LOG-ALLOCATOR");
        this.managerThread.start();
        this.advanceAllocatingFrom(null);
    }

    private boolean atSegmentBufferLimit() {
        return CommitLogSegment.usesBufferPool(this.commitLog) && bufferPool.atLimit();
    }

    private void maybeFlushToReclaim() {
        long unused = this.unusedCapacity();
        if (unused < 0L) {
            long flushingSize = 0L;
            ArrayList<CommitLogSegment> segmentsToRecycle = new ArrayList<CommitLogSegment>();
            for (CommitLogSegment segment : this.activeSegments) {
                if (segment == this.allocatingFrom) break;
                segmentsToRecycle.add(segment);
                if ((flushingSize += segment.onDiskSize()) + unused < 0L) continue;
                break;
            }
            this.flushDataFrom(segmentsToRecycle, false);
        }
    }

    public abstract CommitLogSegment.Allocation allocate(Mutation var1, int var2);

    abstract void handleReplayedSegment(File var1);

    abstract CommitLogSegment createSegment();

    abstract void discard(CommitLogSegment var1, boolean var2);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void advanceAllocatingFrom(CommitLogSegment old) {
        while (true) {
            AbstractCommitLogSegmentManager abstractCommitLogSegmentManager = this;
            synchronized (abstractCommitLogSegmentManager) {
                if (this.allocatingFrom != old) {
                    return;
                }
                if (this.availableSegment != null) {
                    this.allocatingFrom = this.availableSegment;
                    this.activeSegments.add(this.allocatingFrom);
                    this.availableSegment = null;
                    break;
                }
            }
            this.awaitAvailableSegment(old);
        }
        this.wakeManager();
        if (old != null) {
            this.commitLog.archiver.maybeArchive(old);
            old.discardUnusedTail();
        }
        this.commitLog.requestExtraSync();
    }

    void awaitAvailableSegment(CommitLogSegment currentAllocatingFrom) {
        do {
            WaitQueue.Signal prepared = this.segmentPrepared.register(this.commitLog.metrics.waitingOnSegmentAllocation.time());
            if (this.availableSegment == null && this.allocatingFrom == currentAllocatingFrom) {
                prepared.awaitUninterruptibly();
                continue;
            }
            prepared.cancel();
        } while (this.availableSegment == null && this.allocatingFrom == currentAllocatingFrom);
    }

    void forceRecycleAll(Iterable<UUID> droppedCfs) {
        ArrayList<CommitLogSegment> segmentsToRecycle = new ArrayList<CommitLogSegment>(this.activeSegments);
        CommitLogSegment last = (CommitLogSegment)segmentsToRecycle.get(segmentsToRecycle.size() - 1);
        this.advanceAllocatingFrom(last);
        last.waitForModifications();
        Keyspace.writeOrder.awaitNewBarrier();
        Future<?> future = this.flushDataFrom(segmentsToRecycle, true);
        try {
            future.get();
            for (CommitLogSegment segment : this.activeSegments) {
                for (UUID cfId : droppedCfs) {
                    segment.markClean(cfId, CommitLogPosition.NONE, segment.getCurrentCommitLogPosition());
                }
            }
            for (CommitLogSegment segment : this.activeSegments) {
                if (!segment.isUnused()) continue;
                this.archiveAndDiscard(segment);
            }
            CommitLogSegment first = this.activeSegments.peek();
            if (first != null && first.id <= last.id) {
                logger.error("Failed to force-recycle all segments; at least one segment is still in use with dirty CFs.");
            }
        }
        catch (Throwable t) {
            logger.error("Failed waiting for a forced recycle of in-use commit log segments", t);
        }
    }

    void archiveAndDiscard(CommitLogSegment segment) {
        boolean archiveSuccess = this.commitLog.archiver.maybeWaitForArchiving(segment.getName());
        if (!this.activeSegments.remove(segment)) {
            return;
        }
        logger.debug("Segment {} is no longer active and will be deleted {}", (Object)segment, (Object)(archiveSuccess ? "now" : "by the archive script"));
        this.discard(segment, archiveSuccess);
    }

    void addSize(long addedSize) {
        this.size.addAndGet(addedSize);
    }

    public long onDiskSize() {
        return this.size.get();
    }

    private long unusedCapacity() {
        long total = DatabaseDescriptor.getTotalCommitlogSpaceInMB() * 1024L * 1024L;
        long currentSize = this.size.get();
        logger.trace("Total active commitlog segment space used is {} out of {}", (Object)currentSize, (Object)total);
        return total - currentSize;
    }

    private Future<?> flushDataFrom(List<CommitLogSegment> segments, boolean force) {
        if (segments.isEmpty()) {
            return Futures.immediateFuture(null);
        }
        CommitLogPosition maxCommitLogPosition = segments.get(segments.size() - 1).getCurrentCommitLogPosition();
        LinkedHashMap<UUID, ListenableFuture<CommitLogPosition>> flushes = new LinkedHashMap<UUID, ListenableFuture<CommitLogPosition>>();
        for (CommitLogSegment segment : segments) {
            for (UUID dirtyCFId : segment.getDirtyCFIDs()) {
                Pair<String, String> pair = Schema.instance.getCF(dirtyCFId);
                if (pair == null) {
                    logger.trace("Marking clean CF {} that doesn't exist anymore", (Object)dirtyCFId);
                    segment.markClean(dirtyCFId, CommitLogPosition.NONE, segment.getCurrentCommitLogPosition());
                    continue;
                }
                if (flushes.containsKey(dirtyCFId)) continue;
                String keyspace = (String)pair.left;
                ColumnFamilyStore cfs = Keyspace.open(keyspace).getColumnFamilyStore(dirtyCFId);
                flushes.put(dirtyCFId, (ListenableFuture<CommitLogPosition>)(force ? cfs.forceFlush() : cfs.forceFlush(maxCommitLogPosition)));
            }
        }
        return Futures.allAsList(flushes.values());
    }

    public void stopUnsafe(boolean deleteSegments) {
        logger.debug("CLSM closing and clearing existing commit log segments...");
        this.shutdown();
        try {
            this.awaitTermination();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        for (CommitLogSegment segment : this.activeSegments) {
            this.closeAndDeleteSegmentUnsafe(segment, deleteSegments);
        }
        this.activeSegments.clear();
        this.size.set(0L);
        logger.trace("CLSM done with closing and clearing existing commit log segments.");
    }

    void awaitManagementTasksCompletion() {
        if (this.availableSegment == null && !this.atSegmentBufferLimit()) {
            this.awaitAvailableSegment(this.allocatingFrom);
        }
    }

    private void closeAndDeleteSegmentUnsafe(CommitLogSegment segment, boolean delete) {
        try {
            this.discard(segment, delete);
        }
        catch (AssertionError assertionError) {
            // empty catch block
        }
    }

    public void shutdown() {
        assert (!this.shutdown);
        this.shutdown = true;
        this.discardAvailableSegment();
        this.wakeManager();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void discardAvailableSegment() {
        CommitLogSegment next = null;
        AbstractCommitLogSegmentManager abstractCommitLogSegmentManager = this;
        synchronized (abstractCommitLogSegmentManager) {
            next = this.availableSegment;
            this.availableSegment = null;
        }
        if (next != null) {
            next.discard(true);
        }
    }

    public void awaitTermination() throws InterruptedException {
        this.managerThread.join();
        this.managerThread = null;
        for (CommitLogSegment segment : this.activeSegments) {
            segment.close();
        }
        bufferPool.shutdown();
    }

    @VisibleForTesting
    public Collection<CommitLogSegment> getActiveSegments() {
        return Collections.unmodifiableCollection(this.activeSegments);
    }

    CommitLogPosition getCurrentPosition() {
        return this.allocatingFrom.getCurrentCommitLogPosition();
    }

    public void sync() throws IOException {
        CommitLogSegment current = this.allocatingFrom;
        for (CommitLogSegment segment : this.getActiveSegments()) {
            if (segment.id > current.id) {
                return;
            }
            segment.sync();
        }
    }

    SimpleCachedBufferPool getBufferPool() {
        return bufferPool;
    }

    void wakeManager() {
        LockSupport.unpark(this.managerThread);
    }

    void notifyBufferFreed() {
        this.wakeManager();
    }

    CommitLogSegment allocatingFrom() {
        return this.allocatingFrom;
    }
}

