/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.zip.CRC32;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputBufferFixed;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.SyncUtil;
import org.apache.cassandra.utils.Throwables;

final class HintsWriter
implements AutoCloseable {
    static final int PAGE_SIZE = 4096;
    private final File directory;
    private final HintsDescriptor descriptor;
    private final File file;
    private final FileChannel channel;
    private final int fd;
    private final CRC32 globalCRC;
    private volatile long lastSyncPosition = 0L;

    private HintsWriter(File directory, HintsDescriptor descriptor, File file, FileChannel channel, int fd, CRC32 globalCRC) {
        this.directory = directory;
        this.descriptor = descriptor;
        this.file = file;
        this.channel = channel;
        this.fd = fd;
        this.globalCRC = globalCRC;
    }

    static HintsWriter create(File directory, HintsDescriptor descriptor) throws IOException {
        File file = new File(directory, descriptor.fileName());
        FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);
        int fd = CLibrary.getfd(channel);
        CRC32 crc = new CRC32();
        try (DataOutputBuffer dob = new DataOutputBuffer();){
            descriptor.serialize(dob);
            ByteBuffer descriptorBytes = dob.buffer();
            FBUtilities.updateChecksum(crc, descriptorBytes);
            channel.write(descriptorBytes);
        }
        catch (Throwable e) {
            channel.close();
            throw e;
        }
        return new HintsWriter(directory, descriptor, file, channel, fd, crc);
    }

    HintsDescriptor descriptor() {
        return this.descriptor;
    }

    private void writeChecksum() {
        File checksumFile = new File(this.directory, this.descriptor.checksumFileName());
        try (OutputStream out = Files.newOutputStream(checksumFile.toPath(), new OpenOption[0]);){
            out.write(Integer.toHexString((int)this.globalCRC.getValue()).getBytes(StandardCharsets.UTF_8));
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, checksumFile);
        }
    }

    @Override
    public void close() {
        Throwables.DiscreteAction[] discreteActionArray = new Throwables.DiscreteAction[2];
        discreteActionArray[0] = this::doFsync;
        discreteActionArray[1] = this.channel::close;
        Throwables.perform(this.file, Throwables.FileOpType.WRITE, discreteActionArray);
        this.writeChecksum();
    }

    public void fsync() {
        Throwables.perform(this.file, Throwables.FileOpType.WRITE, this::doFsync);
    }

    private void doFsync() throws IOException {
        SyncUtil.force(this.channel, true);
        this.lastSyncPosition = this.channel.position();
    }

    Session newSession(ByteBuffer buffer) {
        try {
            return new Session(buffer, this.channel.size());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.file);
        }
    }

    final class Session
    implements AutoCloseable {
        private final ByteBuffer buffer;
        private final long initialSize;
        private long bytesWritten;

        Session(ByteBuffer buffer, long initialSize) {
            buffer.clear();
            this.bytesWritten = 0L;
            this.buffer = buffer;
            this.initialSize = initialSize;
        }

        long position() {
            return this.initialSize + this.bytesWritten;
        }

        void append(ByteBuffer hint) throws IOException {
            this.bytesWritten += (long)hint.remaining();
            if (hint.remaining() <= this.buffer.remaining()) {
                this.buffer.put(hint);
                return;
            }
            this.buffer.flip();
            FBUtilities.updateChecksum(HintsWriter.this.globalCRC, this.buffer);
            FBUtilities.updateChecksum(HintsWriter.this.globalCRC, hint);
            HintsWriter.this.channel.write(new ByteBuffer[]{this.buffer, hint});
            this.buffer.clear();
        }

        void append(Hint hint) throws IOException {
            int hintSize = (int)Hint.serializer.serializedSize(hint, HintsWriter.this.descriptor.messagingVersion());
            int totalSize = hintSize + 12;
            if (totalSize > this.buffer.remaining()) {
                this.flushBuffer();
            }
            ByteBuffer hintBuffer = totalSize <= this.buffer.remaining() ? this.buffer : ByteBuffer.allocate(totalSize);
            CRC32 crc = new CRC32();
            try (DataOutputBufferFixed out = new DataOutputBufferFixed(hintBuffer);){
                out.writeInt(hintSize);
                FBUtilities.updateChecksumInt(crc, hintSize);
                out.writeInt((int)crc.getValue());
                Hint.serializer.serialize(hint, (DataOutputPlus)out, HintsWriter.this.descriptor.messagingVersion());
                FBUtilities.updateChecksum(crc, hintBuffer, hintBuffer.position() - hintSize, hintSize);
                out.writeInt((int)crc.getValue());
            }
            if (hintBuffer == this.buffer) {
                this.bytesWritten += (long)totalSize;
            } else {
                this.append((ByteBuffer)hintBuffer.flip());
            }
        }

        @Override
        public void close() throws IOException {
            this.flushBuffer();
            this.maybeFsync();
            this.maybeSkipCache();
        }

        private void flushBuffer() throws IOException {
            this.buffer.flip();
            if (this.buffer.remaining() > 0) {
                FBUtilities.updateChecksum(HintsWriter.this.globalCRC, this.buffer);
                HintsWriter.this.channel.write(this.buffer);
            }
            this.buffer.clear();
        }

        private void maybeFsync() {
            if (this.position() >= HintsWriter.this.lastSyncPosition + (long)DatabaseDescriptor.getTrickleFsyncIntervalInKb() * 1024L) {
                HintsWriter.this.fsync();
            }
        }

        private void maybeSkipCache() {
            long position = this.position();
            if (position >= (long)DatabaseDescriptor.getTrickleFsyncIntervalInKb() * 1024L) {
                CLibrary.trySkipCache(HintsWriter.this.fd, 0L, position - position % 4096L, HintsWriter.this.file.getPath());
            }
        }
    }
}

