/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import java.io.Closeable;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.db.partitions.ImmutableBTreePartition;
import org.apache.cassandra.db.partitions.Partition;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.db.rows.UnfilteredRowIterators;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.SSTableRewriter;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.format.SSTableWriter;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.OutputHandler;
import org.apache.cassandra.utils.UUIDGen;

public class Scrubber
implements Closeable {
    private final ColumnFamilyStore cfs;
    private final SSTableReader sstable;
    private final LifecycleTransaction transaction;
    private final File destination;
    private final boolean skipCorrupted;
    private final boolean isCommutative;
    private final boolean isIndex;
    private final boolean checkData;
    private final long expectedBloomFilterSize;
    private final RandomAccessReader dataFile;
    private final RandomAccessReader indexFile;
    private final ScrubInfo scrubInfo;
    private final RowIndexEntry.IndexSerializer rowIndexEntrySerializer;
    private final boolean isOffline;
    private int goodRows;
    private int badRows;
    private int emptyRows;
    private ByteBuffer currentIndexKey;
    private ByteBuffer nextIndexKey;
    long currentRowPositionFromIndex;
    long nextRowPositionFromIndex;
    private final OutputHandler outputHandler;
    private static final Comparator<Partition> partitionComparator = new Comparator<Partition>(){

        @Override
        public int compare(Partition r1, Partition r2) {
            return r1.partitionKey().compareTo(r2.partitionKey());
        }
    };
    private final SortedSet<Partition> outOfOrder = new TreeSet<Partition>(partitionComparator);

    public Scrubber(ColumnFamilyStore cfs, LifecycleTransaction transaction, boolean skipCorrupted, boolean isOffline, boolean checkData) throws IOException {
        this(cfs, transaction, skipCorrupted, new OutputHandler.LogOutput(), isOffline, checkData);
    }

    public Scrubber(ColumnFamilyStore cfs, LifecycleTransaction transaction, boolean skipCorrupted, OutputHandler outputHandler, boolean isOffline, boolean checkData) throws IOException {
        this.cfs = cfs;
        this.transaction = transaction;
        this.sstable = transaction.onlyOne();
        this.outputHandler = outputHandler;
        this.skipCorrupted = skipCorrupted;
        this.isOffline = isOffline;
        this.rowIndexEntrySerializer = this.sstable.descriptor.version.getSSTableFormat().getIndexSerializer(this.sstable.metadata, this.sstable.descriptor.version, this.sstable.header);
        List<SSTableReader> toScrub = Collections.singletonList(this.sstable);
        this.destination = cfs.getDirectories().getWriteableLocationAsFile(cfs.getExpectedCompactedFileSize(toScrub, OperationType.SCRUB));
        if (this.destination == null) {
            throw new IOException("disk full");
        }
        this.isCommutative = cfs.metadata.isCounter();
        boolean hasIndexFile = new File(this.sstable.descriptor.filenameFor(Component.PRIMARY_INDEX)).exists();
        this.isIndex = cfs.isIndex();
        if (!hasIndexFile) {
            outputHandler.warn("Missing component: " + this.sstable.descriptor.filenameFor(Component.PRIMARY_INDEX));
        }
        this.checkData = checkData && !this.isIndex;
        this.expectedBloomFilterSize = Math.max((long)cfs.metadata.params.minIndexInterval, hasIndexFile ? SSTableReader.getApproximateKeyCount(toScrub) : 0L);
        this.dataFile = isOffline ? this.sstable.openDataReader() : this.sstable.openDataReader(CompactionManager.instance.getRateLimiter());
        this.indexFile = hasIndexFile ? RandomAccessReader.open(new File(this.sstable.descriptor.filenameFor(Component.PRIMARY_INDEX))) : null;
        this.scrubInfo = new ScrubInfo(this.dataFile, this.sstable);
        this.currentRowPositionFromIndex = 0L;
        this.nextRowPositionFromIndex = 0L;
    }

    private UnfilteredRowIterator withValidation(UnfilteredRowIterator iter, String filename) {
        return this.checkData ? UnfilteredRowIterators.withValidation(iter, filename) : iter;
    }

    public void scrub() {
        ArrayList<SSTableReader> finished = new ArrayList<SSTableReader>();
        boolean completed = false;
        this.outputHandler.output(String.format("Scrubbing %s (%s bytes)", this.sstable, this.dataFile.length()));
        try (SSTableRewriter writer = new SSTableRewriter(this.transaction, this.sstable.maxDataAge, this.isOffline);){
            ByteBuffer byteBuffer = this.nextIndexKey = this.indexAvailable() ? ByteBufferUtil.readWithShortLength(this.indexFile) : null;
            if (this.indexAvailable()) {
                long firstRowPositionFromIndex = this.rowIndexEntrySerializer.deserialize((DataInputPlus)this.indexFile).position;
                assert (firstRowPositionFromIndex == 0L) : firstRowPositionFromIndex;
            }
            writer.switchWriter(CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, this.sstable.getSSTableMetadata().repairedAt, this.sstable, this.transaction));
            DecoratedKey prevKey = null;
            while (!this.dataFile.isEOF()) {
                if (this.scrubInfo.isStopRequested()) {
                    throw new CompactionInterruptedException(this.scrubInfo.getCompactionInfo());
                }
                long rowStart = this.dataFile.getFilePointer();
                this.outputHandler.debug("Reading row at " + rowStart);
                DecoratedKey key = null;
                try {
                    key = this.sstable.decorateKey(ByteBufferUtil.readWithShortLength(this.dataFile));
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                }
                this.updateIndexKey();
                long dataStart = this.dataFile.getFilePointer();
                long dataStartFromIndex = -1L;
                long dataSizeFromIndex = -1L;
                if (this.currentIndexKey != null) {
                    dataStartFromIndex = this.currentRowPositionFromIndex + 2L + (long)this.currentIndexKey.remaining();
                    dataSizeFromIndex = this.nextRowPositionFromIndex - dataStartFromIndex;
                }
                String keyName = key == null ? "(unreadable key)" : ByteBufferUtil.bytesToHex(key.getKey());
                this.outputHandler.debug(String.format("row %s is %s bytes", keyName, dataSizeFromIndex));
                assert (this.currentIndexKey != null || !this.indexAvailable());
                try {
                    if (key == null) {
                        throw new IOError(new IOException("Unable to read row key from data file"));
                    }
                    if (this.currentIndexKey != null && !key.getKey().equals(this.currentIndexKey)) {
                        throw new IOError(new IOException(String.format("Key from data file (%s) does not match key from index file (%s)", "_too big_", ByteBufferUtil.bytesToHex(this.currentIndexKey))));
                    }
                    if (this.indexFile != null && dataSizeFromIndex > this.dataFile.length()) {
                        throw new IOError(new IOException("Impossible row size (greater than file length): " + dataSizeFromIndex));
                    }
                    if (this.indexFile != null && dataStart != dataStartFromIndex) {
                        this.outputHandler.warn(String.format("Data file row position %d differs from index file row position %d", dataStart, dataStartFromIndex));
                    }
                    UnfilteredRowIterator iterator = this.withValidation(new SSTableIdentityIterator(this.sstable, this.dataFile, key), this.dataFile.getPath());
                    Throwable throwable = null;
                    try {
                        if (prevKey != null && prevKey.compareTo(key) > 0) {
                            this.saveOutOfOrderRow(prevKey, key, iterator);
                            continue;
                        }
                        if (writer.tryAppend(iterator) == null) {
                            ++this.emptyRows;
                        } else {
                            ++this.goodRows;
                        }
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (iterator == null) continue;
                        if (throwable != null) {
                            try {
                                iterator.close();
                            }
                            catch (Throwable throwable3) {
                                throwable.addSuppressed(throwable3);
                            }
                            continue;
                        }
                        iterator.close();
                        continue;
                    }
                    prevKey = key;
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                    this.outputHandler.warn("Error reading row (stacktrace follows):", th);
                    if (!(this.currentIndexKey == null || key != null && key.getKey().equals(this.currentIndexKey) && dataStart == dataStartFromIndex)) {
                        this.outputHandler.output(String.format("Retrying from row index; data is %s bytes starting at %s", dataSizeFromIndex, dataStartFromIndex));
                        key = this.sstable.decorateKey(this.currentIndexKey);
                        try {
                            this.dataFile.seek(dataStartFromIndex);
                            UnfilteredRowIterator iterator = this.withValidation(new SSTableIdentityIterator(this.sstable, this.dataFile, key), this.dataFile.getPath());
                            Throwable throwable = null;
                            try {
                                if (prevKey != null && prevKey.compareTo(key) > 0) {
                                    this.saveOutOfOrderRow(prevKey, key, iterator);
                                    continue;
                                }
                                if (writer.tryAppend(iterator) == null) {
                                    ++this.emptyRows;
                                } else {
                                    ++this.goodRows;
                                }
                            }
                            catch (Throwable throwable4) {
                                throwable = throwable4;
                                throw throwable4;
                            }
                            finally {
                                if (iterator == null) continue;
                                if (throwable != null) {
                                    try {
                                        iterator.close();
                                    }
                                    catch (Throwable throwable5) {
                                        throwable.addSuppressed(throwable5);
                                    }
                                    continue;
                                }
                                iterator.close();
                                continue;
                            }
                            prevKey = key;
                        }
                        catch (Throwable th2) {
                            this.throwIfFatal(th2);
                            this.throwIfCannotContinue(key, th2);
                            this.outputHandler.warn("Retry failed too. Skipping to next row (retry's stacktrace follows)", th2);
                            ++this.badRows;
                            this.seekToNextRow();
                        }
                        continue;
                    }
                    this.throwIfCannotContinue(key, th);
                    this.outputHandler.warn("Row starting at position " + dataStart + " is unreadable; skipping to next");
                    ++this.badRows;
                    if (this.currentIndexKey == null) continue;
                    this.seekToNextRow();
                }
            }
            if (!this.outOfOrder.isEmpty()) {
                SSTableReader newInOrderSstable;
                long repairedAt = this.badRows > 0 ? 0L : this.sstable.getSSTableMetadata().repairedAt;
                try (SSTableWriter inOrderWriter = CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, repairedAt, this.sstable, this.transaction);){
                    for (Partition partition : this.outOfOrder) {
                        inOrderWriter.append(partition.unfilteredIterator());
                    }
                    newInOrderSstable = inOrderWriter.finish(-1L, this.sstable.maxDataAge, true);
                }
                this.transaction.update(newInOrderSstable, false);
                finished.add(newInOrderSstable);
                this.outputHandler.warn(String.format("%d out of order rows found while scrubbing %s; Those have been written (in order) to a new sstable (%s)", this.outOfOrder.size(), this.sstable, newInOrderSstable));
            }
            finished.addAll((Collection<SSTableReader>)writer.setRepairedAt(this.badRows > 0 ? 0L : this.sstable.getSSTableMetadata().repairedAt).finish());
            completed = true;
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
        finally {
            if (this.transaction.isOffline()) {
                finished.forEach(sstable -> sstable.selfRef().release());
            }
        }
        if (completed) {
            if (this.badRows > 0) {
                this.outputHandler.warn("No valid rows found while scrubbing " + this.sstable + "; it is marked for deletion now. If you want to attempt manual recovery, you can find a copy in the pre-scrub snapshot");
            } else {
                this.outputHandler.output("Scrub of " + this.sstable + " complete; looks like all " + this.emptyRows + " rows were tombstoned");
            }
        } else {
            this.outputHandler.output("Scrub of " + this.sstable + " complete: " + this.goodRows + " rows in new sstable and " + this.emptyRows + " empty (tombstoned) rows dropped");
            if (this.badRows > 0) {
                this.outputHandler.warn("Unable to recover " + this.badRows + " rows that were skipped.  You can attempt manual recovery from the pre-scrub snapshot.  You can also run nodetool repair to transfer the data from a healthy replica, if any");
            }
        }
    }

    private void updateIndexKey() {
        this.currentIndexKey = this.nextIndexKey;
        this.currentRowPositionFromIndex = this.nextRowPositionFromIndex;
        try {
            this.nextIndexKey = !this.indexAvailable() ? null : ByteBufferUtil.readWithShortLength(this.indexFile);
            this.nextRowPositionFromIndex = !this.indexAvailable() ? this.dataFile.length() : this.rowIndexEntrySerializer.deserialize((DataInputPlus)this.indexFile).position;
        }
        catch (Throwable th) {
            JVMStabilityInspector.inspectThrowable(th);
            this.outputHandler.warn("Error reading index file", th);
            this.nextIndexKey = null;
            this.nextRowPositionFromIndex = this.dataFile.length();
        }
    }

    private boolean indexAvailable() {
        return this.indexFile != null && !this.indexFile.isEOF();
    }

    private void seekToNextRow() {
        while (this.nextRowPositionFromIndex < this.dataFile.length()) {
            try {
                this.dataFile.seek(this.nextRowPositionFromIndex);
                return;
            }
            catch (Throwable th) {
                this.throwIfFatal(th);
                this.outputHandler.warn(String.format("Failed to seek to next row position %d", this.nextRowPositionFromIndex), th);
                ++this.badRows;
                this.updateIndexKey();
            }
        }
    }

    private void saveOutOfOrderRow(DecoratedKey prevKey, DecoratedKey key, UnfilteredRowIterator iterator) {
        this.outputHandler.warn(String.format("Out of order row detected (%s found after %s)", key, prevKey));
        this.outOfOrder.add(ImmutableBTreePartition.create(iterator));
    }

    private void throwIfFatal(Throwable th) {
        if (th instanceof Error && !(th instanceof AssertionError) && !(th instanceof IOError)) {
            throw (Error)th;
        }
    }

    private void throwIfCannotContinue(DecoratedKey key, Throwable th) {
        if (this.isIndex) {
            this.outputHandler.warn(String.format("An error occurred while scrubbing the row with key '%s' for an index table. Scrubbing will abort for this table and the index will be rebuilt.", key));
            throw new IOError(th);
        }
        if (this.isCommutative && !this.skipCorrupted) {
            this.outputHandler.warn(String.format("An error occurred while scrubbing the row with key '%s'.  Skipping corrupt rows in counter tables will result in undercounts for the affected counters (see CASSANDRA-2759 for more details), so by default the scrub will stop at this point.  If you would like to skip the row anyway and continue scrubbing, re-run the scrub with the --skip-corrupted option.", key));
            throw new IOError(th);
        }
    }

    @Override
    public void close() {
        FileUtils.closeQuietly(this.dataFile);
        FileUtils.closeQuietly(this.indexFile);
    }

    public CompactionInfo.Holder getScrubInfo() {
        return this.scrubInfo;
    }

    @VisibleForTesting
    public ScrubResult scrubWithResult() {
        this.scrub();
        return new ScrubResult(this);
    }

    public static final class ScrubResult {
        public final int goodRows;
        public final int badRows;
        public final int emptyRows;

        public ScrubResult(Scrubber scrubber) {
            this.goodRows = scrubber.goodRows;
            this.badRows = scrubber.badRows;
            this.emptyRows = scrubber.emptyRows;
        }
    }

    private static class ScrubInfo
    extends CompactionInfo.Holder {
        private final RandomAccessReader dataFile;
        private final SSTableReader sstable;
        private final UUID scrubCompactionId;

        public ScrubInfo(RandomAccessReader dataFile, SSTableReader sstable) {
            this.dataFile = dataFile;
            this.sstable = sstable;
            this.scrubCompactionId = UUIDGen.getTimeUUID();
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            try {
                return new CompactionInfo(this.sstable.metadata, OperationType.SCRUB, this.dataFile.getFilePointer(), this.dataFile.length(), this.scrubCompactionId);
            }
            catch (Exception e) {
                throw new RuntimeException();
            }
        }
    }
}

